﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
* @examplesource{HidGesture_Main.cpp,PageSampleHidGesture}
*
* @brief
*  nn::hid ライブラリのジェスチャ APIプログラム
*/

/**
* @page PageSampleHidGesture ジェスチャの取得
* @tableofcontents
*
* @brief
* nn::hid ライブラリを用いてジェスチャを取得するサンプルプログラムの解説です。
*
* @section PageSampleHidGesture_SectionBrief 概要
* ジェスチャの状態を表示します。
*
* @section PageSampleHidGesture_SectionFileStructure ファイル構成
* 本サンプルプログラムは
* @link ../../../Samples/Sources/Applications/HidGesture
* Samples/Sources/Applications/HidGesture @endlink 以下にあります。
*
* @section PageSampleHidGesture_SectionNecessaryEnvironment 必要な環境
* Windows 環境においては、タッチまたはマウス操作が入力として扱われます。
*
* @section PageSampleHidGesture_SectionHowToOperate 操作方法
* サンプルプログラムを実行するとウィンドウが立ち上がります。
* ウインドウ上には、左半分に矩形状に数字が、右半分には操作ログが表示され、
* 矩形にはピンチや回転といった連続的な操作の結果が、
* 操作ログにはタップやスワイプと言った単発的な操作の結果が反映されます。
* 操作ログからはコンテキスト番号を確認することもできます。
* また、タッチ位置にはカーソルが描画されます。
*
* - ジェスチャ API を Initialize する。
* - 利用可能なジェスチャを全て取得する。
* - eventNumber から新規に発生したジェスチャを選り分ける。
* - 新規に発生したジェスチャを画面表示に反映させる。
*/

#include "Gesture.h"

namespace frm{

    bool IsTriggerPrev = false;

    /* Singleton パターン */
    Gesture& Gesture::GetInstance() NN_NOEXCEPT
    {
        static Gesture Instance;
        return Instance;
    }

    void Gesture::Initialize() NN_NOEXCEPT
    {
        nn::hid::InitializeGesture();
    }

    bool Gesture::GetTouchTrigger(GestureState* state) NN_NOEXCEPT
    {
        bool result;
        bool IsTrigger = false;

        for (auto i = state->Count - 1; i >= 0; --i)
        {
            if (state->Type[i] == nn::hid::GestureType_Tap)
            {
                IsTrigger = true;
            }
        }
        result = IsTrigger & (IsTrigger ^ IsTriggerPrev);

        IsTriggerPrev = IsTrigger;

        return result;
    }

    SwipeState Gesture::GetSwipe() NN_NOEXCEPT
    {
        SwipeState swipe;

        swipe.IsSwipe = false;
        swipe.Direction = nn::hid::GestureDirection_None;

        for (auto i = m_State.Count - 1; i >= 0; --i)
        {
            if (m_State.Type[i] == nn::hid::GestureType_Swipe)
            {
                swipe.IsSwipe = true;
                swipe.Direction = m_State.Direction[i];
                swipe.x = m_State.Data[i].x;
                swipe.y = m_State.Data[i].y;
            }
        }
        return swipe;
    }

    PanState Gesture::GetPan() NN_NOEXCEPT
    {
        PanState pan;

        pan.IsPan = false;

        for (auto i = m_State.Count - 1; i >= 0; --i)
        {
            if (m_State.Type[i] == nn::hid::GestureType_Pan)
            {
                pan.IsPan = true;
                pan.x = m_State.Data[i].x;
                pan.y = m_State.Data[i].y;
                pan.deltaX = m_State.Data[i].deltaX;
                pan.deltaY = m_State.Data[i].deltaY;
            }
        }
        return pan;
    }

    // 指定された画角へのTouchトリガ入力がされたかを返します
    bool Gesture::GetTouchTriggerInRange(float x, float y, float width, float height) NN_NOEXCEPT
    {
        bool result = false;

        for (auto count = 0; count < m_State.Count; ++count)
        {
            if (m_State.TouchTrigger)
            {
                // タッチ範囲の判定
                if ((m_State.Data[count].x >= x) && (m_State.Data[count].x <= x + width) && (m_State.Data[count].y >= y) && (m_State.Data[count].y <= y + height))
                {
                    result = true;
                }
            }
        }
        return result;
    }

    bool Gesture::UpdateState(GestureState* state) NN_NOEXCEPT
    {
        nn::hid::GestureState t_state[nn::hid::GestureStateCountMax];
        auto count = nn::hid::GetGestureStates(t_state, nn::hid::GestureStateCountMax);

        state->Count = 0;

        for (auto i = count - 1; i >= 0; --i)
        {
            // 新たな入力である場合、値を更新
            if (m_EventNumberPrev < t_state[i].eventNumber)
            {
                state->IsUpdate[i] = true;
                state->Count++;
                state->Data[i] = t_state[i];
                state->Type[i] = t_state[i].GetGestureType();
                state->Direction[i] = t_state[i].GetGestureDirection();
                m_EventNumberPrev = t_state[i].eventNumber;
            }
            else
            {
                state->IsUpdate[i] = false;
            }
        }
        state->TouchTrigger = GetTouchTrigger(state);

        m_State = *state;

        return true;
    }

    GestureState Gesture::GetState()
    {
        return m_State;
    }

    void Gesture::WriteCursor(
        nn::gfx::util::DebugFontTextWriter* pTextWriter,
        int32_t x, int32_t y) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pTextWriter);

        const float barOffsetX = pTextWriter->CalculateStringWidth("|") / 2;
        const float barOffsetY = pTextWriter->CalculateStringHeight("|") / 2;
        const float minusOffsetX = pTextWriter->CalculateStringWidth("-") / 2;
        const float minusOffsetY = pTextWriter->CalculateStringHeight("-") / 2;
        const float plusOffsetX = pTextWriter->CalculateStringWidth("+") / 2;
        const float plusOffsetY = pTextWriter->CalculateStringHeight("+") / 2;

        pTextWriter->SetCursor(x - barOffsetX - 30, y - barOffsetY);
        pTextWriter->Print("|");
        pTextWriter->SetCursor(x - barOffsetX + 30, y - barOffsetY);
        pTextWriter->Print("|");
        pTextWriter->SetCursor(x - minusOffsetX, y - minusOffsetY - 30);
        pTextWriter->Print("-");
        pTextWriter->SetCursor(x - minusOffsetX, y - minusOffsetY + 30);
        pTextWriter->Print("-");
        pTextWriter->SetCursor(x - plusOffsetX - 30, y - plusOffsetY - 30);
        pTextWriter->Print("+");
        pTextWriter->SetCursor(x - plusOffsetX - 30, y - plusOffsetY + 30);
        pTextWriter->Print("+");
        pTextWriter->SetCursor(x - plusOffsetX + 30, y - plusOffsetY - 30);
        pTextWriter->Print("+");
        pTextWriter->SetCursor(x - plusOffsetX + 30, y - plusOffsetY + 30);
        pTextWriter->Print("+");
    }

    void Gesture::WritePoints(nn::gfx::util::DebugFontTextWriter* pTextWriter, const GestureState state) NN_NOEXCEPT
    {
        pTextWriter->SetTextColor(Color::Green);

        for (auto count = 0; count < state.Count; ++count)
        {
            for (size_t i = 0; i < state.Data[count].pointCount; ++i)
            {
                WriteCursor(pTextWriter, state.Data[count].points[i].x, state.Data[count].points[i].y);
            }
        }
    }

    const char* Gesture::ToString(nn::hid::GestureDirection direction) NN_NOEXCEPT
    {
        switch (direction)
        {
            case nn::hid::GestureDirection_None:
                return "None";
            case nn::hid::GestureDirection_Left:
                return "Left";
            case nn::hid::GestureDirection_Up:
                return "Up";
            case nn::hid::GestureDirection_Right:
                return "Right";
            case nn::hid::GestureDirection_Down:
                return "Down";
            default:
                return "Undefined";
        }
    }

    const char* Gesture::ToString(nn::hid::GestureType type) NN_NOEXCEPT
    {
        switch (type)
        {
            case nn::hid::GestureType_Idle:
                return "Idle";
            case nn::hid::GestureType_Complete:
                return "Complete";
            case nn::hid::GestureType_Cancel:
                return "Cancel";
            case nn::hid::GestureType_Touch:
                return "Touch";
            case nn::hid::GestureType_Press:
                return "Press";
            case nn::hid::GestureType_Tap:
                return "Tap";
            case nn::hid::GestureType_Pan:
                return "Pan";
            case nn::hid::GestureType_Swipe:
                return "Swipe";
            case nn::hid::GestureType_Pinch:
                return "Pinch";
            case nn::hid::GestureType_Rotate:
                return "Rotate";
            default:
                return "Undefined";
        }
    }

    bool Gesture::MakeButton(nn::gfx::util::DebugFontTextWriter* pTextWriter, GraphicsSystem* pGraphicsSystem, const char* title, float x, float y) NN_NOEXCEPT
    {
        bool isTap = false;

        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();

        const float Width = pTextWriter->CalculateStringWidth(title) + 50.f;
        const float Height = 30.f;

        // タッチ状態の判定
        for (auto count = 0; count < m_State.Count; ++count)
        {
            if (m_State.TouchTrigger)
            {
                // タッチ範囲の判定
                if ((m_State.Data[count].x >= x) && (m_State.Data[count].x <= x + Width) && (m_State.Data[count].y >= y) && (m_State.Data[count].y <= y + Height))
                {
                    isTap = true;
                }
            }
        }

        NN_LOG("%d(Count : %d)\n", isTap, m_State.Count);

        // 枠を描画する
        pPrimitiveRenderer->SetLineWidth(1.f);
        pPrimitiveRenderer->SetColor(Color::White);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), x, y, Width, Height);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), x + 2, y + 2, Width - 4, Height - 4);

        if (isTap)
        {
            pPrimitiveRenderer->SetColor(ClearColor::White);
            pTextWriter->SetTextColor(Color::Black);
        }
        else
        {
            pPrimitiveRenderer->SetColor(ClearColor::Violet);
            pTextWriter->SetTextColor(Color::White);
        }
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DRect(&pGraphicsSystem->GetCommandBuffer(), x, y, Width, Height);

        // タイトルを描画する
        pTextWriter->SetScale(1.f, 1.f);
        pTextWriter->SetCursor(x + 25, y + 3);
        pTextWriter->Print("%s\n", title);

        return isTap;
    }
} // namespace
