﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @brief
    アプリ全体のレイアウトを描画します
*/
#include "ApplicationLayout.h"

const int Width = 1280;
const int Height = 720;

namespace frm{

    /* ------------------------------------------------------------ */
    // PUBLIC関数
    /* ------------------------------------------------------------ */
    Layout& Layout::GetInstance() NN_NOEXCEPT
    {
        static Layout Instance;
        return Instance;
    }

    void Layout::DrawDisplay(GraphicsSystem * pGraphicsSystem,
                             nn::gfx::util::DebugFontTextWriter* pTextWriter,
                             frm::NpadState * state,
                             frm::SixAxis * sixAxis,
                             util::PalmaController* palmaStatus,
                             util::BleState* bleState,
                             frm::CursolPosition cursol)
    {
        gLayout.DrawOverView(pGraphicsSystem, pTextWriter);
        gLayout.DrawActivity(pGraphicsSystem, pTextWriter, cursol);
        gLayout.DrawControler(pGraphicsSystem, pTextWriter, state, sixAxis, palmaStatus, cursol);
        gLayout.DrawGattInfomation(pGraphicsSystem, pTextWriter, bleState);
    }

    /* ------------------------------------------------------------ */
    // PRIVATE関数
    /* ------------------------------------------------------------ */
    void Layout::UpdateFrameTwinkleCounter()
    {
        if (m_FrameTwinkleSwitchFlag == false)
        {
            m_FrameTwinkleCounter = m_FrameTwinkleCounter + 5;
            if (m_FrameTwinkleCounter >= 250)
            {
                m_FrameTwinkleSwitchFlag = true;
            }
        }
        else
        {
            m_FrameTwinkleCounter = m_FrameTwinkleCounter - 5;
            if (m_FrameTwinkleCounter <= 50)
            {
                m_FrameTwinkleSwitchFlag = false;
            }
        }
    }

    void Layout::UpdateSmoothCounter(size_t npadIndex)
    {
        m_smoothCounter = m_smoothCounter != 0 ? (m_smoothCounter > 0 ? m_smoothCounter - 1 : m_smoothCounter + 1) : 0;

        if (npadIndex > m_npadIndexOld)
        {
            m_smoothCounter = -10;
        }
        else if (npadIndex < m_npadIndexOld)
        {
            m_smoothCounter = 10;
        }
        else
        {
            // do nothing
        }

        if (npadIndex == 8 && m_npadIndexOld == 0)
        {
            m_smoothCounter = 10;
        }
        if (npadIndex == 0 && m_npadIndexOld == 8)
        {
            m_smoothCounter = -10;
        }
    }

    void Layout::UpdateMenuSmoothCounter(size_t menuIndex)
    {
        m_menuSmoothCounter = m_menuSmoothCounter != 0 ? (m_menuSmoothCounter > 0 ? m_menuSmoothCounter - 1 : m_menuSmoothCounter + 1) : 0;

        if (menuIndex > m_menuIndexOld)
        {
            m_menuSmoothCounter = -10;
        }
        else if (menuIndex < m_menuIndexOld)
        {
            m_menuSmoothCounter = 10;
        }
        else
        {
            // do nothing
        }

        if (menuIndex == frm::NpadMenuNum - 1 && m_menuIndexOld == 0)
        {
            m_menuSmoothCounter = 10;
        }
        if (menuIndex == 0 && m_menuIndexOld == frm::NpadMenuNum - 1)
        {
            m_menuSmoothCounter = -10;
        }
    }

    void Layout::UpdateScaningCounter()
    {
        if (m_scaningSwitchFlag == false)
        {
            m_scaningCounter++;
            if (m_scaningCounter >= 60)
            {
                m_scaningSwitchFlag = true;
            }
        }
        else
        {
            m_scaningCounter--;
            if (m_scaningCounter <= 0)
            {
                m_scaningSwitchFlag = false;
            }
        }
    }

    void Layout::DrawOverView(GraphicsSystem* pGraphicsSystem,
                      nn::gfx::util::DebugFontTextWriter* pTextWriter) NN_NOEXCEPT
    {
        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();

        UpdateFrameTwinkleCounter();

        // ToolTitleの作成
        nn::oe::DisplayVersion version;
        nn::oe::GetDisplayVersion(&version);
        pTextWriter->SetScale(1.0, 1.0);
        pTextWriter->SetTextColor(Color::White);
        pTextWriter->SetCursor(10, 0);
        pTextWriter->Print("PalmaTestToolForMct(ToolVer. %s)",version.value);

        // 背景色の作成
        pPrimitiveRenderer->SetColor(Color::Black);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DRect(&pGraphicsSystem->GetCommandBuffer(), 0, 0, Width, Height);

        // SCAN中表示の作成
        DrawIsScaning(pGraphicsSystem, pTextWriter, gPalma.m_isScaning, gBle.m_IsScanningSmartDevice);
    }

    void Layout::DrawControler(GraphicsSystem* pGraphicsSystem,
                               nn::gfx::util::DebugFontTextWriter* pTextWriter,
                               frm::NpadState* npadState,
                               frm::SixAxis* sixAxis,
                               util::PalmaController* palmaState,
                               frm::CursolPosition cursol)
    {
        UpdateSmoothCounter(cursol.npadNo);
        m_npadIndexOld = cursol.npadNo;

        int X = 10;
        int Y = 30 - (m_smoothCounter * 30);
        int Width = 900;
        int Height = 335;

        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();



        // コントローラ上段枠の作成
        pPrimitiveRenderer->SetColor(ClearColor::Green);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DRect(&pGraphicsSystem->GetCommandBuffer(), X, Y, Width, Height);
        pPrimitiveRenderer->SetColor(ClearColor::White);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DRect(&pGraphicsSystem->GetCommandBuffer(), X + 10, Y + 10, Width - 600, Height - 20);
        pPrimitiveRenderer->SetLineWidth(1.f);
        pPrimitiveRenderer->SetColor(Color::Gray);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X + 10, Y + 10, Width - 600, Height - 20);

        // コントローラ下段枠の作成
        pPrimitiveRenderer->SetColor(ClearColor::Green);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DRect(&pGraphicsSystem->GetCommandBuffer(), X, Y + Height + 10, Width, Height);
        pPrimitiveRenderer->SetColor(ClearColor::White);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DRect(&pGraphicsSystem->GetCommandBuffer(), X + 10, Y + Height + 20, Width - 600, Height - 20);
        pPrimitiveRenderer->SetLineWidth(1.f);
        pPrimitiveRenderer->SetColor(Color::Gray);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X + 10, Y + Height + 20, Width - 600, Height - 20);

        // カーソル選択枠の作成
        if (cursol.topForcus == 0)
        {
            if (cursol.CursolStatus == CursolStatus_Top)
            {
                pPrimitiveRenderer->SetLineWidth(3.f);
                nn::util::Color4u8 color = { 50,  255, 255, m_FrameTwinkleCounter };
                pPrimitiveRenderer->SetColor(color);
                pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X, Y, Width, Height);
            }
        }

        auto index = cursol.npadNo;

        // 上段の各種コントローラ状態描画
        DrawNpadState(pGraphicsSystem, pTextWriter, npadState[index], 0);
        if (npadState[index].isConnected == true)
        {
            DrawPalmaStatus(pGraphicsSystem, pTextWriter, palmaState[index], npadState[index], 0);
            DrawNpadStartScan(pGraphicsSystem, pTextWriter, cursol, index);
            DrawNpadWriteActivity(pGraphicsSystem, pTextWriter, cursol, index);
            DrawNpadPlayActivity(pGraphicsSystem, pTextWriter, cursol, index);
            DrawNpadCheckReserved(pGraphicsSystem, pTextWriter, cursol, index);
            DrawNpadPlayLog(pGraphicsSystem, pTextWriter, cursol, index);
            DrawNpadWriteDatabaseIdVer(pGraphicsSystem, pTextWriter, cursol, index);
            DrawNpadSetFrMode(pGraphicsSystem, pTextWriter, cursol, index);
            DrawNpadSuspendFeatureSet(pGraphicsSystem, pTextWriter, cursol, index);
            DrawNpadBoostMode(pGraphicsSystem, pTextWriter, cursol, index);
            DrawNpadReadAppData(pGraphicsSystem, pTextWriter, cursol, index);
            DrawNpadWriteAppData(pGraphicsSystem, pTextWriter, cursol, index);
            DrawInputState(pGraphicsSystem, pTextWriter, npadState[index], sixAxis[index].state[0], 0);
            DrawNpadMenu(pGraphicsSystem, pTextWriter, cursol.npadFunction[index], cursol.CursolStatus, 0);
            DrawResult(pGraphicsSystem, pTextWriter, palmaState[index], 0);
            DrawObject(pGraphicsSystem, npadState[index], sixAxis[index].relationalQuaternion[0], 0);
        }

        if (index < 8)
        {
            // 下段の各種コントローラ状態描画
            DrawNpadState(pGraphicsSystem, pTextWriter, npadState[index + 1], 1);
            if (npadState[index + 1].isConnected == true)
            {
                DrawPalmaStatus(pGraphicsSystem, pTextWriter, palmaState[index + 1], npadState[index + 1], 1);
                DrawInputState(pGraphicsSystem, pTextWriter, npadState[index + 1], sixAxis[index + 1].state[0], 1);
                //DrawNpadMenu(pGraphicsSystem, pTextWriter, cursol.npadFunction[index + 1], cursol.CursolStatus, 1);
                DrawObject(pGraphicsSystem, npadState[index + 1], sixAxis[index + 1].relationalQuaternion[0], 1);
            }
        }
        // 最後尾のNpadの場合、次はNpad:0を表示する
        else
        {
            DrawNpadState(pGraphicsSystem, pTextWriter, npadState[0], 1);
            if (npadState[0].isConnected == true)
            {
                DrawPalmaStatus(pGraphicsSystem, pTextWriter, palmaState[0], npadState[0], 1);
                DrawInputState(pGraphicsSystem, pTextWriter, npadState[0], sixAxis[0].state[0], 1);
                //DrawNpadMenu(pGraphicsSystem, pTextWriter, cursol.npadFunction[0], cursol.CursolStatus, 1);
                DrawObject(pGraphicsSystem, npadState[0], sixAxis[0].relationalQuaternion[0], 1);
            }
        }
    }

    void Layout::DrawNpadState(GraphicsSystem * pGraphicsSystem, nn::gfx::util::DebugFontTextWriter * pTextWriter, frm::NpadState state, size_t drawPotiion)
    {
        int NpadX = 25;
        int NpadY;

        if (drawPotiion == 0)
        {
            NpadY = 40 - (m_smoothCounter * 30);
        }
        else
        {
            NpadY = 385 - (m_smoothCounter * 30);
        }
        pTextWriter->SetTextColor(Color::White);
        pTextWriter->SetScale(1.0, 1.0);
        pTextWriter->SetCursor(NpadX, NpadY);
        pTextWriter->Print("Npad:%s(Style:%s)", gNpad.ToString(state.id), gNpad.ToString(state.style));
    }

    void Layout::DrawInputState(GraphicsSystem* pGraphicsSystem,
                                    nn::gfx::util::DebugFontTextWriter* pTextWriter,
                                    frm::NpadState state,
                                    nn::hid::SixAxisSensorState sixAxisState,
                                    size_t drawPotiion)
    {
        int NpadX = 25;
        int NpadY;

        if (drawPotiion == 0)
        {
            NpadY = 255 - (m_smoothCounter * 30);
        }
        else
        {
            NpadY = 600 - (m_smoothCounter * 30);
        }
        pTextWriter->SetTextColor(Color::White);
        pTextWriter->SetScale(0.8, 0.8);

        pTextWriter->SetCursor(NpadX, NpadY);
        pTextWriter->Print("SamplingRatio:(%.1f [%])", state.samplingNumber,state.packetLossRatio);

        pTextWriter->SetCursor(NpadX, NpadY + 20);
        pTextWriter->Print("Button:(%s)", gNpad.ToString(state.press).c_str());

        pTextWriter->SetCursor(NpadX, NpadY + 40);
        pTextWriter->Print("Accl:(%.4f, %.4f, %.4f)"
                            , sixAxisState.acceleration.x
                            , sixAxisState.acceleration.y
                            , sixAxisState.acceleration.z
        );
        pTextWriter->SetCursor(NpadX, NpadY + 60);
        pTextWriter->Print("Rate:(%.4f, %.4f, %.4f)"
                            , sixAxisState.angularVelocity.x
                            , sixAxisState.angularVelocity.y
                            , sixAxisState.angularVelocity.z
        );
        pTextWriter->SetCursor(NpadX, NpadY + 80);
        pTextWriter->Print("Angle:(%.4f, %.4f, %.4f)"
                           , sixAxisState.angle.x
                           , sixAxisState.angle.y
                           , sixAxisState.angle.z
        );
    }

    void Layout::DrawObject(GraphicsSystem* pGraphicsSystem,
                            frm::NpadState state,
                            nn::util::Vector4fType quatanion,
                            size_t ViewportScissorIndex)
    {
        size_t ViewportScissorArray;

        if (ViewportScissorIndex == 0)
        {
            ViewportScissorArray = 250 - (m_smoothCounter * 30) < 1 ? 1 : 250 - (m_smoothCounter * 30);
        }
        else
        {
            if (595 - (m_smoothCounter * 30) > 719)
            {
                return;
            }
            ViewportScissorArray = 595 - (m_smoothCounter * 30) > 719 ? 719 : 595 - (m_smoothCounter * 30);
        }

        nn::gfx::ViewportScissorState* pViewportScissorState = &pGraphicsSystem->GetViewportScissor(ViewportScissorArray);
        pGraphicsSystem->GetCommandBuffer().SetViewportScissorState(pViewportScissorState);

        nn::util::Vector3fType size = { 2.f, 2.f, 1.f };
        float diameter = 2.5f;

        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();

        nn::util::Matrix4x3fType viewMatrix;
        nn::util::Matrix4x4fType projectionMatrix;
        nn::util::Matrix4x3f modelMatrix;

        nn::util::Vector3fType cameraPosition = { 0.0f, 0.0f, 3.f };
        nn::util::Vector3fType cameraTarget = { 0.f, 0.f, 0.f };
        nn::util::Vector3fType cammeraUp = { 0.f, 1.f, 0.f };
        nn::util::MatrixLookAtRightHanded(&viewMatrix, cameraPosition, cameraTarget, cammeraUp);
        pPrimitiveRenderer->SetViewMatrix(&viewMatrix);

        pPrimitiveRenderer->SetLineWidth(2.f);

        // プロジェクションを初期化
        const float Fovy = nn::util::FloatPi / 3.0f;
        const float Aspect = static_cast< float >(Width) / static_cast< float >(Height);
        nn::util::MatrixPerspectiveFieldOfViewRightHanded(&projectionMatrix, Fovy, Aspect, 0.1f, 100.f);
        pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);

        nn::util::Vector3f vecZero;
        nn::util::VectorZero(&vecZero);

        modelMatrix = nn::util::MatrixRowMajor4x3f::MakeRotation(quatanion);
        nn::util::MatrixSetAxisW(&modelMatrix, vecZero);
        pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

        // 3D描画(深度テスト)の有効化
        nns::gfx::PrimitiveRenderer::DepthStencilType info = nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthWriteTest;
        pGraphicsSystem->GetPrimitiveRenderer().SetDepthStencilState(&pGraphicsSystem->GetCommandBuffer(), info);

        if (state.style.Test<nn::hid::NpadStylePalma>())
        {
            nn::util::Vector3fType center = { 0.f, 0.f, 0.f };
            pPrimitiveRenderer->SetColor(Color::Red);
            pGraphicsSystem->GetPrimitiveRenderer().DrawSphere(&pGraphicsSystem->GetCommandBuffer(),
                                                               nns::gfx::PrimitiveRenderer::Surface_Normal,
                                                               nns::gfx::PrimitiveRenderer::Subdiv_Normal,
                                                               center,
                                                               diameter);

            pPrimitiveRenderer->SetColor(HalfColor::White);
            pGraphicsSystem->GetPrimitiveRenderer().DrawSphere(&pGraphicsSystem->GetCommandBuffer(),
                                                               nns::gfx::PrimitiveRenderer::Surface_Wired,
                                                               nns::gfx::PrimitiveRenderer::Subdiv_Normal,
                                                               center,
                                                               diameter);

            diameter = 0.55f;
            center = { 0.f, 0.f, 1.f };
            pPrimitiveRenderer->SetColor(Color::White);
            pGraphicsSystem->GetPrimitiveRenderer().DrawSphere(&pGraphicsSystem->GetCommandBuffer(),
                                                               nns::gfx::PrimitiveRenderer::Surface_Solid,
                                                               nns::gfx::PrimitiveRenderer::Subdiv_Normal,
                                                               center,
                                                               diameter);
        }
        else
        {
            nn::util::Vector3fType center = { 0.f, -1.f, 0.f };
            pPrimitiveRenderer->SetColor(Color::Blue);
            pGraphicsSystem->GetPrimitiveRenderer().DrawCylinder(&pGraphicsSystem->GetCommandBuffer(),
                                                               nns::gfx::PrimitiveRenderer::Surface_Normal,
                                                               center,
                                                               size);

            pPrimitiveRenderer->SetColor(HalfColor::White);
            pGraphicsSystem->GetPrimitiveRenderer().DrawCylinder(&pGraphicsSystem->GetCommandBuffer(),
                                                               nns::gfx::PrimitiveRenderer::Surface_Wired,
                                                               center,
                                                               size);
        }


        // 3D描画(深度テスト)の再無効化
        info = nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest;
        pGraphicsSystem->GetPrimitiveRenderer().SetDepthStencilState(&pGraphicsSystem->GetCommandBuffer(), info);

        pViewportScissorState = &pGraphicsSystem->GetViewportScissor(0);
        pGraphicsSystem->GetCommandBuffer().SetViewportScissorState(pViewportScissorState);

        // レンダラの各パラメータ初期化
        pPrimitiveRenderer->SetDefaultParameters();
    }

    void Layout::DrawNpadMenu(GraphicsSystem * pGraphicsSystem,
                              nn::gfx::util::DebugFontTextWriter * pTextWriter,
                              int npadFunction,
                              frm::CursolStatus status,
                              size_t drawPotiion)
    {
        UpdateMenuSmoothCounter(npadFunction);
        m_menuIndexOld = npadFunction;

        int X = 350;
        int Y;

        if (drawPotiion == 0)
        {
            Y = 30 - (m_smoothCounter * 30);
        }
        else
        {
            Y = 375 - (m_smoothCounter * 30);
        }

        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();

        Y += 25;

        pPrimitiveRenderer->SetLineWidth(1.f);
        pPrimitiveRenderer->SetColor(ExClearColor::Orange);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DRect(&pGraphicsSystem->GetCommandBuffer(), X, Y - 5, 240, 260);
        pPrimitiveRenderer->SetColor(Color::Gray);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X, Y - 5, 240, 260);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X - 2, Y - 7, 240 + 4, 260 + 4);

        if (status == CursolStatus_FixedNpad)
        {
            // カーソル選択枠の作成
            pPrimitiveRenderer->SetLineWidth(3.f);
            nn::util::Color4u8 color = { 50,  255, 255, m_FrameTwinkleCounter };
            pPrimitiveRenderer->SetColor(color);

            X += 5;

            for (auto index = npadFunction; index < npadFunction + 10; index++)
            {
                if (index == npadFunction)
                {
                    pTextWriter->SetTextColor(color);
                }
                else
                {
                    pTextWriter->SetTextColor(Color::Gray);
                }
                pTextWriter->SetScale(1.1f, 1.1f);
                pTextWriter->SetCursor(X, Y - m_menuSmoothCounter);
                pTextWriter->Print("%s", frm::NpadMenu[index % (frm::NpadMenuNum)]);

                Y += 25;
            }
        }
    }

    void Layout::DrawResult(GraphicsSystem * pGraphicsSystem,
                              nn::gfx::util::DebugFontTextWriter * pTextWriter,
                              util::PalmaController palmaState,
                              size_t drawPotiion)
    {
        int X = 350;
        int Y;

        if (drawPotiion == 0)
        {
            Y = 320 - (m_smoothCounter * 30);
        }
        else
        {
            Y = 665 - (m_smoothCounter * 30);
        }

        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();

        pPrimitiveRenderer->SetLineWidth(1.f);
        pPrimitiveRenderer->SetColor(ExClearColor::Orange);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DRect(&pGraphicsSystem->GetCommandBuffer(), X, Y, 240, 30);
        pPrimitiveRenderer->SetColor(Color::Gray);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X, Y, 240, 30);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X - 2, Y - 2, 240 + 4, 30 + 4);

        // カーソル選択枠の作成
        nn::util::Color4u8 color = { 255,  255, 255, palmaState.apiResult.viewCounter };
        pTextWriter->SetTextColor(color);

        X += 5;

        pTextWriter->SetScale(1.f, 1.f);
        pTextWriter->SetCursor(X, Y + 3);
        pTextWriter->Print("-> %s", palmaState.apiResult.string);
    }

    void Layout::DrawIsScaning(GraphicsSystem* pGraphicsSystem,
                                 nn::gfx::util::DebugFontTextWriter* pTextWriter,
                                 bool isScaning,
                                 bool isScaningSmartDevice)
    {
        int X = 350;
        int Y = 0;

        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();

        if (isScaning == true || isScaningSmartDevice == true)
        {
            UpdateScaningCounter();

            pPrimitiveRenderer->SetLineWidth(1.f);
            pPrimitiveRenderer->SetColor(Color::Gray);
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X, Y, 150 + 20, 23);
            pPrimitiveRenderer->SetColor(ClearColor::White);
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DRect(&pGraphicsSystem->GetCommandBuffer(), X, Y, 150 + 20, 23);

            if (isScaning == true && isScaningSmartDevice == true)
            {
                pPrimitiveRenderer->SetColor(HalfColor::Green);
            }
            else if (isScaningSmartDevice)
            {
                pPrimitiveRenderer->SetColor(HalfColor::Indigo);
            }
            else
            {
                pPrimitiveRenderer->SetColor(HalfColor::Orange);
            }
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DRect(&pGraphicsSystem->GetCommandBuffer(), X + (2.5f * m_scaningCounter), Y, 20, 23);

            pTextWriter->SetScale(1.f, 1.f);
            pTextWriter->SetCursor(X + 5, Y);
            pTextWriter->SetTextColor(QuorterColor::White);
            pTextWriter->Print("Device Scaning.....");
        }
    }

    void Layout::DrawPalmaStatus(GraphicsSystem * pGraphicsSystem,
                              nn::gfx::util::DebugFontTextWriter * pTextWriter,
                              util::PalmaController palmaState,
                              frm::NpadState state,
                              size_t drawPotiion)
    {
        int X = 620;
        int Y;

        if (drawPotiion == 0)
        {
            Y = 45 - (m_smoothCounter * 30);
        }
        else
        {
            Y = 390 - (m_smoothCounter * 30);
        }

        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();

        // 表題の作成
        pTextWriter->SetScale(1.f, 1.f);
        pTextWriter->SetCursor(X, Y);
        pTextWriter->SetTextColor(Color::White);
        pTextWriter->Print("[Controller Status]");

        Y += 20;

        pPrimitiveRenderer->SetLineWidth(1.f);
        pPrimitiveRenderer->SetColor(Color::White);

        pTextWriter->SetCursor(X + 5, Y + 3);
        pTextWriter->Print("Unique Code");
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X, Y, 160, 60);
        pTextWriter->SetCursor(X + 165, Y + 3);
        for (auto i = 0; i <= 7; i++)
        {
            pTextWriter->Print("%c", static_cast<char>(palmaState.uniqueCode[i]));
        }
        pTextWriter->SetCursor(X + 165, Y + 33);
        for (auto i = 8; i <= 15; i++)
        {
            pTextWriter->Print("%c", static_cast<char>(palmaState.uniqueCode[i]));
        }
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X + 160, Y, 120, 60);

        Y += 60;

        pTextWriter->SetCursor(X + 5, Y + 3);
        pTextWriter->Print("Palma Step");
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X, Y, 160, 30);
        pTextWriter->SetCursor(X + 165, Y + 3);
        pTextWriter->Print("%d",palmaState.step);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X + 160, Y, 120, 30);

        Y += 30;

        pTextWriter->SetCursor(X + 5, Y + 3);
        pTextWriter->Print("DataBaseID Ver");
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X, Y, 160, 30);
        pTextWriter->SetCursor(X + 165, Y + 3);
        pTextWriter->Print("%d", palmaState.databaseIdVersion);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X + 160, Y, 120, 30);

        Y += 30;

        pTextWriter->SetCursor(X + 5, Y + 3);
        pTextWriter->Print("FrMode");
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X, Y, 160, 30);
        pTextWriter->SetCursor(X + 165, Y + 3);

        pTextWriter->Print("%s", palmaState.feature.Test<nn::hid::PalmaFeature::FrMode>() ? "ON" : "OFF");
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X + 160, Y, 120, 30);

        Y += 30;

        pTextWriter->SetCursor(X + 5, Y + 3);
        pTextWriter->Print("RumbleFeedback");
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X, Y, 160, 30);
        pTextWriter->SetCursor(X + 165, Y + 3);

        pTextWriter->Print("%s", palmaState.feature.Test<nn::hid::PalmaFeature::RumbleFeedback>() ? "ON" : "OFF");
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X + 160, Y, 120, 30);

        Y += 30;

        pTextWriter->SetCursor(X + 5, Y + 3);
        pTextWriter->Print("Step");
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X, Y, 160, 30);
        pTextWriter->SetCursor(X + 165, Y + 3);

        pTextWriter->Print("%s", palmaState.feature.Test<nn::hid::PalmaFeature::Step>() ? "ON" : "OFF");
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X + 160, Y, 120, 30);
    }

    void Layout::DrawNpadStartScan(GraphicsSystem * pGraphicsSystem,
                                      nn::gfx::util::DebugFontTextWriter * pTextWriter,
                                      frm::CursolPosition cursol,
                                      size_t index)
    {
        int X = 350;
        int Y = 200;

        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();

        if (cursol.CursolStatus == CursolStatus_FixedNpadStartScan)
        {
            // 枠の作成
            pTextWriter->SetScale(1.f, 1.f);
            pTextWriter->SetCursor(X, Y);
            pTextWriter->SetTextColor(Color::Black);
            pTextWriter->Print("Scan Device Select...");

            pPrimitiveRenderer->SetLineWidth(1.f);
            pPrimitiveRenderer->SetColor(QuorterColor::White);
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DRect(&pGraphicsSystem->GetCommandBuffer(), X, Y, 330, 100);
            pPrimitiveRenderer->SetColor(Color::Gray);
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X, Y, 330, 100);

            // 項目の作成
            X += 20;
            Y += 40;

            pTextWriter->SetCursor(X, Y);
            pTextWriter->Print("ScanDevice");
            pTextWriter->SetCursor(X + 150, Y);

            pTextWriter->SetTextColor(Color::Black);

            if (cursol.scanDeviseType == 0)
            {
                pTextWriter->Print("Palma");
            }
            else
            {
                pTextWriter->Print("SmartDevice");
            }

            // カーソル選択枠の作成
            pPrimitiveRenderer->SetLineWidth(3.f);
            nn::util::Color4u8 color = { 50,  255, 255, m_FrameTwinkleCounter };
            pPrimitiveRenderer->SetColor(color);

            pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X + 145, 240, 130, 25);
        }
    }

    void Layout::DrawNpadWriteActivity(GraphicsSystem * pGraphicsSystem,
                              nn::gfx::util::DebugFontTextWriter * pTextWriter,
                              frm::CursolPosition cursol,
                              size_t index)
    {
        int X = 350;
        int Y = 200;

        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();

        if (cursol.CursolStatus == CursolStatus_FixedNpadWriteActivity)
        {
            // 枠の作成
            pTextWriter->SetScale(1.f, 1.f);
            pTextWriter->SetCursor(X, Y);
            pTextWriter->SetTextColor(Color::Black);
            pTextWriter->Print("[Write Activity Select...]");

            pPrimitiveRenderer->SetLineWidth(1.f);
            pPrimitiveRenderer->SetColor(QuorterColor::White);
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DRect(&pGraphicsSystem->GetCommandBuffer(), X, Y, 330, 180);
            pPrimitiveRenderer->SetColor(Color::Gray);
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X, Y, 330, 180);

            // 項目の作成
            X += 20;
            Y += 40;
            pTextWriter->SetCursor(X, Y);
            pTextWriter->Print("Sound Index No.");
            pTextWriter->SetCursor(X + 200, Y);
            if (cursol.activityElement[index].Entry.waveIndex >= frm::SoundIndexMax)
            {
                pTextWriter->SetTextColor(Color::Red);
            }
            else
            {
                pTextWriter->SetTextColor(Color::Black);
            }

            if (cursol.activityElement[index].Entry.waveIndex != frm::SoundIndexMax + 9)
            {
                pTextWriter->Print("%d", cursol.activityElement[index].Entry.waveIndex);
            }
            else
            {
                pTextWriter->Print("Empty");
            }
            pTextWriter->SetTextColor(Color::Black);

            Y += 30;
            pTextWriter->SetCursor(X, Y);
            pTextWriter->Print("Sound Type");
            pTextWriter->SetCursor(X + 200, Y);
            pTextWriter->Print("%s", gPalma.ToString(cursol.activityElement[index].Entry.waveSet));

            Y += 30;
            pTextWriter->SetCursor(X, Y);
            pTextWriter->Print("Led Index No.");
            pTextWriter->SetCursor(X + 200, Y);
            if (cursol.activityElement[index].Entry.rgbLedPatternIndex >= frm::RgbLedIndexMax)
            {
                pTextWriter->SetTextColor(Color::Red);
            }
            else
            {
                pTextWriter->SetTextColor(Color::Black);
            }

            if (cursol.activityElement[index].Entry.rgbLedPatternIndex != frm::RgbLedIndexMax + 9)
            {
                pTextWriter->Print("%d", cursol.activityElement[index].Entry.rgbLedPatternIndex);
            }
            else
            {
                pTextWriter->Print("Empty");
            }
            pTextWriter->SetTextColor(Color::Black);

            Y += 30;
            pTextWriter->SetCursor(X, Y);
            pTextWriter->Print("Write ActivityIndex No.");
            pTextWriter->SetCursor(X + 200, Y);
            if (cursol.activityElement[index].Index >= frm::ActivityIndexMax)
            {
                pTextWriter->SetTextColor(Color::Red);
            }
            else
            {
                pTextWriter->SetTextColor(Color::Black);
            }
            pTextWriter->Print("%d", cursol.activityElement[index].Index);
            pTextWriter->SetTextColor(Color::Black);

            // カーソル選択枠の作成
            pPrimitiveRenderer->SetLineWidth(3.f);
            nn::util::Color4u8 color = { 50,  255, 255, m_FrameTwinkleCounter };
            pPrimitiveRenderer->SetColor(color);

            pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X + 195, 240 + (30 * cursol.activityPosition[index]), 80, 25);
        }
    }

    void Layout::DrawNpadPlayActivity(GraphicsSystem * pGraphicsSystem,
                                       nn::gfx::util::DebugFontTextWriter * pTextWriter,
                                       frm::CursolPosition cursol,
                                       size_t index)
    {
        int X = 350;
        int Y = 200;

        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();

        if (cursol.CursolStatus == CursolStatus_FixedNpadPlayActivity)
        {
            // 枠の作成
            pTextWriter->SetScale(1.f, 1.f);
            pTextWriter->SetCursor(X, Y);
            pTextWriter->SetTextColor(Color::Black);
            pTextWriter->Print("Play Activity Select...");

            pPrimitiveRenderer->SetLineWidth(1.f);
            pPrimitiveRenderer->SetColor(QuorterColor::White);
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DRect(&pGraphicsSystem->GetCommandBuffer(), X, Y, 330, 100);
            pPrimitiveRenderer->SetColor(Color::Gray);
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X, Y, 330, 100);

            // 項目の作成
            X += 20;
            Y += 40;

            pTextWriter->SetCursor(X, Y);
            pTextWriter->Print("Play ActivityIndex No.");
            pTextWriter->SetCursor(X + 200, Y);
            if (cursol.activityElement[index].Index >= frm::ActivityIndexMax)
            {
                pTextWriter->SetTextColor(Color::Red);
            }
            else
            {
                pTextWriter->SetTextColor(Color::Black);
            }
            pTextWriter->Print("%d", cursol.activityElement[index].Index);

            // カーソル選択枠の作成
            pPrimitiveRenderer->SetLineWidth(3.f);
            nn::util::Color4u8 color = { 50,  255, 255, m_FrameTwinkleCounter };
            pPrimitiveRenderer->SetColor(color);

            pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X + 195, 240, 80, 25);
        }
    }

    void Layout::DrawNpadCheckReserved(GraphicsSystem * pGraphicsSystem,
                                      nn::gfx::util::DebugFontTextWriter * pTextWriter,
                                      frm::CursolPosition cursol,
                                      size_t index)
    {
        int X = 350;
        int Y = 200;

        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();

        if (cursol.CursolStatus == CursolStatus_FixedNpadCheckReserved)
        {
            // 枠の作成
            pTextWriter->SetScale(1.f, 1.f);
            pTextWriter->SetCursor(X, Y);
            pTextWriter->SetTextColor(Color::Black);
            pTextWriter->Print("CheckReservedArea Select...");

            pPrimitiveRenderer->SetLineWidth(1.f);
            pPrimitiveRenderer->SetColor(QuorterColor::White);
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DRect(&pGraphicsSystem->GetCommandBuffer(), X, Y, 330, 100);
            pPrimitiveRenderer->SetColor(Color::Gray);
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X, Y, 330, 100);

            // 項目の作成
            X += 20;
            Y += 40;

            pTextWriter->SetCursor(X, Y);
            pTextWriter->Print("CheckReservedArea");
            pTextWriter->SetCursor(X + 200, Y);

            pTextWriter->SetTextColor(Color::Black);

            if (cursol.reservedCheckIndex[index] == 0)
            {
                pTextWriter->Print("First");
            }
            else if (cursol.reservedCheckIndex[index] == 1)
            {
                pTextWriter->Print("Second");
            }
            else
            {
                pTextWriter->Print("Third");
            }

            // カーソル選択枠の作成
            pPrimitiveRenderer->SetLineWidth(3.f);
            nn::util::Color4u8 color = { 50,  255, 255, m_FrameTwinkleCounter };
            pPrimitiveRenderer->SetColor(color);

            pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X + 195, 240, 80, 25);
        }
    }

    void Layout::DrawNpadWriteDatabaseIdVer(GraphicsSystem * pGraphicsSystem,
                                      nn::gfx::util::DebugFontTextWriter * pTextWriter,
                                      frm::CursolPosition cursol,
                                      size_t index)
    {
        int X = 350;
        int Y = 200;

        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();

        if (cursol.CursolStatus == CursolStatus_FixedNpadWriteDataBaseIdVer)
        {
            // 枠の作成
            pTextWriter->SetScale(1.f, 1.f);
            pTextWriter->SetCursor(X, Y);
            pTextWriter->SetTextColor(Color::Black);
            pTextWriter->Print("Write Database Select...");

            pPrimitiveRenderer->SetLineWidth(1.f);
            pPrimitiveRenderer->SetColor(QuorterColor::White);
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DRect(&pGraphicsSystem->GetCommandBuffer(), X, Y, 330, 100);
            pPrimitiveRenderer->SetColor(Color::Gray);
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X, Y, 330, 100);

            // 項目の作成
            X += 20;
            Y += 40;

            pTextWriter->SetCursor(X, Y);
            pTextWriter->Print("Write Database ID No.");
            pTextWriter->SetCursor(X + 200, Y);
            pTextWriter->Print("%d", cursol.dataBaseIdVer[index]);

            // カーソル選択枠の作成
            pPrimitiveRenderer->SetLineWidth(3.f);
            nn::util::Color4u8 color = { 50,  255, 255, m_FrameTwinkleCounter };
            pPrimitiveRenderer->SetColor(color);

            pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X + 195, 240, 80, 25);
        }
    }

    void Layout::DrawNpadSetFrMode(GraphicsSystem * pGraphicsSystem,
                                      nn::gfx::util::DebugFontTextWriter * pTextWriter,
                                      frm::CursolPosition cursol,
                                      size_t index)
    {
        int X = 350;
        int Y = 200;

        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();

        if (cursol.CursolStatus == CursolStatus_FixedNpadSetFrModeType)
        {
            // 枠の作成
            pTextWriter->SetScale(1.f, 1.f);
            pTextWriter->SetCursor(X, Y);
            pTextWriter->SetTextColor(Color::Black);
            pTextWriter->Print("Fr Mode Select...");

            pPrimitiveRenderer->SetLineWidth(1.f);
            pPrimitiveRenderer->SetColor(QuorterColor::White);
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DRect(&pGraphicsSystem->GetCommandBuffer(), X, Y, 330, 100);
            pPrimitiveRenderer->SetColor(Color::Gray);
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X, Y, 330, 100);

            // 項目の作成
            X += 20;
            Y += 40;

            pTextWriter->SetCursor(X, Y);
            pTextWriter->Print("Fr Mode");
            pTextWriter->SetCursor(X + 160, Y);
            pTextWriter->Print("%s", gPalma.ToString(static_cast<nn::hid::PalmaFrModeType>(cursol.frMode[index])));

            // カーソル選択枠の作成
            pPrimitiveRenderer->SetLineWidth(3.f);
            nn::util::Color4u8 color = { 50,  255, 255, m_FrameTwinkleCounter };
            pPrimitiveRenderer->SetColor(color);

            pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X + 155, 240, 120, 25);
        }
    }

    void Layout::DrawNpadSuspendFeatureSet(GraphicsSystem * pGraphicsSystem,
                                       nn::gfx::util::DebugFontTextWriter * pTextWriter,
                                       frm::CursolPosition cursol,
                                       size_t index)
    {
        int X = 350;
        int Y = 200;

        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();

        if (cursol.CursolStatus == CursolStatus_FixedNpadSuspendFeatureSet)
        {
            // 枠の作成
            pTextWriter->SetScale(1.f, 1.f);
            pTextWriter->SetCursor(X, Y);
            pTextWriter->SetTextColor(Color::Black);
            pTextWriter->Print("[Suspend FeatureSet Select...]");

            pPrimitiveRenderer->SetLineWidth(1.f);
            pPrimitiveRenderer->SetColor(QuorterColor::White);
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DRect(&pGraphicsSystem->GetCommandBuffer(), X, Y, 330, 180);
            pPrimitiveRenderer->SetColor(Color::Gray);
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X, Y, 330, 180);

            // 項目の作成
            X += 20;
            Y += 40;
            pTextWriter->SetCursor(X, Y);
            pTextWriter->Print("FrMode");
            pTextWriter->SetCursor(X + 180, Y);
            pTextWriter->Print("%s", cursol.featureSet[index].Test<nn::hid::PalmaFeature::FrMode>() ? "Suspend" : "ReStart");

            Y += 30;
            pTextWriter->SetCursor(X, Y);
            pTextWriter->Print("RumbleFeedBack");
            pTextWriter->SetCursor(X + 180, Y);
            pTextWriter->Print("%s", cursol.featureSet[index].Test<nn::hid::PalmaFeature::RumbleFeedback>() ? "Suspend" : "ReStart");

            Y += 30;
            pTextWriter->SetCursor(X, Y);
            pTextWriter->Print("Step");
            pTextWriter->SetCursor(X + 180, Y);
            pTextWriter->Print("%s", cursol.featureSet[index].Test<nn::hid::PalmaFeature::Step>() ? "Suspend" : "ReStart");

            Y += 30;
            pTextWriter->SetCursor(X, Y);
            pTextWriter->Print("MuteSwitch");
            pTextWriter->SetCursor(X + 180, Y);
            pTextWriter->Print("%s", cursol.featureSet[index].Test<nn::hid::PalmaFeature::MuteSwitch>() ? "Suspend" : "ReStart");

            // カーソル選択枠の作成
            pPrimitiveRenderer->SetLineWidth(3.f);
            nn::util::Color4u8 color = { 50,  255, 255, m_FrameTwinkleCounter };
            pPrimitiveRenderer->SetColor(color);

            pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X + 175, 240 + (30 * cursol.featureSetPosition[index]), 100, 25);
        }
    }

    void Layout::DrawNpadBoostMode(GraphicsSystem * pGraphicsSystem,
                                     nn::gfx::util::DebugFontTextWriter * pTextWriter,
                                     frm::CursolPosition cursol,
                                     size_t index)
    {
        int X = 350;
        int Y = 200;

        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();

        if (cursol.CursolStatus == CursolStatus_FixedNpadSetBoostMode)
        {
            // 枠の作成
            pTextWriter->SetScale(1.f, 1.f);
            pTextWriter->SetCursor(X, Y);
            pTextWriter->SetTextColor(Color::Black);
            pTextWriter->Print("Select BoostMode...");

            pPrimitiveRenderer->SetLineWidth(1.f);
            pPrimitiveRenderer->SetColor(QuorterColor::White);
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DRect(&pGraphicsSystem->GetCommandBuffer(), X, Y, 330, 100);
            pPrimitiveRenderer->SetColor(Color::Gray);
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X, Y, 330, 100);

            // 項目の作成
            X += 20;
            Y += 40;

            pTextWriter->SetCursor(X, Y);
            pTextWriter->Print("Boost Mode");
            pTextWriter->SetCursor(X + 200, Y);
            pTextWriter->SetTextColor(Color::Black);
            pTextWriter->Print("%s", cursol.isBoostMode[index] == true ? "ON" : "OFF");

            // カーソル選択枠の作成
            pPrimitiveRenderer->SetLineWidth(3.f);
            nn::util::Color4u8 color = { 50,  255, 255, m_FrameTwinkleCounter };
            pPrimitiveRenderer->SetColor(color);

            pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X + 195, 240, 80, 25);
        }
    }

    void Layout::DrawNpadReadAppData(GraphicsSystem * pGraphicsSystem,
                                      nn::gfx::util::DebugFontTextWriter * pTextWriter,
                                      frm::CursolPosition cursol,
                                      size_t index)
    {
        int X = 350;
        int Y = 200;

        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();

        if (cursol.CursolStatus == CursolStatus_FixedNpadReadAppSection)
        {
            // 枠の作成
            pTextWriter->SetScale(1.f, 1.f);
            pTextWriter->SetCursor(X, Y);
            pTextWriter->SetTextColor(Color::Black);
            pTextWriter->Print("[Read AppSection...]");

            pPrimitiveRenderer->SetLineWidth(1.f);
            pPrimitiveRenderer->SetColor(QuorterColor::White);
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DRect(&pGraphicsSystem->GetCommandBuffer(), X, Y, 330, 180);
            pPrimitiveRenderer->SetColor(Color::Gray);
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X, Y, 330, 180);

            // 項目の作成
            X += 20;
            Y += 40;
            pTextWriter->SetCursor(X, Y);
            pTextWriter->Print("Size");
            pTextWriter->SetCursor(X + 120 + 5, Y);

            if (cursol.appSection[index].readSize > frm::AppSectionSizeMax)
            {
                pTextWriter->SetTextColor(Color::Red);
            }
            else
            {
                pTextWriter->SetTextColor(Color::Black);
            }
            pTextWriter->Print("%d", cursol.appSection[index].readSize);

            Y += 30;
            pTextWriter->SetTextColor(Color::Black);
            pTextWriter->SetCursor(X, Y);
            pTextWriter->Print("Address");
            pTextWriter->SetCursor(X + 120 + 5, Y);
            pTextWriter->Print("%d", cursol.appSection[index].address);

            Y += 30;
            pTextWriter->SetTextColor(Color::Black);
            pTextWriter->SetCursor(X, Y);
            pTextWriter->Print("FullRead");
            pTextWriter->SetCursor(X + 120 + 5, Y);
            pTextWriter->Print("%s", cursol.appSection[index].isFull == true ? "TRUE" : "FALSE");

            // カーソル選択枠の作成
            pPrimitiveRenderer->SetLineWidth(3.f);
            nn::util::Color4u8 color = { 50,  255, 255, m_FrameTwinkleCounter };
            pPrimitiveRenderer->SetColor(color);

            pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X + 120, 240 + (30 * cursol.appSection[index].position), 180, 25);
        }
    }

    void Layout::DrawNpadWriteAppData(GraphicsSystem * pGraphicsSystem,
                                           nn::gfx::util::DebugFontTextWriter * pTextWriter,
                                           frm::CursolPosition cursol,
                                           size_t index)
    {
        int X = 350;
        int Y = 200;

        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();

        if (cursol.CursolStatus == CursolStatus_FixedNpadWriteAppSection)
        {
            // 枠の作成
            pTextWriter->SetScale(1.f, 1.f);
            pTextWriter->SetCursor(X, Y);
            pTextWriter->SetTextColor(Color::Black);
            pTextWriter->Print("[Write AppSection...]");

            pPrimitiveRenderer->SetLineWidth(1.f);
            pPrimitiveRenderer->SetColor(QuorterColor::White);
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DRect(&pGraphicsSystem->GetCommandBuffer(), X, Y, 330, 180);
            pPrimitiveRenderer->SetColor(Color::Gray);
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X, Y, 330, 180);

            // 項目の作成
            X += 20;
            Y += 40;
            pTextWriter->SetCursor(X, Y);
            pTextWriter->Print("Data");
            pTextWriter->SetCursor(X + 120 + 5, Y);
            pTextWriter->Print("%s", gPalma.m_Appdata.data[cursol.appSection[index].fileIndex].c_str());

            Y += 30;
            pTextWriter->SetCursor(X, Y);
            pTextWriter->Print("Address");
            pTextWriter->SetCursor(X + 120 + 5, Y);
            pTextWriter->Print("%d", cursol.appSection[index].address);

            Y += 30;
            pTextWriter->SetTextColor(Color::Black);
            pTextWriter->SetCursor(X, Y);
            pTextWriter->Print("FullWrite");
            pTextWriter->SetCursor(X + 120 + 5, Y);
            pTextWriter->Print("%s", cursol.appSection[index].isFull == true ? "TRUE" : "FALSE");

            // カーソル選択枠の作成
            pPrimitiveRenderer->SetLineWidth(3.f);
            nn::util::Color4u8 color = { 50,  255, 255, m_FrameTwinkleCounter };
            pPrimitiveRenderer->SetColor(color);

            pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X + 120, 240 + (30 * cursol.appSection[index].position), 180, 25);
        }
    }

    void Layout::DrawNpadPlayLog(GraphicsSystem * pGraphicsSystem,
                                      nn::gfx::util::DebugFontTextWriter * pTextWriter,
                                      frm::CursolPosition cursol,
                                      size_t index)
    {
        int X = 350;
        int Y = 200;

        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();

        if (cursol.CursolStatus == CursolStatus_FixedNpadReadPlayLog || cursol.CursolStatus == CursolStatus_FixedNpadResetPlayLog)
        {
            // 枠の作成
            pTextWriter->SetScale(1.f, 1.f);
            pTextWriter->SetCursor(X, Y);
            pTextWriter->SetTextColor(Color::Black);
            pTextWriter->Print("PlayLog Index Select...");

            pPrimitiveRenderer->SetLineWidth(1.f);
            pPrimitiveRenderer->SetColor(QuorterColor::White);
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DRect(&pGraphicsSystem->GetCommandBuffer(), X, Y, 330, 100);
            pPrimitiveRenderer->SetColor(Color::Gray);
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X, Y, 330, 100);

            // 項目の作成
            X += 20;
            Y += 40;

            pTextWriter->SetCursor(X, Y);
            pTextWriter->Print("PlayLogIndex No.");
            pTextWriter->SetCursor(X + 200, Y);

            pTextWriter->SetTextColor(Color::Black);

            pTextWriter->Print("%d", cursol.playLogIndex[index]);

            // カーソル選択枠の作成
            pPrimitiveRenderer->SetLineWidth(3.f);
            nn::util::Color4u8 color = { 50,  255, 255, m_FrameTwinkleCounter };
            pPrimitiveRenderer->SetColor(color);

            pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X + 195, 240, 80, 25);
        }
    }

    void Layout::DrawGattInfomation(GraphicsSystem * pGraphicsSystem, nn::gfx::util::DebugFontTextWriter * pTextWriter, util::BleState* state)
    {
        int X = 920;
        int Y = 30;

        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();

        // 枠の作成
        pPrimitiveRenderer->SetColor(HalfColor::White);
        pPrimitiveRenderer->SetLineWidth(1.f);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X, Y, 350, 210);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X + 2, Y + 2, 350 - 4, 210 - 4);

        pTextWriter->SetScale(1.f, 1.f);
        pTextWriter->SetTextColor(Color::White);
        pTextWriter->SetCursor(X + 10, Y);
        pTextWriter->Print("[Gatt Info.]");
        for (auto index = 0; index < state->connectionInfo.dataNum; index++)
        {
            pTextWriter->SetScale(1.f, 1.f);
            pTextWriter->SetCursor(X + 10, Y + 25 + (index * 60));
            pTextWriter->Print("Address%d : ",index);
            pTextWriter->SetCursor(X + 110, Y + 25 + (index * 60));
            for (auto i = 0; i < 6; i++)
            {
                pTextWriter->Print("%02x ", state->connectionInfo.data[index].address.address[i]);
            }
            pTextWriter->SetScale(0.8f, 0.8f);
            pTextWriter->SetCursor(X + 10, Y + 50 + (index * 60));
            pTextWriter->Print(">>Ver : ", index);
            pTextWriter->SetCursor(X + 65, Y + 50 + (index * 60));
            for (auto len = 0; len < state->connectionInfo.Serv[index].fwVerLength; len++)
            {
                pTextWriter->Print("%c", static_cast<char>(state->connectionInfo.Serv[index].fwVer[len]));
            }
        }
    }

    void Layout::DrawActivity(GraphicsSystem * pGraphicsSystem, nn::gfx::util::DebugFontTextWriter * pTextWriter, frm::CursolPosition cursol)
    {
        int X = 920;
        int Y = 30;

        nn::util::Color4u8 color = { 50,  255, 255, m_FrameTwinkleCounter };

        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();

        // PalmaActivity枠の作成
        pPrimitiveRenderer->SetColor(ClearColor::Gray);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DRect(&pGraphicsSystem->GetCommandBuffer(), X, Y, 350, 680);

        pTextWriter->SetScale(1.f, 1.f);
        pTextWriter->SetTextColor(Color::White);
        pTextWriter->SetCursor(X + 30, Y + 235);
        pTextWriter->Print("[Path]");

        pTextWriter->SetCursor(X + 85, Y + 235);
        pTextWriter->Print("├ %s", gFs.m_mount.rootPath.c_str());

        pPrimitiveRenderer->SetLineWidth(1.f);

        pPrimitiveRenderer->SetColor(HalfColor::Red);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DRect(&pGraphicsSystem->GetCommandBuffer(), X + 20, Y + 235, 310, 25);
        pPrimitiveRenderer->SetColor(HalfColor::White);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X + 20, Y + 235, 310, 25);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X + 20 + 2, Y + 235 + 2, 310 - 4, 25 - 4);
        if (cursol.CursolStatus == CursolStatus_FixedActivity)
        {
            if (cursol.activity == 0)
            {
                pPrimitiveRenderer->SetLineWidth(3.f);
                pPrimitiveRenderer->SetColor(color);
                pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X + 20, Y + 235, 310, 25);   // カーソル選択枠
            }
        }

        // 音声データ枠の作成
        DrawSound(pGraphicsSystem, pTextWriter, cursol);

        // LEDデータ枠の作成
        DrawRgbLed(pGraphicsSystem, pTextWriter, cursol);

        // カーソル選択枠の作成
        if (cursol.topForcus == 1)
        {
            pPrimitiveRenderer->SetLineWidth(2.5f);
            pPrimitiveRenderer->SetColor(color);

            if (cursol.CursolStatus == CursolStatus_Top)
            {
                pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X, Y, 350, 680);
            }
        }
        DrawWriteSound(pGraphicsSystem, pTextWriter, cursol);
        DrawLoadFile(pGraphicsSystem, pTextWriter, cursol);

        DrawWriteRgbLed(pGraphicsSystem, pTextWriter, cursol);
    }

    void Layout::DrawSound(GraphicsSystem * pGraphicsSystem, nn::gfx::util::DebugFontTextWriter * pTextWriter, frm::CursolPosition cursol)
    {
        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();

        int X = 940;
        int Y = 300;
        nn::util::Color4u8 color = { 50,  255, 255, m_FrameTwinkleCounter };

        pTextWriter->SetScale(1.f, 1.f);
        pTextWriter->SetTextColor(Color::White);
        pTextWriter->SetCursor(X + 10, Y);
        pTextWriter->Print("[Sound Data]");

        // Sound枠の作成
        pPrimitiveRenderer->SetLineWidth(1.f);
        pPrimitiveRenderer->SetColor(HalfColor::Red);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DRect(&pGraphicsSystem->GetCommandBuffer(), X, Y, 310, 170);
        pPrimitiveRenderer->SetColor(HalfColor::White);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X, Y, 310, 170);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X + 2, Y + 2, 310 - 4, 170 - 4);
        if (cursol.CursolStatus == CursolStatus_FixedActivity)
        {
            if (cursol.activity == 1)
            {
                pPrimitiveRenderer->SetLineWidth(3.f);
                pPrimitiveRenderer->SetColor(color);
                pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X, Y, 310, 170);   // カーソル選択枠
            }
        }

        // 項目の作成
        X += 30;
        Y += 30;
        if (cursol.CursolStatus == CursolStatus_FixedActivitySound)
        {
            pPrimitiveRenderer->SetLineWidth(2.5f);
            pPrimitiveRenderer->SetColor(color);
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X + 20, Y, 240, 25);  // カーソル選択枠
        }

        for (auto index = cursol.sound; index < cursol.sound + 5; index++)
        {
            if (index > gPalma.m_ActivityFile.waveCount - 1)
            {
                break;
            }
            pTextWriter->SetCursor(X, Y);
            pTextWriter->Print("├ %s", gPalma.m_ActivityFile.wave[index].c_str());

            Y += 25;
        }
    }

    void Layout::DrawRgbLed(GraphicsSystem * pGraphicsSystem, nn::gfx::util::DebugFontTextWriter * pTextWriter, frm::CursolPosition cursol)
    {
        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();

        int X = 940;
        int Y = 480;
        nn::util::Color4u8 color = { 50,  255, 255, m_FrameTwinkleCounter };

        pTextWriter->SetCursor(X + 10, Y);
        pTextWriter->Print("[RgbLed Data]");

        // RgbLed枠の作成
        pPrimitiveRenderer->SetLineWidth(1.f);
        pPrimitiveRenderer->SetColor(HalfColor::Red);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DRect(&pGraphicsSystem->GetCommandBuffer(), X, Y, 310, 170);
        pPrimitiveRenderer->SetColor(HalfColor::White);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X, Y, 310, 170);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X + 2, Y + 2, 310 - 4, 170 - 4);
        if (cursol.CursolStatus == CursolStatus_FixedActivity)
        {
            if (cursol.activity == 2)
            {
                pPrimitiveRenderer->SetLineWidth(3.f);
                pPrimitiveRenderer->SetColor(color);
                pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X, Y, 310, 170);   // カーソル選択枠
            }
        }

        // 項目の作成
        X += 30;
        Y += 30;
        if (cursol.CursolStatus == CursolStatus_FixedActivityRgbLed)
        {
            pPrimitiveRenderer->SetLineWidth(2.5f);
            pPrimitiveRenderer->SetColor(color);
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X + 20, Y, 240, 25);   // カーソル選択枠
        }

        for (auto index = cursol.rgbLed; index < cursol.rgbLed + 5; index++)
        {
            if (index > gPalma.m_ActivityFile.rgbLedCount - 1)
            {
                break;
            }
            pTextWriter->SetCursor(X, Y);
            pTextWriter->Print("├ %s", gPalma.m_ActivityFile.rgbLed[index].c_str());
            Y += 25;
        }
    }

    void Layout::DrawLoadFile(GraphicsSystem * pGraphicsSystem, nn::gfx::util::DebugFontTextWriter * pTextWriter, frm::CursolPosition cursol)
    {
        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();

        int X = 940;
        int Y = 660;
        nn::util::Color4u8 color = { 50,  255, 255, m_FrameTwinkleCounter };

        // ボタンの作成
        pPrimitiveRenderer->SetLineWidth(1.f);
        pPrimitiveRenderer->SetColor(ClearColor::Red);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DRect(&pGraphicsSystem->GetCommandBuffer(), X, Y, 310, 40);
        pPrimitiveRenderer->SetLineWidth(1.f);
        pPrimitiveRenderer->SetColor(HalfColor::White);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X, Y, 310, 40);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X + 2, Y + 2, 310 - 4, 40 - 4);

        // ボタン文字の作成
        pTextWriter->SetScale(1.2, 1.2);
        auto length = pTextWriter->CalculateStringWidth("Load File");
        pTextWriter->SetTextColor(Color::White);
        pTextWriter->SetCursor(X + (310 / 2) - (length / 2), Y + 5);
        pTextWriter->Print("Load File");

        if (cursol.CursolStatus == CursolStatus_FixedActivity)
        {
            if (cursol.activity == 3)
            {
                pPrimitiveRenderer->SetLineWidth(3.f);
                pPrimitiveRenderer->SetColor(color);
                pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X, Y, 310, 40);   // カーソル選択枠
            }
        }
    }

    void Layout::DrawWriteSound(GraphicsSystem * pGraphicsSystem,
                                       nn::gfx::util::DebugFontTextWriter * pTextWriter,
                                       frm::CursolPosition cursol)
    {
        int X = 940;
        int Y = 480;

        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();

        if (cursol.CursolStatus == CursolStatus_WriteSound)
        {
            // 枠の作成
            pTextWriter->SetScale(1.f, 1.f);
            pTextWriter->SetCursor(X, Y);
            pTextWriter->SetTextColor(Color::Black);
            pTextWriter->Print("[Write Sound Select...]");

            pPrimitiveRenderer->SetLineWidth(1.f);
            pPrimitiveRenderer->SetColor(QuorterColor::White);
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DRect(&pGraphicsSystem->GetCommandBuffer(), X, Y, 310, 170);
            pPrimitiveRenderer->SetColor(Color::Gray);
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X, Y, 310, 170);

            // 項目の作成
            X += 20;
            Y += 40;

            pTextWriter->SetCursor(X, Y);
            pTextWriter->Print("Write Npad No.");
            pTextWriter->SetCursor(X + 160, Y);
            pTextWriter->Print("%s", gNpad.ToString(gNpad.ConvertNpadIdFromIndex(cursol.writeElement.npadIndex)));

            Y += 30;
            pTextWriter->SetCursor(X, Y);
            pTextWriter->Print("Sound Index No.");
            pTextWriter->SetCursor(X + 160, Y);
            if (cursol.writeElement.Entry.waveIndex >= frm::SoundIndexMax)
            {
                pTextWriter->SetTextColor(Color::Red);
            }
            else
            {
                pTextWriter->SetTextColor(Color::Black);
            }
            pTextWriter->Print("%d", cursol.writeElement.Entry.waveIndex);
            pTextWriter->SetTextColor(Color::Black);

            Y += 30;
            pTextWriter->SetCursor(X, Y);
            pTextWriter->Print("Sound Type");
            pTextWriter->SetCursor(X + 160, Y);
            pTextWriter->Print("%s", gPalma.ToString(cursol.writeElement.Entry.waveSet));

            // カーソル選択枠の作成
            pPrimitiveRenderer->SetLineWidth(3.f);
            nn::util::Color4u8 color = { 50,  255, 255, m_FrameTwinkleCounter };
            pPrimitiveRenderer->SetColor(color);

            pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X + 155, Y - 60 + (30 * cursol.writePosition), 100, 25);
        }
    }

    void Layout::DrawWriteRgbLed(GraphicsSystem * pGraphicsSystem,
                                nn::gfx::util::DebugFontTextWriter * pTextWriter,
                                frm::CursolPosition cursol)
    {
        int X = 940;
        int Y = 300;

        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();

        if (cursol.CursolStatus == CursolStatus_WriteRgbLed)
        {
            // 枠の作成
            pTextWriter->SetScale(1.f, 1.f);
            pTextWriter->SetCursor(X, Y);
            pTextWriter->SetTextColor(Color::Black);
            pTextWriter->Print("[Write RgbLed Select...]");

            pPrimitiveRenderer->SetLineWidth(1.f);
            pPrimitiveRenderer->SetColor(QuorterColor::White);
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DRect(&pGraphicsSystem->GetCommandBuffer(), X, Y, 310, 170);
            pPrimitiveRenderer->SetColor(Color::Gray);
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X, Y, 310, 170);

            // 項目の作成
            X += 20;
            Y += 40;

            pTextWriter->SetCursor(X, Y);
            pTextWriter->Print("Write Npad No.");
            pTextWriter->SetCursor(X + 160, Y);
            pTextWriter->Print("%s", gNpad.ToString(gNpad.ConvertNpadIdFromIndex(cursol.writeElement.npadIndex)));

            Y += 30;
            pTextWriter->SetCursor(X, Y);
            pTextWriter->Print("RgbLed Index No.");
            pTextWriter->SetCursor(X + 160, Y);
            if (cursol.writeElement.Entry.rgbLedPatternIndex >= frm::SoundIndexMax)
            {
                pTextWriter->SetTextColor(Color::Red);
            }
            else
            {
                pTextWriter->SetTextColor(Color::Black);
            }
            pTextWriter->Print("%d", cursol.writeElement.Entry.rgbLedPatternIndex);
            pTextWriter->SetTextColor(Color::Black);

            // カーソル選択枠の作成
            pPrimitiveRenderer->SetLineWidth(3.f);
            nn::util::Color4u8 color = { 50,  255, 255, m_FrameTwinkleCounter };
            pPrimitiveRenderer->SetColor(color);

            pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), X + 155, Y - 30 + (30 * cursol.writePosition), 100, 25);
        }
    }
}
