﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>

/*
 * We use OpenGL ES 3.2 context and APIs.
 */
#include <GLES3/gl32.h>

#include "detail/GraphicsHelper.h"


namespace s2d
{

/**
 * @brief   リザルトコード
 */
enum class ResultCode
{
    Ok,                 //!< 成功

    NotSupported,       //!< サポート外の操作
    OutOfMemory,        //!< メモリ (バッファ) 不足
    ReadFailed,         //!< ファイルの読み込みに失敗
    DecodeFailed        //!< データのデコードに失敗
};

/**
 * @brief   描画時の原点位置
 */
enum class OriginPosition
{
    Default,    //!< 中央原点、Y 座標上向き
    TopLeft     //!< 左上原点、Y 座標下向き
};

/**
 * @brief   ブレンドモード
 */
enum class BlendMode
{
    Normal,     //!< 通常
    Add,        //!< 加算
    Subtract,   //!< 減算
    Multiple,   //!< 乗算
    Screen,     //!< スクリーン
    Xor,        //!< XOR
    Inverse     //!< 反転
};

/**
 * @brief   2D 座標
 */
union Point2D
{
    struct
    {
        float x;    //!< X 座標
        float y;    //!< Y 座標
    };
    float v[2];
};

/**
 * @brief   3D 座標
 */
union Point3D
{
    struct
    {
        float x;    //!< X 座標
        float y;    //!< Y 座標
        float z;    //!< Z 座標
    };
    float v[3];
};

/**
 * @brief   サイズ
 */
union Size
{
    struct
    {
        float width;    //!< 幅
        float height;   //!< 高さ
    };
    float v[2];
};

/**
 * @brief   矩形
 */
union Rectangle
{
    struct
    {
        float x;            //!< X 座標
        float y;            //!< Y 座標
        float width;        //!< 幅
        float height;       //!< 高さ
    };
    struct
    {
        Point2D position;   //!< 位置
        Size    size;       //!< サイズ
    };
    float v[4];
};

/**
 * @brief   色
 */
union Color
{
    struct
    {
        uint8_t r;  //!< 赤
        uint8_t g;  //!< 緑
        uint8_t b;  //!< 青
        uint8_t a;  //!< アルファ値
    };
    uint8_t v[4];

    /**
     * @brief   アルファ値を変更した色を取得
     */
    Color Alpha(uint8_t alpha) const NN_NOEXCEPT
    {
        return Color{ { r, g, b, alpha } };
    }
};

/**
 * @brief   画像
 */
struct ImageData
{
    GLuint textureId;   //!< テクスチャ ID
    Size   size;        //!< 画像のサイズ
    bool   isDirty;     //!< ピクセルデータの変更済みフラグ
    char*  pPixelData;  //!< ピクセルデータ
};

/**
 * @brief   初期化パラメータ
 */
struct InitializationParameters
{
    int             vertexCountMax;     //!< 最大頂点数
    OriginPosition  origin;             //!< 原点位置
    Size            canvasSize;         //!< 描画領域のサイズ
    int             baseFontSize;       //!< 基準となるフォントサイズ

    /**
     * @brief   初期化パラメータのデフォルト値を生成
     *
     * @details 各パラメータにデフォルト値を格納した状態で生成します。
     *          下記の値が格納されます。
     *
     *          - vertexCountMax = 16384
     *          - origin = OriginPosition::TopLeft
     *          - canvasSize = { 1280, 720 }
     *          - baseFontSize = 48
     */
    static InitializationParameters CreateDefault() NN_NOEXCEPT;
};

/**
 * @brief   色のプリセット
 */
class Colors
{
    NN_DISALLOW_COPY(Colors);
    NN_DISALLOW_MOVE(Colors);

public:
    static const Color Black;
    static const Color Shadow;
    static const Color Smoke;
    static const Color DimGray;
    static const Color Gray;
    static const Color DarkGray;
    static const Color Silver;
    static const Color White;
    static const Color Red;
    static const Color DarkRed;
    static const Color Crymson;
    static const Color NeonRed;
    static const Color Green;
    static const Color Blue;
    static const Color Yellow;
    static const Color Magenta;
    static const Color Cyan;
    static const Color Orange;
    static const Color Gold;
    static const Color GoldenRod;
    static const Color DarkGoldenRod;
    static const Color Violet;
    static const Color LimeGreen;
    static const Color PaleGreen;
    static const Color Navy;
    static const Color RoyalBlue;
    static const Color NeonBlue;

private:
    Colors() NN_NOEXCEPT {}
    ~Colors() NN_NOEXCEPT {}
};

/**
 * @brief   デフォルトの描画領域サイズ
 */
const Size DefaultCanvasSize = { { 1280, 720 } };

/**
 * @brief   外字コード
 */
enum ExtensionCharacter : uint16_t
{
    // ボタン
    ExtensionCharacter_ButtonA = 0xE0A0u,
    ExtensionCharacter_ButtonB,
    ExtensionCharacter_ButtonX,
    ExtensionCharacter_ButtonY,
    ExtensionCharacter_ButtonL,
    ExtensionCharacter_ButtonR,
    ExtensionCharacter_ButtonZl,
    ExtensionCharacter_ButtonZr,
    ExtensionCharacter_ButtonSl,
    ExtensionCharacter_ButtonSr,
    ExtensionCharacter_ButtonArrow,
    ExtensionCharacter_ButtonRight,
    ExtensionCharacter_ButtonDown,
    ExtensionCharacter_ButtonUp,
    ExtensionCharacter_ButtonLeft,
    ExtensionCharacter_ButtonArrowUp,
    ExtensionCharacter_ButtonArrowDown,
    ExtensionCharacter_ButtonArrowLeft,
    ExtensionCharacter_ButtonArrowRight,
    ExtensionCharacter_ButtonPlus,
    ExtensionCharacter_ButtonMinus,
    ExtensionCharacter_ButtonJoyPlus,
    ExtensionCharacter_ButtonJoyMinus,
    ExtensionCharacter_ButtonPower,
    ExtensionCharacter_ButtonPower2,
    ExtensionCharacter_ButtonHome,
    ExtensionCharacter_ButtonCapture,

    ExtensionCharacter_ButtonDownRight = 0xE0CAu,
    ExtensionCharacter_ButtonLeftRight,
    ExtensionCharacter_ButtonUpRight,
    ExtensionCharacter_ButtonDownLeft,
    ExtensionCharacter_ButtonUpDown,
    ExtensionCharacter_ButtonUpLeft,
    ExtensionCharacter_ButtonCross,
    ExtensionCharacter_ButtonCrossUp,
    ExtensionCharacter_ButtonCrossDown,
    ExtensionCharacter_ButtonCrossLeft,
    ExtensionCharacter_ButtonCrossRight,
    ExtensionCharacter_ButtonCrossUpDown,
    ExtensionCharacter_ButtonCrossLeftRight,

    // ボタン (色反転)
    ExtensionCharacter_ButtonAInvert = 0xE0E0u,
    ExtensionCharacter_ButtonBInvert,
    ExtensionCharacter_ButtonXInvert,
    ExtensionCharacter_ButtonYInvert,
    ExtensionCharacter_ButtonLInvert,
    ExtensionCharacter_ButtonRInvert,
    ExtensionCharacter_ButtonZlInvert,
    ExtensionCharacter_ButtonZrInvert,
    ExtensionCharacter_ButtonSlInvert,
    ExtensionCharacter_ButtonSrInvert,
    ExtensionCharacter_ButtonArrowInvert,
    ExtensionCharacter_ButtonArrowUpInvert,
    ExtensionCharacter_ButtonArrowDownInvert,
    ExtensionCharacter_ButtonArrowLeftInvert,
    ExtensionCharacter_ButtonArrowRightInvert,
    ExtensionCharacter_ButtonPlusInvert,
    ExtensionCharacter_ButtonMinusInvert,
    ExtensionCharacter_ButtonJoyPlusInvert,
    ExtensionCharacter_ButtonJoyMinusInvert,
    ExtensionCharacter_ButtonPowerInvert,
    ExtensionCharacter_ButtonHomeInvert,
    ExtensionCharacter_ButtonCaptureInvert,

    ExtensionCharacter_ButtonCrossInvert = 0xE110u,
    ExtensionCharacter_ButtonCrossUpInvert,
    ExtensionCharacter_ButtonCrossDownInvert,
    ExtensionCharacter_ButtonCrossLeftInvert,
    ExtensionCharacter_ButtonCrossRightInvert,
    ExtensionCharacter_ButtonCrossUpDownInvert,
    ExtensionCharacter_ButtonCrossLeftRightInvert,

    // スティック
    ExtensionCharacter_Stick = 0xE0C0u,
    ExtensionCharacter_StickL,
    ExtensionCharacter_StickR,
    ExtensionCharacter_StickPush,
    ExtensionCharacter_StickPushL,
    ExtensionCharacter_StickPushR,
    ExtensionCharacter_StickRollLeftL,
    ExtensionCharacter_StickRollLeftR,
    ExtensionCharacter_StickRollRightL,
    ExtensionCharacter_StickRollRightR,

    ExtensionCharacter_StickInvert = 0xE100u,
    ExtensionCharacter_StickLInvert,
    ExtensionCharacter_StickRInvert,
    ExtensionCharacter_StickPushInvert,
    ExtensionCharacter_StickPushLInvert,
    ExtensionCharacter_StickPushRInvert,

    // デバイス
    ExtensionCharacter_Handheld = 0xE121u,
    ExtensionCharacter_JoyDual,
    ExtensionCharacter_JoyLeft,
    ExtensionCharacter_JoyRight,
    ExtensionCharacter_JoyLeftWithButton,
    ExtensionCharacter_JoyRightWithButton,
    ExtensionCharacter_JoyHorizontal,
    ExtensionCharacter_JoyLeftHorizontal,
    ExtensionCharacter_JoyRightHorizontal,
    ExtensionCharacter_JoyConGrip,
    ExtensionCharacter_JoyConGripEmpty,
    ExtensionCharacter_SwitchProController,

    // その他アイコン
    ExtensionCharacter_OceanSettings = 0xE130u,
    ExtensionCharacter_OceanFriend,
    ExtensionCharacter_OceanGameNews,
    ExtensionCharacter_OceanShop,
    ExtensionCharacter_OceanAlbum,
    ExtensionCharacter_OceanFullLauncher,
    ExtensionCharacter_OceanController,
    ExtensionCharacter_Mail,
    ExtensionCharacter_SmartDevice,
    ExtensionCharacter_Pc,
    ExtensionCharacter_Tv,
    ExtensionCharacter_Headphone,
    ExtensionCharacter_Speaker,

    ExtensionCharacter_ExclamationCircle = 0xE140u,
    ExtensionCharacter_ExclamationSquare,
    ExtensionCharacter_QuestionCircle,
    ExtensionCharacter_LineArrowUp,
    ExtensionCharacter_LineArrowDown,
    ExtensionCharacter_LineArrowLeft,
    ExtensionCharacter_LineArrowRight,
    ExtensionCharacter_ArrowUp,
    ExtensionCharacter_ArrowDown,
    ExtensionCharacter_ArrowLeft,
    ExtensionCharacter_ArrowRight,
    ExtensionCharacter_Check,
    ExtensionCharacter_Cross,
    ExtensionCharacter_InformationCircle,
    ExtensionCharacter_ExclamationCircleInvert,
    ExtensionCharacter_ExclamationSquareInvert,
    ExtensionCharacter_QuestionCircleInvert,
    ExtensionCharacter_InformationCircleInvert
};

}  // s2d
