﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <vector>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>

#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/system/hid_Npad.h>

#include "StartupArgument.h"

namespace {
void ProceedNpadStyleSet(const uint32_t& _command);
void ProceedNpadIdType(const uint32_t& _command);
void ProceedApplyNpadSystemCommonPolicy(const uint32_t& _command);

class StartupArgumentManager
{
NN_DISALLOW_COPY(StartupArgumentManager);
private:
    struct CommandData
    {
        const char* Name;
        const char* ShortName;
        uint32_t    Param;
        void(*Func)(const uint32_t& _command);
    };
    std::vector<CommandData> m_Commands;
    StartupArgumentManager() {}
public:
    struct ParamList {
        // static const uint16_t n =  0b00000000;    // null
        // static const uint16_t e =    0x01; // 0b00000001 enable
        static const uint16_t d =    0x02; // 0b00000010 disable / dual
        static const uint16_t h =    0x04; // 0b00000100 handheld / horizontal
        static const uint16_t r =    0x08; // 0b00001000 right
        static const uint16_t l =    0x10; // 0b00010000 left
        static const uint16_t f =    0x20; // 0b00100000 fullKey
        // static const uint16_t v =    0x40; // 0b01000000 vertical
        static const uint16_t used = 0x80; // 0b10000000 used_flag / length
    };
    void SetCommandData(const CommandData& _data) NN_NOEXCEPT
    {
        m_Commands.push_back(_data);
    }

    bool InputCommand(const char* _command) NN_NOEXCEPT
    {
        static const char s_ParamName[] = { 'e', 'd', 'h', 'r', 'l', 'f', 'v' };
        static int s_SelectCommand = -1;

        if (_command[0] != '-' && s_SelectCommand >= 0)
        {
            if (_command[0] >= '0' && _command[0] <= '8')
            {
                m_Commands.at(s_SelectCommand).Param |= ((_command[0] - '0') << 8) & 0xFF00;
                return true;
            }
            else
            {
                for (uint32_t i = 1, cnt = 0; i < ParamList::used && cnt < sizeof(s_ParamName) / sizeof(s_ParamName[0]); i <<= 1, ++cnt)
                {
                    if (_command[0] == s_ParamName[cnt])
                    {
                        m_Commands.at(s_SelectCommand).Param |= i & 0xFF;
                        return true;
                    }
                }
            }
        }
        else
        {
            size_t commandLength = std::strlen(_command) - 1;

            if (commandLength < 2) { return false; }

            s_SelectCommand = -1;
            for (size_t i = 0; i < m_Commands.size(); ++i)
            {
                size_t nameLength = std::strlen(m_Commands.at(i).Name);
                size_t shortNameLength = std::strlen(m_Commands.at(i).ShortName);

                if ((commandLength != nameLength) && (commandLength != shortNameLength)) { continue; }

                if (std::strncmp(&_command[1], m_Commands.at(i).Name, nameLength) == 0 ||
                    std::strncmp(&_command[1], m_Commands.at(i).ShortName, shortNameLength) == 0)
                {
                    m_Commands.at(i).Param |= ParamList::used & 0xFF;
                    s_SelectCommand = static_cast<int>(i);
                    return true;
                }
            }
        }
        return false;
    }

    void ProceedCommands() NN_NOEXCEPT
    {
        for (size_t i = 0; i < m_Commands.size(); ++i)
        {
            if (m_Commands.at(i).Param & ParamList::used)
            {
                m_Commands.at(i).Func(m_Commands.at(i).Param);
            }
        }
    }


    void ShowUsedCommands() NN_NOEXCEPT
    {
        for (size_t i = 0; i < m_Commands.size(); ++i)
        {
            if (m_Commands.at(i).Param & ParamList::used)
            {
                NN_LOG("% 20s : 0x%08x\n", m_Commands.at(i).Name, m_Commands.at(i).Param);
            }
        }
    }

    bool IsUsedSystemCommonPolicy() NN_NOEXCEPT
    {
        for (size_t i = 0; i < m_Commands.size(); ++i)
        {
            if (m_Commands.at(i).Param & ParamList::used &&
                m_Commands.at(i).Func == ProceedApplyNpadSystemCommonPolicy)
            {
                return true;
            }
        }
        return false;
    }

    static StartupArgumentManager& GetInstance()
    {
        static StartupArgumentManager s_Instance;
        return s_Instance;
    }
};

} // namespace

//------------------------------------------------------------------------------
//  引数起動
//------------------------------------------------------------------------------

void StartupArgument()
{
    static StartupArgumentManager& s_Arg = StartupArgumentManager::GetInstance();

    const int commandCount = nn::os::GetHostArgc();
    char**    command      = nn::os::GetHostArgv();

    // コマンドの追加
    s_Arg.SetCommandData({ "SystemCommon", "sys",   0, ProceedApplyNpadSystemCommonPolicy });
    s_Arg.SetCommandData({ "NpadId",       "id",    0, ProceedNpadIdType });
    s_Arg.SetCommandData({ "NpadStyle",    "style", 0, ProceedNpadStyleSet });

    for (int i = 1; i < commandCount; ++i)
    {
        if (!s_Arg.InputCommand(command[i]))
        {
            // 無効な値が入力されたときの処理
        }
    }

    s_Arg.ProceedCommands();

    s_Arg.ShowUsedCommands();
}

namespace {

void ProceedNpadStyleSet(const uint32_t& _command)
{
    if (StartupArgumentManager::GetInstance().IsUsedSystemCommonPolicy()) { return; }

    const uint32_t command = _command & 0xFF;
    nn::hid::NpadStyleSet styles;
    styles.Reset();

    styles.Set<nn::hid::NpadStyleFullKey>(!(command & StartupArgumentManager::ParamList::f));
    styles.Set<nn::hid::NpadStyleHandheld>(!(command & StartupArgumentManager::ParamList::h));
    styles.Set<nn::hid::NpadStyleJoyDual>(!(command & StartupArgumentManager::ParamList::d));
    styles.Set<nn::hid::NpadStyleJoyLeft>(!(command & StartupArgumentManager::ParamList::l));
    styles.Set<nn::hid::NpadStyleJoyRight>(!(command & StartupArgumentManager::ParamList::r));
    nn::hid::SetSupportedNpadStyleSet(styles);
}

void ProceedNpadIdType(const uint32_t& _command)
{
//    if (StartupArgumentManager::GetInstance().IsUsedSystemCommonPolicy()) { return; }
//    const int num = (_command & 0xFF00) >> 8;
// TODO : ID の起動パラメーター追加
// nn::hid::SetSupportedNpadIdTypes(supportedIds, NN_ARRAY_SIZE(supportedIds));
}

void ProceedApplyNpadSystemCommonPolicy(const uint32_t& _command)
{
    const uint32_t command = _command & 0xFF;

    if (!(command & StartupArgumentManager::ParamList::d))
    {
        nn::hid::system::ApplyNpadSystemCommonPolicy();
    }
}

} // namespace
