﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cmath>
#include <nn/nn_Assert.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/hid/detail/hid_SensorTypes.h>
#include <nn/util/util_MathTypes.h>

#include <nn/nn_Log.h>

struct SixAxisSensorStateEx
{
    ::nn::util::Float3        acceleration;     //!< 加速度センサーの各方向ごとの加速度の値です。単位は G です。
    ::nn::util::Float3        angularVelocity;  //!< ジャイロセンサーの各方向ごとの角速度の値です。360dps を 1.0 とする値です。
    ::nn::util::Float3        angle;            //!< 各方向ごとの角速度の値を積算して得られる回転角の値です。360度 を 1.0 とする値です。
    ::nn::hid::DirectionState direction;
    float                     accelerationLength[3];
    float                     accelerationLength3d;

    float GetLength(int index1, int index2) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_EQUAL(index1, index2);
        NN_SDK_REQUIRES_LESS_EQUAL(index1 + index2, 3);

        return accelerationLength[index1 + index2 - 1];
    }

    float GetAngle(int index1, int index2) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_EQUAL(index1, index2);
        NN_SDK_REQUIRES_LESS(index1, 3);
        NN_SDK_REQUIRES_LESS(index2, 3);

        return std::atan2(acceleration.v[index1], acceleration.v[index2]) * 180 / nn::util::FloatPi;
    }
};

//!< 6軸センサーの入力状態を蓄積する FIFO
class SixAxisSensorStateFifo : public RingBuffer<SixAxisSensorStateEx, 800>
{
private:
    SixAxisSensorStateEx m_Max;
    SixAxisSensorStateEx m_Min;
    bool m_IsMaxMinSet;
    nn::hid::SixAxisSensorHandle m_Handle;
    DeviceIndex m_Index;
    bool m_IsHandleAvailable;
    int64_t m_LastSamplingNumber;

public:
    void SetHandle(const nn::hid::SixAxisSensorHandle& handle, DeviceIndex index) NN_NOEXCEPT
    {
        m_Handle = handle;
        m_Index = index;
        m_IsHandleAvailable = true;
        nn::hid::StartSixAxisSensor(m_Handle);
        this->Clear();
    }

    void ClearHandle() NN_NOEXCEPT
    {
        m_IsHandleAvailable = false;
        m_Index = DeviceIndex_Standard;
        m_LastSamplingNumber = std::numeric_limits<int64_t>::min();
        this->Clear();
    }

    void Update() NN_NOEXCEPT
    {
        if (m_IsHandleAvailable)
        {
            nn::hid::SixAxisSensorState states[nn::hid::SixAxisSensorStateCountMax];
            auto count = nn::hid::GetSixAxisSensorStates(states, NN_ARRAY_SIZE(states), m_Handle);
            for (int i = count - 1; i >= 0; --i)
            {
                if (states[i].samplingNumber > m_LastSamplingNumber)
                {
                    SixAxisSensorStateEx stateEx;
                    stateEx.acceleration     = states[i].acceleration;
                    stateEx.angularVelocity  = states[i].angularVelocity;
                    stateEx.angle            = states[i].angle;
                    stateEx.direction        = states[i].direction;
                    stateEx.accelerationLength[0] = std::sqrt(stateEx.acceleration.x * stateEx.acceleration.x + stateEx.acceleration.y * stateEx.acceleration.y);
                    stateEx.accelerationLength[1] = std::sqrt(stateEx.acceleration.x * stateEx.acceleration.x + stateEx.acceleration.z * stateEx.acceleration.z);
                    stateEx.accelerationLength[2] = std::sqrt(stateEx.acceleration.z * stateEx.acceleration.z + stateEx.acceleration.y * stateEx.acceleration.y);
                    stateEx.accelerationLength3d = std::sqrt(stateEx.acceleration.x * stateEx.acceleration.x + stateEx.acceleration.y * stateEx.acceleration.y + stateEx.acceleration.z * stateEx.acceleration.z);
                    this->push_back(stateEx);

                    if (m_IsMaxMinSet == false)
                    {
                        m_Max = stateEx;
                        m_Min = stateEx;
                        m_IsMaxMinSet = true;
                    }
                    for (int axis = 0; axis < 3; ++axis)
                    {
                        m_Max.acceleration.v[axis] = std::max(m_Max.acceleration.v[axis], stateEx.acceleration.v[axis]);
                        m_Max.angularVelocity.v[axis] = std::max(m_Max.angularVelocity.v[axis], stateEx.angularVelocity.v[axis]);
                        m_Max.accelerationLength[axis] = std::max(m_Max.accelerationLength[axis], stateEx.accelerationLength[axis]);
                        m_Min.acceleration.v[axis] = std::min(m_Min.acceleration.v[axis], stateEx.acceleration.v[axis]);
                        m_Min.angularVelocity.v[axis] = std::min(m_Min.angularVelocity.v[axis], stateEx.angularVelocity.v[axis]);
                        m_Min.accelerationLength[axis] = std::min(m_Min.accelerationLength[axis], stateEx.accelerationLength[axis]);
                    }
                    m_Max.accelerationLength3d = std::max(m_Max.accelerationLength3d, stateEx.accelerationLength3d);
                    m_Min.accelerationLength3d = std::min(m_Min.accelerationLength3d, stateEx.accelerationLength3d);

                    m_LastSamplingNumber = states[i].samplingNumber;
                }
            }
        }
    }

    void Clear() NN_NOEXCEPT
    {
        m_Max = SixAxisSensorStateEx();
        m_Min = SixAxisSensorStateEx();
        m_IsMaxMinSet = false;
        this->clear();
    }

    SixAxisSensorStateEx GetMax() NN_NOEXCEPT
    {
        return m_Max;
    }

    SixAxisSensorStateEx GetMin() NN_NOEXCEPT
    {
        return m_Min;
    }

    DeviceIndex GetDeviceIndex() NN_NOEXCEPT
    {
        return m_Index;
    }
};


//!< 生の6軸センサーの入力状態を蓄積する FIFO
class SixAxisSensorDriverStateFifo : public RingBuffer<nn::hid::detail::SixAxisSensorDriverState, 800>
{
private:
    nn::hid::detail::SixAxisSensorDriverState m_Max;
    nn::hid::detail::SixAxisSensorDriverState m_Min;
    bool m_IsMaxMinSet;

public:
    void AddSample(const nn::hid::detail::SixAxisSensorDriverState& input) NN_NOEXCEPT
    {
        if (this->size() == 0 ||
            this->back().sampleNumber < input.sampleNumber)
        {
            this->push_back(input);

            if (m_IsMaxMinSet == false)
            {
                m_Max = input;
                m_Min = input;
                m_IsMaxMinSet = true;
            }
            m_Max.accelerometer.x = std::max(m_Max.accelerometer.x, input.accelerometer.x);
            m_Max.accelerometer.y = std::max(m_Max.accelerometer.y, input.accelerometer.y);
            m_Max.accelerometer.z = std::max(m_Max.accelerometer.z, input.accelerometer.z);
            m_Max.gyroscope.x = std::max(m_Max.gyroscope.x, input.gyroscope.x);
            m_Max.gyroscope.y = std::max(m_Max.gyroscope.y, input.gyroscope.y);
            m_Max.gyroscope.z = std::max(m_Max.gyroscope.z, input.gyroscope.z);
        }
    }

    void Clear() NN_NOEXCEPT
    {
        m_Max = nn::hid::detail::SixAxisSensorDriverState();
        m_Min = nn::hid::detail::SixAxisSensorDriverState();
        m_IsMaxMinSet = false;
        this->clear();
    }

    nn::hid::detail::SixAxisSensorDriverState GetMax() NN_NOEXCEPT
    {
        return m_Max;
    }

    nn::hid::detail::SixAxisSensorDriverState GetMin() NN_NOEXCEPT
    {
        return m_Min;
    }
};
