﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>

#include "ScreenStatics.h"
#include "ScreenUtil.h"
#include "s2d/Simple2D.h"

namespace {

s2d::Color ConvertColor(const nn::util::Color4u8Type& color) NN_NOEXCEPT
{
    return { {color.v[0], color.v[1], color.v[2], 255} };
}

s2d::Color GetTextColor(const nn::util::Color4u8Type& color) NN_NOEXCEPT
{
    auto brightness = color.v[0] * 3 + color.v[1] * 6 + color.v[3] * 1;
    if (brightness > 1270)
    {
        return { {0, 0, 0, 255} };
    }

    return { {255, 255, 255, 255} };
}

} // namespace

std::string GetDeviceTypeString(const nn::hid::system::DeviceTypeSet& deviceType) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(deviceType.CountPopulation(), 1);

    if (deviceType.Test<nn::hid::system::DeviceType::FullKeyController>())
    {
        return "Pro Controller";
    }
    if (deviceType.Test<nn::hid::system::DeviceType::HandheldJoyLeft>())
    {
        return "Joy-Con(L) on Rail";
    }
    if (deviceType.Test<nn::hid::system::DeviceType::HandheldJoyRight>())
    {
        return "Joy-Con(R) on Rail";
    }
    if (deviceType.Test<nn::hid::system::DeviceType::JoyConLeft>())
    {
        return "Joy-Con(L)";
    }
    if (deviceType.Test<nn::hid::system::DeviceType::JoyConRight>())
    {
        return "Joy-Con(R)";
    }
    if (deviceType.Test<TestDeviceType::UsbCompatibleController>())
    {
        return "CompatFullKey(Usb)";
    }
    if (deviceType.Test<nn::hid::system::DeviceType::Palma>())
    {
        return "Palma";
    }
    if (deviceType.Test<nn::hid::system::DeviceType::Unknown>())
    {
        return "Unknown";
    }

    return "Future";
}

std::string GetDeviceTypeShort(const nn::hid::system::DeviceTypeSet& deviceType) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(deviceType.CountPopulation(), 1);

    if (deviceType.Test<nn::hid::system::DeviceType::FullKeyController>())
    {
        return "Pro";
    }
    if (deviceType.Test<nn::hid::system::DeviceType::HandheldJoyLeft>())
    {
        return "JoyL";
    }
    if (deviceType.Test<nn::hid::system::DeviceType::HandheldJoyRight>())
    {
        return "JoyR";
    }
    if (deviceType.Test<nn::hid::system::DeviceType::JoyConLeft>())
    {
        return "JoyL";
    }
    if (deviceType.Test<nn::hid::system::DeviceType::JoyConRight>())
    {
        return "JoyR";
    }
    if (deviceType.Test<TestDeviceType::UsbCompatibleController>())
    {
        return "Cmpt";
    }
    if (deviceType.Test<nn::hid::system::DeviceType::Unknown>())
    {
        return "---";
    }

    return "---";
}

void DrawDeviceType(RecordablePosition& position, const nn::hid::system::DeviceTypeSet deviceType) NN_NOEXCEPT
{
    s2d::SetTextColor(ColorSceneTextStrong01);
    s2d::DrawText(position.Get(), "%s", GetDeviceTypeString(deviceType).c_str());
    s2d::SetTextColor(ColorSceneTextDefault);
    position.x += XSizeDeviceType;
}

void DrawIndicator(RecordablePosition& position, ::nn::Bit8 pattern) NN_NOEXCEPT
{
    if (pattern == 0)
    {
        position.x += XSizeSingleIndicator * 1;
        s2d::DrawText(position.Get(), "\uE121");
        position.x += XSizeSingleIndicator * 3;
    }
    else
    {
        s2d::SetTextScale(TextScaleIndicator);
        for (int i = 0; i < 4; i++)
        {
            auto isOn = ((pattern & (0x01 << i)) != 0);
            s2d::SetTextColor(isOn ? ColorSceneIndicator : ColorSceneTextDefault);
            s2d::DrawText(position.Get(), "■");
            position.x += XSizeSingleIndicator;
        }
        s2d::SetTextScale(TextScaleDefault);
    }
    s2d::SetTextColor(ColorSceneTextDefault);
    position.x += XSizePadding;
}

void DrawColor(RecordablePosition& position, const nn::util::Color4u8Type& color) NN_NOEXCEPT
{
    auto text = GetTextColor(color);
    s2d::Rectangle background = { { position.x, position.y.value, XSizeColor, YSizeColor} };
    s2d::FillRectangle(background, ConvertColor(color));
    s2d::SetTextColor(text);
    s2d::DrawText(position.Get(), "%d-%d-%d", color.v[0], color.v[1], color.v[2]);
    s2d::SetTextColor(ColorSceneTextDefault);
    position.x += XSizeColor + XSizePadding;
}

void DrawArrow(const s2d::Point2D& startPoint, const s2d::Point2D& endPoint, const s2d::Color& color, float lineWidth, const float& size) NN_NOEXCEPT
{
    const float rad = 0.30;

    s2d::DrawLine(startPoint,
                  endPoint,
                  color,
                  lineWidth );

    float length = std::sqrt(std::pow((startPoint.x - endPoint.x), 2) + std::pow((startPoint.y - endPoint.y), 2));
    float baseX = (startPoint.x - endPoint.x) / length * size;
    float baseY = (startPoint.y - endPoint.y) / length * size;

    s2d::Point2D arrowEnd = { { endPoint.x + (baseX * std::cos(rad) - baseY * std::sin(rad)),
                                endPoint.y + (baseX * std::sin(rad) + baseY * std::cos(rad)) } };
    s2d::DrawLine(endPoint,
                  arrowEnd,
                  color,
                  lineWidth );


    arrowEnd.x = endPoint.x + (baseX * std::cos(-rad) - baseY * std::sin(-rad));
    arrowEnd.y = endPoint.y + (baseX * std::sin(-rad) + baseY * std::cos(-rad));
    s2d::DrawLine(endPoint,
                  arrowEnd,
                  color,
                  lineWidth );

}

void DrawInterface(RecordablePosition& position, const nn::hid::system::InterfaceType& interfaceType) NN_NOEXCEPT
{
    s2d::SetTextColor(ColorSceneTextStrong01);
    switch (interfaceType)
    {
    case nn::hid::system::InterfaceType_Bluetooth:
        s2d::DrawText(position.Get(), "Bluetooth");
        break;
    case nn::hid::system::InterfaceType_Rail:
        s2d::DrawText(position.Get(), "Rail");
        break;
    case nn::hid::system::InterfaceType_Usb:
        s2d::DrawText(position.Get(), "Usb");
        break;
    default:
        s2d::DrawText(position.Get(), "Unknown(%d)", interfaceType);
        break;
    }
    s2d::SetTextColor(ColorSceneTextDefault);
    position.x += XSizeInterfaceType;
}
