﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <complex>

#include "SceneSixAxisSensorDriver.h"
#include "ScreenStatics.h"
#include "ScreenUtil.h"
#include "s2d/Simple2D.h"

namespace {

const LineChartUiParameterConfigList LineChartUiParameterConfigListSensor =
{
    {
        {
            {200.0f, 600.0f, 20.0f, 1.1f},  // xSampleCount
            {32768, 32768, 2.0f, 1.1f}, // yScale
            {0.0f, 32768, -32768, 0.0f}   // yShift
        }
    }
};

class ILineChartAccessorSensor : public ILineChartDataSetAccessor
{
protected:
    SixAxisSensorDriverStateFifo* m_pFifo;
public:
    virtual int size() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFifo->size();
    }
    ILineChartAccessorSensor(const std::string label, const s2d::Color& color) NN_NOEXCEPT
    {
        m_Label = label;
        m_Color = color;
    }
    void SetSixAxisSensorFifo(SixAxisSensorDriverStateFifo* pFifo) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pFifo);
        m_pFifo = pFifo;
    }
};

enum SixAxisSensorAxis
{
    SixAxisSensorAxis_X = 0,
    SixAxisSensorAxis_Y = 1,
    SixAxisSensorAxis_Z = 2,
};

class LineChartAccessorAccelerometer final : public ILineChartAccessorSensor
{
    SixAxisSensorAxis m_Axis;

public:
    LineChartAccessorAccelerometer(const std::string label, const s2d::Color& color, SixAxisSensorAxis axis) NN_NOEXCEPT :
        ILineChartAccessorSensor(label, color),
        m_Axis(axis)
    {
        // 何もしない
    };
    virtual float at(int index) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFifo->at(index).accelerometer.v[m_Axis];
    }
    virtual const char* backStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%d", m_pFifo->back().accelerometer.v[m_Axis]);
        return buf;
    }
    virtual const char* maxStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%d", m_pFifo->GetMax().accelerometer.v[m_Axis]);
        return buf;
    }
    virtual const char* minStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%d", m_pFifo->GetMin().accelerometer.v[m_Axis]);
        return buf;
    }
};

class LineChartAccessorGyroscope final : public ILineChartAccessorSensor
{
    SixAxisSensorAxis m_Axis;

public:
    LineChartAccessorGyroscope(const std::string label, const s2d::Color& color, SixAxisSensorAxis axis) NN_NOEXCEPT :
        ILineChartAccessorSensor(label, color),
        m_Axis(axis)
    {
        // 何もしない
    };
    virtual float at(int index) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFifo->at(index).gyroscope.v[m_Axis];
    }
    virtual const char* backStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%d", m_pFifo->back().gyroscope.v[m_Axis]);
        return buf;
    }
    virtual const char* maxStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%d", m_pFifo->GetMax().gyroscope.v[m_Axis]);
        return buf;
    }
    virtual const char* minStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%d", m_pFifo->GetMin().gyroscope.v[m_Axis]);
        return buf;
    }
};

} // namespace

SceneSixAxisSensorDriver::SceneSixAxisSensorDriver() NN_NOEXCEPT :
    m_LineChartAccelerometer(LineChartUiParameterConfigListSensor, false),
    m_LineChartGyroscope(LineChartUiParameterConfigListSensor, false),
    m_IsMenuLocked(false)
{
}

void SceneSixAxisSensorDriver::OnSceneFocused() NN_NOEXCEPT
{
    m_pControllers->ClearSelectedController();
}

void SceneSixAxisSensorDriver::SetUniquePadList(UniquePadList* pControllers) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pControllers);
    m_pControllers = pControllers;
}

void SceneSixAxisSensorDriver::DrawImpl(RecordablePosition& position) NN_NOEXCEPT
{
    float xOrigin = position.x;

    LineChartAccessorAccelerometer accXAccessor   = { "AccX", ColorSceneTextStrong01, SixAxisSensorAxis_X };
    LineChartAccessorAccelerometer accYAccessor   = { "AccY", ColorSceneTextStrong04, SixAxisSensorAxis_Y };
    LineChartAccessorAccelerometer accZAccessor   = { "AccZ", ColorSceneTextStrong03, SixAxisSensorAxis_Z };
    LineChartAccessorGyroscope gyroXAccessor   = { "GyroX", ColorSceneTextStrong01, SixAxisSensorAxis_X };
    LineChartAccessorGyroscope gyroYAccessor   = { "GyroY", ColorSceneTextStrong04, SixAxisSensorAxis_Y };
    LineChartAccessorGyroscope gyroZAccessor   = { "GyroZ", ColorSceneTextStrong03, SixAxisSensorAxis_Z };

    ILineChartAccessorSensor* pLineChartAccelerometerAccessors[] = {
        &accXAccessor,
        &accYAccessor,
        &accZAccessor,
    };
    ILineChartAccessorSensor* pLineChartGyroscopeAccessors[] = {
        &gyroXAccessor,
        &gyroYAccessor,
        &gyroZAccessor,
    };
    bool isDeviceAvailable = false;

    UniquePadAccessor* pUniquePadAccessor;
    if (m_pControllers->GetSelectedController(&pUniquePadAccessor) == true)
    {
        auto pFifo = pUniquePadAccessor->GetSixAxisSensorFifo();

        for (auto& pAccessor : pLineChartAccelerometerAccessors)
        {
            pAccessor->SetSixAxisSensorFifo(pFifo);
        }
        for (auto& pAccessor : pLineChartGyroscopeAccessors)
        {
            pAccessor->SetSixAxisSensorFifo(pFifo);
        }
        isDeviceAvailable = true;
    }

    // グラフの描画
    position.x = xOrigin;
    position.y += YSizeLine;
    const s2d::Size size = { {XSizeLineChartFull, YSizeSensorDriverLineChart} };
    m_LineChartAccelerometer.Draw(position.Get(), size, false, reinterpret_cast<ILineChartDataSetAccessor**>(&pLineChartAccelerometerAccessors), (isDeviceAvailable) ? NN_ARRAY_SIZE(pLineChartAccelerometerAccessors) : 0);
    position.y += YSizeSensorDriverLineChartFull + YSizeLine * 2;
    m_LineChartGyroscope.Draw(position.Get(), size, false, reinterpret_cast<ILineChartDataSetAccessor**>(&pLineChartGyroscopeAccessors), (isDeviceAvailable) ? NN_ARRAY_SIZE(pLineChartGyroscopeAccessors) : 0);

}

PackedButton SceneSixAxisSensorDriver::Control(const PackedButton& button) NN_NOEXCEPT
{
    const nn::hid::NpadButtonSet LockButtons = nn::hid::NpadButton::Plus::Mask | nn::hid::NpadButton::Minus::Mask;

    if ((button.longTrigger & LockButtons).IsAnyOn())
    {
        m_IsMenuLocked = !m_IsMenuLocked;
    }
    if (m_IsMenuLocked)
    {
        auto triggerWithRepeat = button.trigger | button.repeat;
        if (triggerWithRepeat.Test<nn::hid::NpadButton::Up>())
        {
            m_LineChartAccelerometer.Decrement(LineChart::UiParameterType_YScale);
            m_LineChartGyroscope.Decrement(LineChart::UiParameterType_YScale);
        }
        if (triggerWithRepeat.Test<nn::hid::NpadButton::Down>())
        {
            m_LineChartAccelerometer.Increment(LineChart::UiParameterType_YScale);
            m_LineChartGyroscope.Increment(LineChart::UiParameterType_YScale);
        }
        if (triggerWithRepeat.Test<nn::hid::NpadButton::Left>())
        {
            m_LineChartAccelerometer.Decrement(LineChart::UiParameterType_XSampleCount);
            m_LineChartGyroscope.Decrement(LineChart::UiParameterType_XSampleCount);
        }
        if (triggerWithRepeat.Test<nn::hid::NpadButton::Right>())
        {
            m_LineChartAccelerometer.Increment(LineChart::UiParameterType_XSampleCount);
            m_LineChartGyroscope.Increment(LineChart::UiParameterType_XSampleCount);
        }
        if (triggerWithRepeat.Test<nn::hid::NpadButton::X>())
        {
            m_LineChartAccelerometer.Increment(LineChart::UiParameterType_YShift);
            m_LineChartGyroscope.Increment(LineChart::UiParameterType_YShift);
        }
        if (triggerWithRepeat.Test<nn::hid::NpadButton::B>())
        {
            m_LineChartAccelerometer.Decrement(LineChart::UiParameterType_YShift);
            m_LineChartGyroscope.Decrement(LineChart::UiParameterType_YShift);
        }
        if (triggerWithRepeat.Test<nn::hid::NpadButton::L>() ||
            triggerWithRepeat.Test<nn::hid::NpadButton::R>())
        {
            m_pControllers->ResetSelectedControllerFifo();
        }

        nn::hid::NpadButtonSet value;
        value.Reset();
        return button & value;
    }

    return button & ~LockButtons;
}
