﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/hid/system/hid_Nfc.h>

#include "NpadIdDefine.h"
#include "SceneControllerStatus.h"
#include "ScreenStatics.h"
#include "ScreenUtil.h"
#include "s2d/Simple2D.h"

s2d::Color ConvertColor(const nn::util::Color4u8Type& color) NN_NOEXCEPT;
void DrawNpadButtonState(RecordablePosition& position, const nn::hid::NpadButtonSet& state, bool unSupportedButtonPress, bool handheldButtonPress) NN_NOEXCEPT;
void DrawNpadAttributes(RecordablePosition& position, const nn::hid::NpadAttributesSet& state) NN_NOEXCEPT;
void DrawAnalogStickState(RecordablePosition& position, const nn::hid::AnalogStickState& state) NN_NOEXCEPT;
void DrawStyleNameFull(RecordablePosition& position, const NpadStyleDefinition& style) NN_NOEXCEPT;
void DrawJoyAssignment(RecordablePosition& position, nn::hid::NpadJoyAssignmentMode mode) NN_NOEXCEPT;
void DrawBattery(RecordablePosition& position, nn::hid::system::NpadPowerInfo power) NN_NOEXCEPT;
void DrawBooleanString(RecordablePosition& position, bool state) NN_NOEXCEPT;
void DrawUiInfo(RecordablePosition& position, NpadControllerUnit& controller) NN_NOEXCEPT;
void DrawNfcState(RecordablePosition& position, bool isAvailable, bool isActivated) NN_NOEXCEPT;
void DrawIrSensorState(RecordablePosition& position, nn::hid::system::IrSensorState state) NN_NOEXCEPT;
void DrawUsbConnected(RecordablePosition& position, bool state) NN_NOEXCEPT;
void DrawSystemExtInfo(RecordablePosition& position, NpadControllerUnit& controller) NN_NOEXCEPT;
void DrawNpadInfo(RecordablePosition& position, NpadControllerUnit& controller) NN_NOEXCEPT;

struct ButtonString
{
    nn::hid::NpadButtonSet button;
    std::string string;
};

const ButtonString ButtonStringList[] = {
    { nn::hid::NpadButton::A::Mask,       "\uE0A0" },
    { nn::hid::NpadButton::B::Mask,       "\uE0A1" },
    { nn::hid::NpadButton::X::Mask,       "\uE0A2" },
    { nn::hid::NpadButton::Y::Mask,       "\uE0A3" },
    { nn::hid::NpadButton::L::Mask,       "\uE0A4" },
    { nn::hid::NpadButton::R::Mask,       "\uE0A5" },
    { nn::hid::NpadButton::ZL::Mask,      "\uE0A6" },
    { nn::hid::NpadButton::ZR::Mask,      "\uE0A7" },
    { nn::hid::NpadButton::Right::Mask,   "\uE0AB" },
    { nn::hid::NpadButton::Down::Mask,    "\uE0AC" },
    { nn::hid::NpadButton::Up::Mask,      "\uE0AD" },
    { nn::hid::NpadButton::Left::Mask,    "\uE0AE" },
    { nn::hid::NpadButton::Plus::Mask,    "\uE0B3" },
    { nn::hid::NpadButton::Minus::Mask,   "\uE0B4" },
    { nn::hid::NpadButton::StickL::Mask,  "\uE0C4" },
    { nn::hid::NpadButton::StickR::Mask,  "\uE0C5" },
    { nn::hid::NpadJoyButton::LeftSL::Mask,  "\uE0A8" },
    { nn::hid::NpadJoyButton::LeftSR::Mask,  "\uE0A9" },
    { nn::hid::NpadJoyButton::RightSL::Mask,  "\uE0A8" },
    { nn::hid::NpadJoyButton::RightSR::Mask,  "\uE0A9" }
};

void DrawNpadButtonState(RecordablePosition& position, const nn::hid::NpadButtonSet& state, bool unsupportedButtonPress, bool handheldButtonPress) NN_NOEXCEPT
{
    position.x += PrintAndStep(position.Get(), "Buttons:", true);

    if (state.IsAllOff() && unsupportedButtonPress == false && handheldButtonPress == false)
    {
        s2d::DrawText(position.Get(), "NONE");
    }
    s2d::SetTextColor(ColorSceneTextStrong02);
    s2d::SetTextScale(TextScaleButton);
    for (int i = 0; i < NN_ARRAY_SIZE(ButtonStringList); ++i)
    {
        auto singleButtonState = (state & ButtonStringList[i].button);
        if (singleButtonState.IsAnyOn())
        {
            if (singleButtonState.Test<nn::hid::NpadJoyButton::LeftSL>() || singleButtonState.Test<nn::hid::NpadJoyButton::LeftSR>())
            {
                s2d::DrawText(position.Get(), "\uE123");
                position.x += XSizeButtonUnit * 0.5f;
            }
            if (singleButtonState.Test<nn::hid::NpadJoyButton::RightSL>() || singleButtonState.Test<nn::hid::NpadJoyButton::RightSR>())
            {
                s2d::DrawText(position.Get(), "\uE124");
                position.x += XSizeButtonUnit * 0.5f;
            }

            s2d::DrawText(position.Get(), "%s", ButtonStringList[i].string.c_str());
            position.x += XSizeButtonUnit;
        }
    }

    if (state.Test<nn::hid::NpadButton::StickLLeft>())
    {
        s2d::SetTextScale(TextScaleStickEmulationHorizontal);
        position.x -= XSizeButtonStickEmuNeg;
        s2d::DrawText(position.Get(), "\uE149");
        s2d::SetTextScale(TextScaleButton);
        position.x += XSizeButtonStickEmuNeg;
        s2d::DrawText(position.Get(), "\uE0C1");
        position.x += XSizeButtonUnit;
    }

    if (state.Test<nn::hid::NpadButton::StickLRight>())
    {
        s2d::SetTextScale(TextScaleStickEmulationHorizontal);
        position.x += XSizeButtonStickEmuPos;
        s2d::DrawText(position.Get(), "\uE14A");
        s2d::SetTextScale(TextScaleButton);
        position.x -= XSizeButtonStickEmuPos;
        s2d::DrawText(position.Get(), "\uE0C1");
        position.x += XSizeButtonUnit;
    }


    if (state.Test<nn::hid::NpadButton::StickLUp>())
    {
        s2d::SetTextScale(TextScaleStickEmulationVertical);
        position.y -= XSizeButtonStickEmuNeg;
        s2d::DrawText(position.Get(), "\uE147");
        s2d::SetTextScale(TextScaleButton);
        position.y += XSizeButtonStickEmuNeg;
        s2d::DrawText(position.Get(), "\uE0C1");
        position.x += XSizeButtonUnit;
    }

    if (state.Test<nn::hid::NpadButton::StickLDown>())
    {
        s2d::SetTextScale(TextScaleStickEmulationVertical);
        position.y += XSizeButtonStickEmuPos;
        s2d::DrawText(position.Get(), "\uE148");
        s2d::SetTextScale(TextScaleButton);
        position.y -= XSizeButtonStickEmuPos;
        s2d::DrawText(position.Get(), "\uE0C1");
        position.x += XSizeButtonUnit;
    }

    if (state.Test<nn::hid::NpadButton::StickRLeft>())
    {
        s2d::SetTextScale(TextScaleStickEmulationHorizontal);
        position.x -= XSizeButtonStickEmuNeg;
        s2d::DrawText(position.Get(), "\uE149");
        s2d::SetTextScale(TextScaleButton);
        position.x += XSizeButtonStickEmuNeg;
        s2d::DrawText(position.Get(), "\uE0C2");
        position.x += XSizeButtonUnit;
    }

    if (state.Test<nn::hid::NpadButton::StickRRight>())
    {
        s2d::SetTextScale(TextScaleStickEmulationHorizontal);
        position.x += XSizeButtonStickEmuPos;
        s2d::DrawText(position.Get(), "\uE14A");
        s2d::SetTextScale(TextScaleButton);
        position.x -= XSizeButtonStickEmuPos;
        s2d::DrawText(position.Get(), "\uE0C2");
        position.x += XSizeButtonUnit;
    }


    if (state.Test<nn::hid::NpadButton::StickRUp>())
    {
        s2d::SetTextScale(TextScaleStickEmulationVertical);
        position.y -= XSizeButtonStickEmuNeg;
        s2d::DrawText(position.Get(), "\uE147");
        s2d::SetTextScale(TextScaleButton);
        position.y += XSizeButtonStickEmuNeg;
        s2d::DrawText(position.Get(), "\uE0C2");
        position.x += XSizeButtonUnit;
    }

    if (state.Test<nn::hid::NpadButton::StickRDown>())
    {
        s2d::SetTextScale(TextScaleStickEmulationVertical);
        position.y += XSizeButtonStickEmuPos;
        s2d::DrawText(position.Get(), "\uE148");
        s2d::SetTextScale(TextScaleButton);
        position.y -= XSizeButtonStickEmuPos;
        s2d::DrawText(position.Get(), "\uE0C2");
        position.x += XSizeButtonUnit;
    }

    if (state.Test<nn::hid::NpadPalmaButton::Palma>())
    {
        s2d::DrawText(position.Get(), "○");
        s2d::DrawText(position.Get(), "Ｐ");
        position.x += XSizeButtonUnit;
    }

    if (unsupportedButtonPress)
    {
        s2d::DrawText(position.Get(), "\uE140");
        position.x += XSizeButtonUnit;
    }

    if (handheldButtonPress)
    {
        s2d::DrawText(position.Get(), "\uE121");
        position.x += XSizeButtonUnit;
    }

    s2d::SetTextColor(ColorSceneTextDefault);
    s2d::SetTextScale(TextScaleDefault);
    position.x += XSizePadding;
} // NOLINT(impl/function_size)

void DrawNpadAttributes(RecordablePosition& position, const nn::hid::NpadAttributesSet& state) NN_NOEXCEPT
{
    position.x += PrintAndStep(position.Get(), "Connection:", true);
    s2d::SetTextColor(ColorSceneTextStrong02);
    position.x += SelectablePrintAndStep(position.Get(), "\uE12C", false,
        (state.Test<nn::hid::NpadJoyAttribute::IsConnected>()));
    position.x += SelectablePrintAndStep(position.Get(), "\uE125", false,
        (state.Test<nn::hid::NpadJoyAttribute::IsLeftConnected>()));
    position.x += SelectablePrintAndStep(position.Get(), "\uE126", true,
        (state.Test<nn::hid::NpadJoyAttribute::IsRightConnected>()));
    s2d::SetTextColor(ColorSceneTextDefault);

    position.x += PrintAndStep(position.Get(), "Cable:");
    s2d::SetTextColor(ColorSceneTextStrong02);
    position.x += SelectablePrintAndStep(position.Get(), "\uE12C", false,
        (state.Test<nn::hid::NpadJoyAttribute::IsWired>()));
    position.x += SelectablePrintAndStep(position.Get(), "\uE125", false,
        (state.Test<nn::hid::NpadJoyAttribute::IsLeftWired>()));
    position.x += SelectablePrintAndStep(position.Get(), "\uE126", true,
        (state.Test<nn::hid::NpadJoyAttribute::IsRightWired>()));
    s2d::SetTextColor(ColorSceneTextDefault);
}

void DrawAnalogStickState(RecordablePosition& position, const nn::hid::AnalogStickState& state) NN_NOEXCEPT
{
    s2d::DrawText(position.Get(), "%05d/%05d ", state.x, state.y);
    position.x += XSizeStick;
}

void DrawStyleNameFull(RecordablePosition& position, const NpadStyleDefinition& style) NN_NOEXCEPT
{
    s2d::SetTextColor(ColorSceneTextStrong02);
    s2d::DrawText(position.Get(), "%s", style.styleString.c_str());
    s2d::SetTextColor(ColorSceneTextDefault);
    position.x += XSizeStyle;
}

void DrawJoyAssignment(RecordablePosition& position, nn::hid::NpadJoyAssignmentMode mode) NN_NOEXCEPT
{
    position.x += PrintAndStep(position.Get(), "Assignment:", true);
    s2d::SetTextColor(ColorSceneTextStrong01);
    s2d::DrawText(position.Get(), "%s", (mode == nn::hid::NpadJoyAssignmentMode_Single) ? "\uE127" : "\uE122");
    s2d::SetTextColor(ColorSceneTextDefault);
    position.x += XSizeAssignment;
}

void DrawBattery(RecordablePosition& position, nn::hid::system::NpadPowerInfo power) NN_NOEXCEPT
{
    position.x += PrintAndStep(position.Get(), "Battery:", true);
    s2d::SetTextScale(TextScaleBattery);
    for (int i = 0; i < 4; ++i)
    {
        s2d::SetTextColor((i < power.batteryLevel) ? ColorSceneBatteryOn : ColorSceneBatteryOff);
        s2d::DrawText(position.Get(), "■");
        position.x += XSizeBatteryUnit;
    }
    position.x += XSizePadding;
    s2d::SetTextScale(TextScaleDefault);
    s2d::SetTextColor(ColorSceneTextDefault);
    position.x += XSizeBatteryUnit;
    position.x += PrintAndStep(position.Get(), "Power:", true);
    DrawBooleanString(position, power.isPowered);
    position.x += XSizePadding;
    position.x += PrintAndStep(position.Get(), "Charge:", true);
    DrawBooleanString(position, power.isCharging);
    position.x += XSizePadding;
}

void DrawBooleanString(RecordablePosition& position, bool state) NN_NOEXCEPT
{
    if (state)
    {
        s2d::SetTextColor(ColorSceneTextStrong01);
        s2d::DrawText(position.Get(), "\uE14B");
        s2d::SetTextColor(ColorSceneTextDefault);
    }
    else
    {
        s2d::DrawText(position.Get(), "\uE14C");
    }
    position.x += XSizeChar;
}

void DrawUiInfo(RecordablePosition& position, NpadControllerUnit& controller) NN_NOEXCEPT
{
    position.x += PrintAndStep(position.Get(), "UI:", true);
    s2d::SetTextColor(ColorSceneTextStrong01);
    if (controller.IsAbxyOriented())
    {
        s2d::DrawText(position.Get(), "\uE0E0");
    }
    else
    {
        s2d::DrawText(position.Get(), "\uE0AB");
    }
    position.x += XSizeChar;
    if (controller.IsSlSrOriented())
    {
        s2d::DrawText(position.Get(), "\uE0E8\uE0E9");
    }
    else
    {
        s2d::DrawText(position.Get(), "\uE0E4\uE0E5");
    }
    position.x += XSizeChar * 2;
    if (controller.IsPlusAvailable())
    {
        s2d::DrawText(position.Get(), "\uE0EF");
    }
    position.x += XSizeChar;
    if (controller.IsMinusAvailable())
    {
        s2d::DrawText(position.Get(), "\uE0F0");
    }
    position.x += XSizeChar;
    s2d::SetTextColor(ColorSceneTextDefault);
    position.x += XSizePadding;
}

void DrawNfcState(RecordablePosition& position, bool isAvailable, bool isAvtivated) NN_NOEXCEPT
{
    position.x += PrintAndStep(position.Get(), "Nfc:", true);
    if (isAvailable)
    {
        if (isAvtivated)
        {
            s2d::SetTextColor(ColorSceneTextStrong01);
            s2d::DrawText(position.Get(), "\uE14B");
            s2d::SetTextColor(ColorSceneTextDefault);
        }
        else
        {
            s2d::DrawText(position.Get(), "\uE14C");
        }
    }
    else
    {
        s2d::DrawText(position.Get(), "―");
    }
    position.x += XSizeChar;
    position.x += XSizePadding;
}

void DrawIrSensorState(RecordablePosition& position, nn::hid::system::IrSensorState state) NN_NOEXCEPT
{
    position.x += PrintAndStep(position.Get(), "Ir:", true);
    switch (state)
    {
    case nn::hid::system::IrSensorState_NotConnected:
        s2d::DrawText(position.Get(), "―");
        break;

    case nn::hid::system::IrSensorState_NotSupported:
        s2d::DrawText(position.Get(), "\uE152");
        break;

    case nn::hid::system::IrSensorState_Deactivated:
        s2d::DrawText(position.Get(), "\uE14C");
        break;

    case nn::hid::system::IrSensorState_Activated:
        s2d::SetTextColor(ColorSceneTextStrong01);
        s2d::DrawText(position.Get(), "\uE14B");
        s2d::SetTextColor(ColorSceneTextDefault);
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }
    position.x += XSizeChar;
    position.x += XSizePadding;
}

void DrawUsbConnected(RecordablePosition& position, bool state) NN_NOEXCEPT
{
    position.x += PrintAndStep(position.Get(), "Usb:", true);
    DrawBooleanString(position, state);
    position.x += XSizeChar;
    position.x += XSizePadding;
}

void DrawSystemExtInfo(RecordablePosition& position, NpadControllerUnit& controller) NN_NOEXCEPT
{
    nn::hid::system::NpadSystemExtMainStyle mainStyle;
    nn::hid::system::NpadSystemExtSubStyle subStyle;

    nn::hid::system::GetNpadSystemExtStyle(&mainStyle, &subStyle, controller.GetId());
    position.x += PrintAndStep(position.Get(), "SystemStyle:", true);
    s2d::SetTextColor(ColorSceneTextStrong01);

    switch (mainStyle)
    {
    case nn::hid::system::NpadSystemExtMainStyle_None:
        s2d::DrawText(position.Get(), "―");
        break;

    case nn::hid::system::NpadSystemExtMainStyle_HandheldNoJoyCon:
        s2d::DrawText(position.Get(), "Handheld(-)");
        break;

    case nn::hid::system::NpadSystemExtMainStyle_HandheldLeftJoyOnly:
        s2d::DrawText(position.Get(), "Handheld(L)");
        break;

    case nn::hid::system::NpadSystemExtMainStyle_HandheldRightJoyOnly:
        s2d::DrawText(position.Get(), "Handheld(R)");
        break;

    case nn::hid::system::NpadSystemExtMainStyle_HandheldJoyCon:
        s2d::DrawText(position.Get(), "Handheld(LR)");
        break;

    case nn::hid::system::NpadSystemExtMainStyle_JoyDual:
        s2d::DrawText(position.Get(), "JoyDual(LR)");
        break;

    case nn::hid::system::NpadSystemExtMainStyle_JoyDualLeftOnly:
        s2d::DrawText(position.Get(), "JoyDual(L)");
        break;

    case nn::hid::system::NpadSystemExtMainStyle_JoyDualRightOnly:
        s2d::DrawText(position.Get(), "JoyDual(R)");
        break;

    case nn::hid::system::NpadSystemExtMainStyle_JoyLeftHorizontal:
        s2d::DrawText(position.Get(), "JoyLeft(H)");
        break;

    case nn::hid::system::NpadSystemExtMainStyle_JoyLeftVertical:
        s2d::DrawText(position.Get(), "JoyLeft(V)");
        break;

    case nn::hid::system::NpadSystemExtMainStyle_JoyRightHorizontal:
        s2d::DrawText(position.Get(), "JoyRight(H)");
        break;

    case nn::hid::system::NpadSystemExtMainStyle_JoyRightVertical:
        s2d::DrawText(position.Get(), "JoyRight(V)");
        break;

    case nn::hid::system::NpadSystemExtMainStyle_SwitchProController:
        s2d::DrawText(position.Get(), "Pro Controller");
        break;

    case nn::hid::system::NpadSystemExtMainStyle_CompatibleProController:
        s2d::DrawText(position.Get(), "Compat Pro");
        break;

    case nn::hid::system::NpadSystemExtMainStyle_CompatibleJoyCon:
        s2d::DrawText(position.Get(), "Compat JoyCon");
        break;

    case nn::hid::system::NpadSystemExtMainStyle_Unknown:
        s2d::DrawText(position.Get(), "Future");
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }
    s2d::SetTextColor(ColorSceneTextDefault);
    position.x += XSizeSystemExtStyle;
}

void DrawNpadInfo(RecordablePosition& position, NpadControllerUnit& controller) NN_NOEXCEPT
{
    // 描画用のカウント
    static int drawCount = 0;

    DrawIndicator(position, nn::hid::GetPlayerLedPattern(controller.GetId()));

    auto deviceType = controller.GetDeviceType();
    DrawJoyAssignment(position, controller.GetJoyAssignmentMode());
    DrawUiInfo(position, controller);

    // TODO : 常時更新をやめる
    nn::hid::NpadIdType npadIdsForNfc[NpadIdCountMax];
    auto nfcCount = nn::hid::system::GetNpadsWithNfc(npadIdsForNfc, NpadIdCountMax);
    bool isNfcAvailable = false;
    for (int i = 0; i < nfcCount; i++)
    {
        if (npadIdsForNfc[i] == controller.GetId())
        {
            isNfcAvailable = true;
            break;
        }
    }

    DrawNfcState(position, isNfcAvailable, controller.IsNfcActivated());
    DrawIrSensorState(position, controller.GetIrSensorState());
    DrawUsbConnected(position, controller.IsUsbConnected());
    DrawSystemExtInfo(position, controller);
    position.NewLine();

    if (deviceType.CountPopulation() > 0)
    {
        position.x = PositionScene.x + XSizeId;
        s2d::DrawText(position.Get(), "Devices");
    }

    for (int i = 0; i < deviceType.GetCount(); i++)
    {
        if (deviceType[i] == true)
        {
            position.x = PositionScene.x + XSizeId + XSizeNpadLabel;
            auto deviceTypeToDraw = nn::hid::system::DeviceTypeSet();
            deviceTypeToDraw.Reset();
            deviceTypeToDraw.Set(i);
            DrawDeviceType(position, deviceTypeToDraw);

            DeviceIndex deviceIndex = DeviceIndex_Standard;

            if (deviceTypeToDraw.Test<nn::hid::system::NpadDeviceType::HandheldJoyLeft>() ||
                deviceTypeToDraw.Test<nn::hid::system::NpadDeviceType::JoyConLeft>())
            {
                deviceIndex = DeviceIndex_Left;
            }
            if (deviceTypeToDraw.Test<nn::hid::system::NpadDeviceType::HandheldJoyRight>() ||
                deviceTypeToDraw.Test<nn::hid::system::NpadDeviceType::JoyConRight>())
            {
                deviceIndex = DeviceIndex_Right;
            }
            DrawBattery(position, controller.GetPowerInfo(deviceIndex));
            nn::hid::NpadControllerColor color;
            int index = (drawCount / 1500) % 2;
            if (controller.GetDeviceType().Test<nn::hid::system::DeviceType::FullKeyController>() == false)
            {
                index = 0;
            }

            if (index == 1)
            {
                nn::util::Color4u8Type leftGrip;
                nn::util::Color4u8Type rightGrip;
                nn::hid::system::GetNpadFullKeyGripColor(&leftGrip, &rightGrip, controller.GetId());
                s2d::DrawText(position.Get(), "Left:");
                position.x += XSizeColorLabel;
                DrawColor(position, leftGrip);
                s2d::DrawText(position.Get(), "Right: ");
                position.x += XSizeColorLabel;
                DrawColor(position, rightGrip);
            }
            else if (controller.GetControllerColor(&color, deviceIndex))
            {
                s2d::DrawText(position.Get(), "Main:");
                position.x += XSizeColorLabel;
                DrawColor(position, color.main);
                s2d::DrawText(position.Get(), "Sub: ");
                position.x += XSizeColorLabel;
                DrawColor(position, color.sub);
            }
            position.NewLine();
        }
    }

    auto currentStyle = controller.GetStyle();
    if (currentStyle.CountPopulation() > 0)
    {
        position.x = PositionScene.x + XSizeId;
        s2d::DrawText(position.Get(), "Styles");
    }

    for (auto& style : NpadStyles)
    {
        if ((style.style & currentStyle).IsAnyOn())
        {
            position.x = PositionScene.x + XSizeId + XSizeNpadLabel;
            DrawStyleNameFull(position, style);
            s2d::DrawText(position.Get(), "[%08lld]", controller.GetStyleHandler(style.style).GetAbstractedNpadState().samplingNumber);
            position.x += XSizeSamplingNumber;
            DrawNpadAttributes(position, controller.GetStyleHandler(style.style).GetAbstractedNpadState().attributes);
            position.NewLine();
            position.x = PositionScene.x + XSizeId + XSizeNpadLabel + XSizeStyle;
            position.x += PrintAndStep(position.Get(), "\uE101", true);
            DrawAnalogStickState(position, controller.GetStyleHandler(style.style).GetAbstractedNpadState().analogStickL);
            position.x += PrintAndStep(position.Get(), "\uE102", true);
            DrawAnalogStickState(position, controller.GetStyleHandler(style.style).GetAbstractedNpadState().analogStickR);

            bool unsupportedButtonPress = (style.style.Test<nn::hid::system::NpadStyleSystemExt>() && controller.IsUnsupportedButtonPressed());
            bool handheldButtonPress = (controller.IsHandheldButtonPressedOnConsoleMode());
            DrawNpadButtonState(position, controller.GetStyleHandler(style.style).GetAbstractedNpadState().buttons, unsupportedButtonPress, handheldButtonPress);
            position.NewLine();
        }
    }

    ++drawCount;
} // NOLINT(impl/function_size)

SceneControllerStatus::SceneControllerStatus() NN_NOEXCEPT :
    m_pCoutrollers(nullptr),
    m_IsMenuLocked(false)
{
    // 何もしない
}

void SceneControllerStatus::SetNpadControllerList(NpadControllerList* pControllers) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pControllers);
    m_pCoutrollers = pControllers;
}

void SceneControllerStatus::DrawImpl(RecordablePosition& position) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pCoutrollers);
    for (auto& controller : *m_pCoutrollers)
    {
        position.x = PositionSceneText.x;

        DrawNpadInfo(position, controller);

        position.x = PositionSceneText.x;
        s2d::Point2D startPoint = position.Get();
        startPoint.x += XSizeControllerBorderIndent;
        startPoint.y += YSizeLine / 2;
        s2d::Point2D endPoint = position.Get();
        endPoint.x += XSizeControllerBorderIndent + SizeControllerBorder;
        endPoint.y = startPoint.y;
        s2d::DrawLine(startPoint, endPoint, ColorSceneBorder, WidthBorderDefault);
        position.NewLine();
    }
}

PackedButton SceneControllerStatus::Control(const PackedButton& button) NN_NOEXCEPT
{
    const nn::hid::NpadButtonSet LockButtons = nn::hid::NpadButton::Plus::Mask | nn::hid::NpadButton::Minus::Mask;

    if ((button.longTrigger & LockButtons).IsAnyOn())
    {
        m_IsMenuLocked = !m_IsMenuLocked;
    }
    if (m_IsMenuLocked)
    {
        nn::hid::NpadButtonSet value;
        value.Reset();
        return button & value;
    }

    return button & ~LockButtons;
}
