﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <complex>

#include "SceneAnalogStick.h"
#include "ScreenStatics.h"
#include "ScreenUtil.h"
#include "s2d/Simple2D.h"

namespace {

const VectorChartUiParameterConfigList VectorChartUiParameterConfigListStick =
{
    {
        {
            {200.0f, 600.0f, 20.0f, 1.1f},  // sampleCount
            {nn::hid::AnalogStickMax, nn::hid::AnalogStickMax, 2.0f, 1.1f}, // scale
        }
    }
};

const LineChartUiParameterConfigList LineChartUiParameterConfigListStick =
{
    {
        {
            {200.0f, 600.0f, 20.0f, 1.1f},  // xSampleCount
            {nn::hid::AnalogStickMax, nn::hid::AnalogStickMax, 2.0f, 1.1f}, // yScale
            {0.0f, nn::hid::AnalogStickMax, -nn::hid::AnalogStickMax, 0.0f}   // yShift
        }
    }
};

template <typename InputT>
inline float GetRatio(const InputT& input) NN_NOEXCEPT
{
    return static_cast<float>(input) / nn::hid::AnalogStickMax;
}

class ILineChartAccessorStick : public ILineChartDataSetAccessor
{
protected:
    AnalogStickStateExFifo* m_pFifo;
public:
    virtual int size() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFifo->size();
    }
    ILineChartAccessorStick(const std::string label, const s2d::Color& color) NN_NOEXCEPT
    {
        m_Label = label;
        m_Color = color;
    }
    void SetDetailedAnalogStickStateFifo(AnalogStickStateExFifo* pFifo) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pFifo);
        m_pFifo = pFifo;
    }
};

class LineChartAccessorStickX final : public ILineChartAccessorStick
{
public:
    LineChartAccessorStickX(const std::string label, const s2d::Color& color) NN_NOEXCEPT : ILineChartAccessorStick(label, color) {};
    virtual float at(int index) NN_NOEXCEPT NN_OVERRIDE
    {
        return (m_pFifo->at(index).stick.x);
    }
    virtual const char* backStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%d", m_pFifo->back().stick.x);
        return buf;
    }
    virtual const char* maxStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%d", m_pFifo->GetMax().stick.x);
        return buf;
    }
    virtual const char* minStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%d", m_pFifo->GetMin().stick.x);
        return buf;
    }
};


class LineChartAccessorStickY final : public ILineChartAccessorStick
{
public:
    LineChartAccessorStickY(const std::string label, const s2d::Color& color) NN_NOEXCEPT : ILineChartAccessorStick(label, color) {};
    virtual float at(int index) NN_NOEXCEPT NN_OVERRIDE
    {
        return (m_pFifo->at(index).stick.y);
    }
    virtual const char* backStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%d", m_pFifo->back().stick.y);
        return buf;
    }
    virtual const char* maxStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%d", m_pFifo->GetMax().stick.y);
        return buf;
    }
    virtual const char* minStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%d", m_pFifo->GetMin().stick.y);
        return buf;
    }
};

class LineChartAccessorStickLength final : public ILineChartAccessorStick
{
public:
    LineChartAccessorStickLength(const std::string label, const s2d::Color& color) NN_NOEXCEPT : ILineChartAccessorStick(label, color) {};
    virtual float at(int index) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFifo->at(index).length;
    }
    virtual const char* backStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%3.2f%%", GetRatio(m_pFifo->back().length) * 100);
        return buf;
    }
    virtual const char* maxStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%3.2f%%", GetRatio(m_pFifo->GetMax().length) * 100);
        return buf;
    }
    virtual const char* minStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%3.2f%%", GetRatio(m_pFifo->GetMin().length) * 100);
        return buf;
    }
};

class VectorChartAccessorStick : public IVectorChartDataSetAccessor
{
private:
    AnalogStickStateExFifo* m_pFifo;
public:
    VectorChartAccessorStick(const std::string label, const s2d::Color& color) NN_NOEXCEPT
    {
        m_Label = label;
        m_Color = color;
    }
    virtual int size() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFifo->size();
    }
    virtual VectorType back() NN_NOEXCEPT NN_OVERRIDE
    {
        VectorType output;
        output.x = m_pFifo->back().stick.x;
        output.y = m_pFifo->back().stick.y;
        output.length = m_pFifo->back().length;
        return output;
    }
    virtual VectorType at(int index) NN_NOEXCEPT NN_OVERRIDE
    {
        VectorType output;
        output.x = m_pFifo->at(index).stick.x;
        output.y = m_pFifo->at(index).stick.y;
        output.length = m_pFifo->at(index).length;
        return output;
    }
    virtual const char* xStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%d", m_pFifo->back().stick.x);
        return buf;
    }

    virtual const char* yStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%d", m_pFifo->back().stick.y);
        return buf;
    }

    virtual const char* dirStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%3.2f", m_pFifo->back().angle);
        return buf;
    }

    virtual const char* lenStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%3.2f%%", GetRatio(m_pFifo->back().length) * 100);
        return buf;
    }

    virtual const char* lenMaxStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%3.2f%%", GetRatio(m_pFifo->GetMax().length) * 100);
        return buf;
    }

    virtual const char* lenMinStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%3.2f%%", GetRatio(m_pFifo->GetMin().length) * 100);
        return buf;
    }

    void SetDetailedAnalogStickStateFifo(AnalogStickStateExFifo* pFifo) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pFifo);
        m_pFifo = pFifo;
    }
};

} // namespace

SceneAnalogStick::SceneAnalogStick() NN_NOEXCEPT :
    m_LineChart(LineChartUiParameterConfigListStick, false),
    m_VectorChart(VectorChartUiParameterConfigListStick, false),
    m_IsMenuLocked(false)
{
}

void SceneAnalogStick::OnSceneFocused() NN_NOEXCEPT
{
    m_pControllers->ClearSelectedController();
}

void SceneAnalogStick::SetNpadControllerList(NpadControllerList* pControllers) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pControllers);
    m_pControllers = pControllers;
}

void SceneAnalogStick::DrawImpl(RecordablePosition& position) NN_NOEXCEPT
{
    float xOrigin = position.x;

    LineChartAccessorStickX      xAccessor      = { "X", ColorSceneTextStrong01 };
    LineChartAccessorStickY      yAccessor      = { "Y", ColorSceneTextStrong04 };
    LineChartAccessorStickLength lengthAccessor = { "Length", ColorSceneTextStrong03 };
    ILineChartAccessorStick* pLineChartAccessors[] =
    {
      &xAccessor,
      &yAccessor,
      &lengthAccessor
    };
    VectorChartAccessorStick     vectorAccessor = { "Stick", ColorSceneTextStrong03 };
    IVectorChartDataSetAccessor* pVectorAccessors[] =
    {
        &vectorAccessor
    };
    bool isDeviceAvailable = false;

    NpadStyleHandler* pStyle;
    DeviceIndex deviceIndex;
    if (m_pControllers->GetSelectedStyleHandler(&pStyle) == true &&
        m_pControllers->GetSelectedDeviceIndex(&deviceIndex) == true)
    {
        auto pLifo = pStyle->GetAnalogStickFifo(deviceIndex);

        vectorAccessor.SetDetailedAnalogStickStateFifo(pLifo);

        for (auto& pAccessor : pLineChartAccessors)
        {
            pAccessor->SetDetailedAnalogStickStateFifo(pLifo);
        }
        isDeviceAvailable = true;
    }

    // ベクトルの描画
    position.x += (SceneSize.width - SizeAnalogStickVectorChart) / 2;
    m_VectorChart.Draw(position.Get(), SizeAnalogStickVectorChart, "x", "y", pVectorAccessors, (isDeviceAvailable) ? NN_ARRAY_SIZE(pVectorAccessors) : 0);

    // グラフの描画
    position.x = xOrigin;
    position.y += SizeAnalogStickVectorChart + YSizeLine;
    const s2d::Size size = { {XSizeLineChartFull, YSizeAnalogStickLineChart} };
    m_LineChart.Draw(position.Get(), size, false, reinterpret_cast<ILineChartDataSetAccessor**>(pLineChartAccessors), (isDeviceAvailable) ? NN_ARRAY_SIZE(pLineChartAccessors) : 0);

}

PackedButton SceneAnalogStick::Control(const PackedButton& button) NN_NOEXCEPT
{
    const nn::hid::NpadButtonSet LockButtons = nn::hid::NpadButton::Plus::Mask | nn::hid::NpadButton::Minus::Mask;

    if ((button.longTrigger & LockButtons).IsAnyOn())
    {
        m_IsMenuLocked = !m_IsMenuLocked;
    }
    if (m_IsMenuLocked)
    {
        auto triggerWithRepeat = button.trigger | button.repeat;
        if (triggerWithRepeat.Test<nn::hid::NpadButton::Up>())
        {
            m_LineChart.Decrement(LineChart::UiParameterType_YScale);
            m_VectorChart.Decrement(VectorChart::UiParameterType_Scale);
        }
        if (triggerWithRepeat.Test<nn::hid::NpadButton::Down>())
        {
            m_LineChart.Increment(LineChart::UiParameterType_YScale);
            m_VectorChart.Increment(VectorChart::UiParameterType_Scale);
        }
        if (triggerWithRepeat.Test<nn::hid::NpadButton::Left>())
        {
            m_LineChart.Decrement(LineChart::UiParameterType_XSampleCount);
            m_VectorChart.Decrement(VectorChart::UiParameterType_SampleCount);
        }
        if (triggerWithRepeat.Test<nn::hid::NpadButton::Right>())
        {
            m_LineChart.Increment(LineChart::UiParameterType_XSampleCount);
            m_VectorChart.Increment(VectorChart::UiParameterType_SampleCount);
        }
        if (triggerWithRepeat.Test<nn::hid::NpadButton::X>())
        {
            m_LineChart.Increment(LineChart::UiParameterType_YShift);
        }
        if (triggerWithRepeat.Test<nn::hid::NpadButton::B>())
        {
            m_LineChart.Decrement(LineChart::UiParameterType_YShift);
        }
        if (triggerWithRepeat.Test<nn::hid::NpadButton::L>() ||
            triggerWithRepeat.Test<nn::hid::NpadButton::R>())
        {
            m_pControllers->ResetSelectedControllerFifo();
        }

        nn::hid::NpadButtonSet value;
        value.Reset();
        return button & value;
    }

    return button & ~LockButtons;
}
