﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <array>
#include <cmath>
#include <nn/nn_Assert.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_NpadColor.h>
#include <nn/hid/system/hid_Irsensor.h>
#include <nn/hid/system/hid_Joy.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/hid/hid_VibrationDeviceApi.h>
#include <nn/util/util_Constant.h>

#include "AnalogStickUtil.h"
#include "ControllerCommonTypes.h"
#include "NpadIdDefine.h"
#include "NpadStyleDefinition.h"
#include "PackedButton.h"
#include "PalmaHandler.h"
#include "RingBuffer.h"
#include "SixAxisSensorUtil.h"

class NpadStyleHandler
{
    NN_DISALLOW_COPY(NpadStyleHandler);
    NN_DISALLOW_MOVE(NpadStyleHandler);

private:
    static const int HandleCountMax = 2;

    NpadStyleDefinition m_Style;
    nn::hid::NpadIdType m_Id;

    SixAxisSensorStateFifo m_SixAxisSensorFifo[HandleCountMax];

    int m_VibrationDeviceHandleCount;
    nn::hid::VibrationDeviceHandle m_VibrationHandles[HandleCountMax];

    PackedButton m_PackedButtons;
    AbstractedNpadState m_CurrentState;

    AnalogStickStateExFifo m_LeftAnalogStickFifo;
    AnalogStickStateExFifo m_RightAnalogStickFifo;

    int64_t m_LastSamplingNumber;

public:
    NpadStyleHandler() { /* 何もしない */ };
    ~NpadStyleHandler() { /* 何もしない */ };

    void Initialize(const NpadStyleDefinition& style, const nn::hid::NpadIdType& id) NN_NOEXCEPT;
    void InitializeSixAxisSensor() NN_NOEXCEPT;
    void InitializeVibrationDevice() NN_NOEXCEPT;
    NpadStyleDefinition GetNpadStyleDefinition() NN_NOEXCEPT;
    void UpdateState() NN_NOEXCEPT;
    void UpdateSixAxisSensorState() NN_NOEXCEPT;
    void UpdateVibration() NN_NOEXCEPT;
    void StopVibration() NN_NOEXCEPT;
    PackedButton GetButtons() NN_NOEXCEPT;
    AbstractedNpadState GetAbstractedNpadState() NN_NOEXCEPT;
    AnalogStickStateExFifo* GetAnalogStickFifo(DeviceIndex index) NN_NOEXCEPT;
    SixAxisSensorStateFifo* GetSixAxisSensorFifo(DeviceIndex index) NN_NOEXCEPT;
    void ResetFifo() NN_NOEXCEPT;
};

class NpadControllerUnit
{
    NN_DISALLOW_COPY(NpadControllerUnit);
    NN_DISALLOW_MOVE(NpadControllerUnit);

private:
    nn::hid::NpadIdType m_Id;
    nn::hid::NpadStyleSet m_CurrentStyle;
    NpadStyleHandler m_StyleHandlers[NpadStyleCountMax];
    PalmaHandler m_PalmaHandler;

public:
    NpadControllerUnit() NN_NOEXCEPT;
    void Initialize(nn::hid::NpadIdType id) NN_NOEXCEPT;
    void Update() NN_NOEXCEPT;
    nn::hid::NpadIdType GetId() NN_NOEXCEPT;
    nn::hid::NpadStyleSet& GetStyle() NN_NOEXCEPT;
    NpadStyleHandler& GetStyleHandler(const nn::hid::NpadStyleSet& style) NN_NOEXCEPT;
    NpadStyleHandler& GetSystemHolder() NN_NOEXCEPT;
    nn::hid::system::NpadDeviceTypeSet GetDeviceType() NN_NOEXCEPT;
    bool GetControllerColor(nn::hid::NpadControllerColor* pOutColor, DeviceIndex index) NN_NOEXCEPT;
    nn::hid::NpadJoyAssignmentMode GetJoyAssignmentMode() NN_NOEXCEPT;
    nn::hid::system::NpadPowerInfo GetPowerInfo(DeviceIndex index) NN_NOEXCEPT;
    bool IsAbxyOriented() NN_NOEXCEPT;
    bool IsSlSrOriented() NN_NOEXCEPT;
    bool IsPlusAvailable() NN_NOEXCEPT;
    bool IsMinusAvailable() NN_NOEXCEPT;
    bool IsUnsupportedButtonPressed() NN_NOEXCEPT;
    bool IsHandheldButtonPressedOnConsoleMode() NN_NOEXCEPT;
    bool IsNfcActivated() NN_NOEXCEPT;
    nn::hid::system::IrSensorState GetIrSensorState() NN_NOEXCEPT;
    bool IsUsbConnected() NN_NOEXCEPT;
};

class NpadControllerList
{
private:
    NpadControllerUnit* m_pSelectedController;
    int m_StyleIndex;
    DeviceIndex m_PositionIndex;
    bool m_Selected;

    typedef std::array<NpadControllerUnit, NpadIdCountMax> Array;
    Array m_List;

public:
    void Initialize() NN_NOEXCEPT;
    void Update() NN_NOEXCEPT;
    PackedButton GetControlButtons() NN_NOEXCEPT;
    bool GetSelectedController(NpadControllerUnit** pOutValue) NN_NOEXCEPT;
    bool GetSelectedStyleHandler(NpadStyleHandler** pOutValue) NN_NOEXCEPT;
    bool GetSelectedDeviceIndex(DeviceIndex* pOutValue) NN_NOEXCEPT;
    void ClearSelectedController() NN_NOEXCEPT;
    void ResetSelectedControllerFifo() NN_NOEXCEPT;

    Array::iterator begin();
    Array::iterator end();

private:
    void UpdateSelectedDevice(NpadControllerUnit* pController,
                              int styleIndex,
                              DeviceIndex positionIndex) NN_NOEXCEPT;
};
