﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_NpadColor.h>
#include <nn/hid/hid_NpadGc.h>
#include <nn/hid/hid_UsbFullKey.h>
#include <nn/hid/system/hid_Irsensor.h>
#include <nn/hid/system/hid_Joy.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/hid/system/hid_Nfc.h>
#include <nn/hid/system/hid_RegisteredDevice.h>
#include <nn/hid/hid_NpadSixAxisSensor.h>
#include <nn/hid/hid_VibrationDeviceApi.h>
#include <nn/hid/hid_VibrationGcErm.h>
#include <nn/util/util_Constant.h>

#include "NpadController.h"

void NpadStyleHandler::Initialize(const NpadStyleDefinition& style, const nn::hid::NpadIdType& id) NN_NOEXCEPT
{
    m_Style = style;
    m_Id = id;
    m_RightAnalogStickFifo.Initialize(nn::hid::AnalogStickState());
    m_LeftAnalogStickFifo.Initialize(nn::hid::AnalogStickState());
}

void NpadStyleHandler::InitializeSixAxisSensor() NN_NOEXCEPT
{
    nn::hid::SixAxisSensorHandle handles[HandleCountMax];
    auto count = nn::hid::GetSixAxisSensorHandles(handles, NN_ARRAY_SIZE(handles), m_Id, m_Style.style);
    for (int i = 0; i < NN_ARRAY_SIZE(m_SixAxisSensorFifo); i++)
    {
        if (i < count)
        {
            auto index = DeviceIndex_Standard;
            if (count > 1)
            {
                index = (i == 0) ? DeviceIndex_Left : DeviceIndex_Right;
            }
            m_SixAxisSensorFifo[i].SetHandle(handles[i], index);
        }
        else
        {
            m_SixAxisSensorFifo[i].ClearHandle();
        }
    }
}

void NpadStyleHandler::InitializeVibrationDevice() NN_NOEXCEPT
{
    m_VibrationDeviceHandleCount = nn::hid::GetVibrationDeviceHandles(m_VibrationHandles, HandleCountMax, m_Id, m_Style.style);
    for (int i = 0; i < m_VibrationDeviceHandleCount; i++)
    {
        nn::hid::InitializeVibrationDevice(m_VibrationHandles[i]);
    }
}

NpadStyleDefinition NpadStyleHandler::GetNpadStyleDefinition() NN_NOEXCEPT
{
    return m_Style;
}

void NpadStyleHandler::UpdateState() NN_NOEXCEPT
{
    AbstractedNpadState states[nn::hid::NpadStateCountMax];

    // 最新のNpadのステートを取得
    auto count = m_Style.pFunc(states, NN_ARRAY_SIZE(states), m_Id);
    m_CurrentState = states[0];
    m_PackedButtons.Update(m_CurrentState.buttons);

    // AnalogStick の FIFO に新しいデータを追加
    for (int i = count - 1; i >= 0; --i)
    {
        if (states[i].samplingNumber > m_LastSamplingNumber)
        {
            m_LeftAnalogStickFifo.AddSample(states[i].analogStickL);
            m_RightAnalogStickFifo.AddSample(states[i].analogStickR);
            m_LastSamplingNumber = states[i].samplingNumber;
        }
    }
}

void NpadStyleHandler::UpdateSixAxisSensorState() NN_NOEXCEPT
{
    for (auto& fifo : m_SixAxisSensorFifo)
    {
        fifo.Update();
    }
}

void NpadStyleHandler::UpdateVibration() NN_NOEXCEPT
{
    static const nn::hid::NpadButtonSet leftControllerButtonSet =
        nn::hid::NpadButton::L::Mask |
        nn::hid::NpadButton::ZL::Mask |
        nn::hid::NpadButton::Minus::Mask |
        nn::hid::NpadButton::StickL::Mask |
        nn::hid::NpadButton::StickLLeft::Mask |
        nn::hid::NpadButton::StickLRight::Mask |
        nn::hid::NpadButton::StickLUp::Mask |
        nn::hid::NpadButton::StickLDown::Mask |
        nn::hid::NpadButton::Left::Mask |
        nn::hid::NpadButton::Right::Mask |
        nn::hid::NpadButton::Up::Mask |
        nn::hid::NpadButton::Down::Mask |
        nn::hid::NpadJoyButton::LeftSL::Mask |
        nn::hid::NpadJoyButton::LeftSR::Mask;

    if (m_Style.style.Test<nn::hid::system::NpadStyleSystem>() || m_Style.style.Test<nn::hid::system::NpadStyleSystemExt>())
    {
        return;
    }

    bool enableHandle[] = { (m_PackedButtons.trigger & leftControllerButtonSet).IsAnyOn(), (m_PackedButtons.trigger & ~leftControllerButtonSet).IsAnyOn() };

    nn::hid::NpadStyleSet dualStyle = (nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleJoyDual::Mask);
    if (nn::hid::GetNpadHandheldActivationMode() == nn::hid::NpadHandheldActivationMode_Single ||
        nn::hid::GetNpadHandheldActivationMode() == nn::hid::NpadHandheldActivationMode_None)
    {
        dualStyle |= nn::hid::NpadStyleHandheld::Mask;
    }

    for (int i = 0; i < m_VibrationDeviceHandleCount; i++)
    {
        if (m_Style.style.Test<nn::hid::NpadStyleGc>())
        {
            auto command = (m_PackedButtons.trigger.IsAnyOn()) ? nn::hid::VibrationGcErmCommand_Start : nn::hid::VibrationGcErmCommand_StopHard;
            nn::hid::SendVibrationGcErmCommand(m_VibrationHandles[i], command);
        }
        else
        {
            auto vibrationValue = nn::hid::VibrationValue::Make();
            if (m_PackedButtons.trigger.IsAnyOn() && ((m_Style.style & dualStyle).IsAllOff() || enableHandle[i]))
            {
                vibrationValue.amplitudeLow = 0.25f * 1.0f;
                vibrationValue.frequencyLow = 160.0f;
                vibrationValue.amplitudeHigh = 0.0f;
                vibrationValue.frequencyHigh = 320.0f;
            }
            nn::hid::SendVibrationValue(m_VibrationHandles[i], vibrationValue);
        }
    }
}

void NpadStyleHandler::StopVibration() NN_NOEXCEPT
{
    for (int i = 0; i < m_VibrationDeviceHandleCount; i++)
    {
        if (m_Style.style.Test<nn::hid::NpadStyleGc>())
        {
            nn::hid::SendVibrationGcErmCommand(m_VibrationHandles[i], nn::hid::VibrationGcErmCommand_Stop);
        }
        else
        {
            auto vibrationValue = nn::hid::VibrationValue::Make();
            nn::hid::SendVibrationValue(m_VibrationHandles[i], vibrationValue);
        }
    }
}

PackedButton NpadStyleHandler::GetButtons() NN_NOEXCEPT
{
    return m_PackedButtons;
}

AbstractedNpadState NpadStyleHandler::GetAbstractedNpadState() NN_NOEXCEPT
{
    return m_CurrentState;
}

AnalogStickStateExFifo* NpadStyleHandler::GetAnalogStickFifo(DeviceIndex index) NN_NOEXCEPT
{
    if (index == DeviceIndex_Left)
    {
        return &m_LeftAnalogStickFifo;
    }
    return &m_RightAnalogStickFifo;
}

SixAxisSensorStateFifo* NpadStyleHandler::GetSixAxisSensorFifo(DeviceIndex index) NN_NOEXCEPT
{
    for (auto& fifo : m_SixAxisSensorFifo)
    {
        if (fifo.GetDeviceIndex() == index)
        {
            return &fifo;
        }
    }
    return &m_SixAxisSensorFifo[0];
}


void NpadStyleHandler::ResetFifo() NN_NOEXCEPT
{
    m_LeftAnalogStickFifo.Clear();
    m_RightAnalogStickFifo.Clear();
    for (auto& fifo : m_SixAxisSensorFifo)
    {
        fifo.Clear();
    }
}

NpadControllerUnit::NpadControllerUnit() NN_NOEXCEPT
{
    m_CurrentStyle.Reset();
}

void NpadControllerUnit::Initialize(nn::hid::NpadIdType id) NN_NOEXCEPT
{
    m_Id = id;
    for (int i = 0; i < NpadStyleCountMax; ++i)
    {
        m_StyleHandlers[i].Initialize(NpadStyles[i], m_Id);
        m_StyleHandlers[i].InitializeSixAxisSensor();
        m_StyleHandlers[i].InitializeVibrationDevice();
    }
    m_PalmaHandler.Initialize(m_Id);
}

void NpadControllerUnit::Update() NN_NOEXCEPT
{
    m_CurrentStyle = nn::hid::GetNpadStyleSet(m_Id);

    for (auto& styleHandler : m_StyleHandlers)
    {
        if ((styleHandler.GetNpadStyleDefinition().style & m_CurrentStyle).IsAnyOn())
        {
            styleHandler.UpdateState();
            styleHandler.UpdateSixAxisSensorState();
            styleHandler.UpdateVibration();
        }
        else if ((styleHandler.GetNpadStyleDefinition().style & (nn::hid::system::NpadStyleSystem::Mask | nn::hid::system::NpadStyleSystemExt::Mask)).IsAnyOn())
        {
            styleHandler.UpdateState();
        }

        if (m_CurrentStyle.IsAllOff())
        {
            styleHandler.StopVibration();
        }
    }

    m_PalmaHandler.Update(m_CurrentStyle.Test<nn::hid::NpadStylePalma>());
}

nn::hid::NpadIdType NpadControllerUnit::GetId() NN_NOEXCEPT
{
    return m_Id;
}

nn::hid::NpadStyleSet& NpadControllerUnit::GetStyle() NN_NOEXCEPT
{
    return m_CurrentStyle;
}

NpadStyleHandler& NpadControllerUnit::GetStyleHandler(const nn::hid::NpadStyleSet& style) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_EQUAL(style.CountPopulation(), 1);

    for (auto& styleHandler : m_StyleHandlers)
    {
        if (styleHandler.GetNpadStyleDefinition().style == style)
        {
            return styleHandler;
        }
    }

    NN_ABORT("Unexpected Style Passed\n");
}

NpadStyleHandler& NpadControllerUnit::GetSystemHolder() NN_NOEXCEPT
{
    return this->GetStyleHandler(nn::hid::system::NpadStyleSystemExt::Mask);
}

nn::hid::system::NpadDeviceTypeSet NpadControllerUnit::GetDeviceType() NN_NOEXCEPT
{
    return nn::hid::system::GetNpadDeviceType(m_Id);
}

bool NpadControllerUnit::GetControllerColor(nn::hid::NpadControllerColor* pOutColor, DeviceIndex index) NN_NOEXCEPT
{
    nn::hid::NpadControllerColor dummy;

    switch (index)
    {
    case DeviceIndex_Standard:
        if (nn::hid::GetNpadControllerColor(pOutColor, m_Id).IsSuccess())
        {
            return true;
        }
        break;
    case DeviceIndex_Left:
        if (nn::hid::GetNpadControllerColor(pOutColor, &dummy, m_Id).IsSuccess())
        {
            return true;
        }
        break;
    case DeviceIndex_Right:
        if (nn::hid::GetNpadControllerColor(&dummy, pOutColor, m_Id).IsSuccess())
        {
            return true;
        }
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }
    return false;
}

nn::hid::NpadJoyAssignmentMode NpadControllerUnit::GetJoyAssignmentMode() NN_NOEXCEPT
{
    return nn::hid::GetNpadJoyAssignment(m_Id);
}

nn::hid::system::NpadPowerInfo NpadControllerUnit::GetPowerInfo(DeviceIndex index) NN_NOEXCEPT
{
    nn::hid::system::NpadPowerInfo left;
    nn::hid::system::NpadPowerInfo right;

    switch (index)
    {
    case DeviceIndex_Standard:
        return nn::hid::system::GetNpadPowerInfo(m_Id);

    case DeviceIndex_Left:
        nn::hid::system::GetNpadPowerInfo(&left, &right, m_Id);
        return left;

    case DeviceIndex_Right:
        nn::hid::system::GetNpadPowerInfo(&left, &right, m_Id);
        return right;

    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

bool NpadControllerUnit::IsAbxyOriented() NN_NOEXCEPT
{
        return nn::hid::system::IsAbxyButtonOriented(m_Id);
    }

bool NpadControllerUnit::IsSlSrOriented() NN_NOEXCEPT
{
    return nn::hid::system::IsSlSrButtonOriented(m_Id);
}

bool NpadControllerUnit::IsPlusAvailable() NN_NOEXCEPT
{
    bool isPlusAvailable = false;
    bool isMinusAvailable = false;
    nn::hid::system::GetPlusMinusButtonCapability(&isPlusAvailable, &isMinusAvailable, m_Id);
    return isPlusAvailable;
}

bool NpadControllerUnit::IsMinusAvailable() NN_NOEXCEPT
{
    bool isPlusAvailable = false;
    bool isMinusAvailable = false;
    nn::hid::system::GetPlusMinusButtonCapability(&isPlusAvailable, &isMinusAvailable, m_Id);
    return isMinusAvailable;
}

bool NpadControllerUnit::IsUnsupportedButtonPressed() NN_NOEXCEPT
{
    return nn::hid::system::IsUnsupportedButtonPressed(m_Id, nn::hid::system::NpadStyleSystemExt::Mask);
}

bool NpadControllerUnit::IsHandheldButtonPressedOnConsoleMode() NN_NOEXCEPT
{
    return nn::hid::system::IsHandheldButtonPressedOnConsoleMode();
}

bool NpadControllerUnit::IsNfcActivated() NN_NOEXCEPT
{
    return nn::hid::system::IsNfcActivated(m_Id);
}

nn::hid::system::IrSensorState NpadControllerUnit::GetIrSensorState() NN_NOEXCEPT
{
    return nn::hid::system::GetIrSensorState(m_Id);
}

bool NpadControllerUnit::IsUsbConnected() NN_NOEXCEPT
{
    return nn::hid::IsUsbFullKeyControllerConnected(m_Id);
}

void NpadControllerList::Initialize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(NpadIdCountMax, m_List.size());

    for (int i = 0; i < NpadIdCountMax; ++i)
    {
        NpadControllerUnit& controller = m_List.at(i);

        //Npadの初期化
        controller.Initialize(NpadIds[i]);
    }

}

PackedButton NpadControllerList::GetControlButtons() NN_NOEXCEPT
{
    PackedButton returnButtons;
    returnButtons.Reset();
    for (auto& controller : m_List)
    {
        returnButtons |= controller.GetSystemHolder().GetButtons();
    }
    return returnButtons;
}

void NpadControllerList::Update() NN_NOEXCEPT
{
    // 各コントローラーの内部状態の更新
    for (auto& controller : m_List)
    {
        controller.Update();
    }

    // 選択されているコントローラーの状態チェック
    if (m_Selected)
    {
        if ((m_pSelectedController->GetStyle() & NpadStyles[m_StyleIndex].style).IsAnyOn() == false)
        {
            // コントローラーが切断された
            m_Selected = false;
        }
    }

    // 選択済みコントローラーの更新
    for (auto& controller : m_List)
    {
        for (int styleIndex = 0; styleIndex < NpadStyleCountMax; ++styleIndex)
        {
            auto& style = NpadStyles[styleIndex];
            if ((style.style & controller.GetStyle()).IsAnyOn())
            {
                auto state = controller.GetStyleHandler(style.style).GetAbstractedNpadState();
                if (state.buttons.Test<nn::hid::NpadButton::StickL>())
                {
                    UpdateSelectedDevice(&controller, styleIndex, DeviceIndex_Left);
                }
                if (state.buttons.Test<nn::hid::NpadButton::StickR>())
                {
                    UpdateSelectedDevice(&controller, styleIndex, DeviceIndex_Right);
                }
            }
        }
    }
}

bool NpadControllerList::GetSelectedController(NpadControllerUnit** pOutValue) NN_NOEXCEPT
{
    if (m_Selected == true)
    {
        *pOutValue = m_pSelectedController;
        return true;
    }

    return false;
}

bool NpadControllerList::GetSelectedStyleHandler(NpadStyleHandler** pOutValue) NN_NOEXCEPT
{
    if (m_Selected == true)
    {
        *pOutValue = &(m_pSelectedController->GetStyleHandler(NpadStyles[m_StyleIndex].style));
        return true;
    }

    return false;
}

bool NpadControllerList::GetSelectedDeviceIndex(DeviceIndex* pOutValue) NN_NOEXCEPT
{
    if (m_Selected == true)
    {
        *pOutValue = m_PositionIndex;
        return true;
    }

    return false;
}

void NpadControllerList::ClearSelectedController() NN_NOEXCEPT
{
    m_Selected = false;
}

void NpadControllerList::ResetSelectedControllerFifo() NN_NOEXCEPT
{
    if (m_Selected == true && m_pSelectedController != nullptr)
    {
        m_pSelectedController->GetStyleHandler(NpadStyles[m_StyleIndex].style).ResetFifo();
    }
}

NpadControllerList::Array::iterator NpadControllerList::begin() NN_NOEXCEPT
{
    return m_List.begin();
}

NpadControllerList::Array::iterator NpadControllerList::end() NN_NOEXCEPT
{
    return m_List.end();
}

void NpadControllerList::UpdateSelectedDevice(NpadControllerUnit* pController,
                                           int styleIndex,
                                           DeviceIndex positionIndex) NN_NOEXCEPT
{
    if (m_pSelectedController != pController ||
        styleIndex != m_StyleIndex ||
        m_PositionIndex != positionIndex ||
        m_Selected == false)
    {
        m_pSelectedController = pController;
        m_StyleIndex = styleIndex;
        m_PositionIndex = positionIndex;
        m_Selected = true;
    }

}
