﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <vector>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/applet/applet_Apis.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/hid/system/hid_Irsensor.h>
#include <nn/hid/system/hid_Nfc.h>

#include "MenuDeviceControl.h"
#include "PalmaHandler.h"

namespace {
    ::nn::applet::AppletResourceUserId GetAppletResourceUserId() NN_NOEXCEPT
    {
#ifdef NN_BUILD_CONFIG_OS_HORIZON
        return ::nn::applet::GetAppletResourceUserId();
#else
        return ::nn::applet::AppletResourceUserId::GetInvalidId();
#endif
    }
} // namespace

// Palma のスキャン制御
class MenuItemPalmaScan final : public MenuItemEnableDisable
{
public:
    MenuItemPalmaScan() NN_NOEXCEPT :
        MenuItemEnableDisable("Scan")
    {
        // 何もしない
    }

protected:
    virtual void Proceed() NN_NOEXCEPT NN_OVERRIDE
    {
        if (this->IsEnableSelected())
        {
            PalmaHandler::StartScan();
        }
        else
        {
            PalmaHandler::StopScan();
        }
    }
};

// Palma のペアリング制御
class MenuItemPalmaPairing final : public MenuItemEnableDisable
{
public:
    MenuItemPalmaPairing() NN_NOEXCEPT :
        MenuItemEnableDisable("Pairing")
    {
        // 何もしない
    }

protected:
    virtual void Proceed() NN_NOEXCEPT NN_OVERRIDE
    {
        if (this->IsEnableSelected())
        {
            PalmaHandler::StartPairing();
        }
        else
        {
            PalmaHandler::StopPairing();
        }
    }
};

// IrSensor 制御 (NpadIdType の選択)
class MenuItemPalma : public MenuItemHolder
{
public:
    MenuItemPalma() NN_NOEXCEPT :
        MenuItemHolder("Palma", true, true)
    {
        AddMenuItem(new MenuItemPalmaScan());
        AddMenuItem(new MenuItemPalmaPairing());
    }
};

// IrSensor 制御の種類
class MenuItemIrSensorControlTypes final : public MenuItemEnableDisable
{
private:
    const nn::hid::NpadIdType NpadId;

public:
    MenuItemIrSensorControlTypes(const std::string& name, const nn::hid::NpadIdType& npadId) NN_NOEXCEPT :
        MenuItemEnableDisable(name),
        NpadId(npadId)
    {
        // 何もしない
    }
protected:
    virtual void Proceed() NN_NOEXCEPT NN_OVERRIDE
    {
        auto result = (this->IsEnableSelected()) ? nn::hid::system::ActivateIrSensor(NpadId, GetAppletResourceUserId()) : nn::hid::system::DeactivateIrSensor(NpadId, GetAppletResourceUserId());
        if (result.IsFailure())
        {
            NN_LOG("IrSensor Activate Failure Result:%d\n", result.GetDescription());
        }
    }
};

// IrSensor 制御 (NpadIdType の選択)
class MenuItemIrSensorControl : public MenuItemNpadIds<MenuItemIrSensorControlTypes>
{
public:
    MenuItemIrSensorControl() NN_NOEXCEPT :
        MenuItemNpadIds<MenuItemIrSensorControlTypes>("IrSensor", true)
    {
        // 何もしない
    }
};

// Nfc 制御の種類
class MenuItemNfcControlTypes final : public MenuItemEnableDisable
{
private:
    const nn::hid::NpadIdType NpadId;

public:
    MenuItemNfcControlTypes(const std::string& name, const nn::hid::NpadIdType& npadId) NN_NOEXCEPT :
        MenuItemEnableDisable(name),
        NpadId(npadId)
    {
        // 何もしない
    }
protected:
    virtual void Proceed() NN_NOEXCEPT NN_OVERRIDE
    {
        auto result = (this->IsEnableSelected()) ? nn::hid::system::ActivateNfc(NpadId, GetAppletResourceUserId()) : nn::hid::system::DeactivateNfc(NpadId, GetAppletResourceUserId());
        if (result.IsFailure())
        {
            NN_LOG("Nfc Activate Failure Result:%d\n", result.GetDescription());
        }
    }
};

// Nfc 制御 (NpadIdType の選択)
class MenuItemNfcControl : public MenuItemNpadIds<MenuItemNfcControlTypes>
{
public:
    MenuItemNfcControl() NN_NOEXCEPT :
        MenuItemNpadIds<MenuItemNfcControlTypes>("Nfc", true)
    {
        // 何もしない
    }
};

// 点滅させるデバイス毎のアイテム
class MenuItemLedDeviceTypeUnit final : public MenuItemBoolean
{
private:
    const nn::hid::system::NpadDeviceTypeSet DeviceType;
    static const bool DefaultSettings = true;

public:
    MenuItemLedDeviceTypeUnit(const std::string& name, const nn::hid::system::NpadDeviceTypeSet& deviceType) NN_NOEXCEPT :
        MenuItemBoolean(name),
        DeviceType(deviceType)
    {
        this->UpdateValue(DefaultSettings);
    }

    nn::hid::system::NpadDeviceTypeSet ReturnDeviceTypeIfEnabled() NN_NOEXCEPT
    {
        if (this->IsEnabled())
        {
            return DeviceType;
        }
        else
        {
            nn::hid::system::DeviceTypeSet deviceType;
            return deviceType.Reset();
        }
    }

    void ResetValue() NN_NOEXCEPT
    {
        this->UpdateValue(DefaultSettings);
    }
};

// 点滅させるデバイスの種類
class MenuItemLedBlinking final : public MenuItemHolder
{
private:
    const nn::hid::NpadIdType NpadId;

public:
    explicit MenuItemLedBlinking(const nn::hid::NpadIdType& npadId) NN_NOEXCEPT :
        MenuItemHolder("Blink", true, true),
        NpadId(npadId)
    {
        AddMenuItem(new MenuItemLedDeviceTypeUnit("Joy-Con (L)", nn::hid::system::NpadDeviceType::JoyConLeft::Mask));
        AddMenuItem(new MenuItemLedDeviceTypeUnit("Joy-Con (R)", nn::hid::system::NpadDeviceType::JoyConRight::Mask));
        AddMenuItem(new MenuItemLedDeviceTypeUnit("Pro Controller", nn::hid::system::NpadDeviceType::SwitchProController::Mask));
    }
protected:
    virtual void Proceed() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::hid::system::NpadDeviceTypeSet deviceTypeToBlink;
        deviceTypeToBlink.Reset();

        for (std::vector<MenuItem*>::iterator it = m_pMenuItems.begin();
             it != m_pMenuItems.end();
             ++it)
        {
            deviceTypeToBlink |= reinterpret_cast<MenuItemLedDeviceTypeUnit*>(*it)->ReturnDeviceTypeIfEnabled();
            reinterpret_cast<MenuItemLedDeviceTypeUnit*>(*it)->ResetValue();
        }
        nn::hid::system::SetNpadPlayerLedBlinkingDevice(NpadId, deviceTypeToBlink);
    }
};

// 点滅停止
class MenuItemLedBlinkStop final : public MenuItemHolder
{
private:
    const nn::hid::NpadIdType NpadId;

public:
    explicit MenuItemLedBlinkStop(const nn::hid::NpadIdType& npadId) NN_NOEXCEPT :
        MenuItemHolder("Stop Blinking", true),
        NpadId(npadId)
    {
    }
protected:
    virtual void Proceed() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::hid::system::NpadDeviceTypeSet deviceTypeToBlink;
        deviceTypeToBlink.Reset();
        nn::hid::system::SetNpadPlayerLedBlinkingDevice(NpadId, deviceTypeToBlink);
    }
};

// Led 制御の種類
class MenuItemLedControlTypes final : public MenuItemHolder
{
public:
    MenuItemLedControlTypes(const std::string& name, const nn::hid::NpadIdType& npadId) NN_NOEXCEPT :
        MenuItemHolder(name, true, true)
    {
        AddMenuItem(new MenuItemLedBlinking(npadId));
        AddMenuItem(new MenuItemLedBlinkStop(npadId));
    }
};

// Led 制御 (NpadIdType の選択)
class MenuItemLedControl : public MenuItemNpadIds<MenuItemLedControlTypes>
{
public:
    MenuItemLedControl() NN_NOEXCEPT :
        MenuItemNpadIds<MenuItemLedControlTypes>("Led", false)
    {
        // 何もしない
    }
};

// 切断
class MenuItemDisconnect final : public MenuItemNpadIdsSelectable
{
public:
    MenuItemDisconnect() NN_NOEXCEPT :
        MenuItemNpadIdsSelectable("Disconnect", true, 1)
    {
        // 何もしない
    }
protected:
    virtual void Proceed() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::hid::NpadIdType targetId;
        auto count = this->GetValue(&targetId, 1);
        NN_ABORT_UNLESS_EQUAL(count, 1);
        nn::hid::DisconnectNpad(targetId);
    }
};

MenuItemDeviceControl::MenuItemDeviceControl() NN_NOEXCEPT :
    MenuItemHolder("Device Control", true, true)
{
    AddMenuItem(new MenuItemDisconnect());               // 切断
    AddMenuItem(new MenuItemLedControl());               // Led
    AddMenuItem(new MenuItemNfcControl());               // Nfc
    AddMenuItem(new MenuItemIrSensorControl());          // IrSensor
    AddMenuItem(new MenuItemPalma());          // Palma
}
