﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <string>
#include <nn/nn_Common.h>

#include "PackedButton.h"
#include "ScreenStatics.h"
#include "ScreenUtil.h"
#include "s2d/Simple2D.h"

class IScene
{
    NN_DISALLOW_COPY(IScene);
    NN_DISALLOW_MOVE(IScene);

private:
    //!< スクロール幅
    float m_ScrollPadding = 0.0f;
    // Screen 長
    float m_ScreenHeight = 0.0f;

public:
    IScene() NN_NOEXCEPT { /* 何もしない */ };
    virtual ~IScene() NN_NOEXCEPT { /* 何もしない */ };
    void Draw() NN_NOEXCEPT
    {
        s2d::SetTextColor(ColorSceneTextDefault);
        s2d::SetTextScale(TextScaleDefault);

        // テキスト描画位置の計算
        float scrollPadding = 0.0f;
        // スクロールする以上のサイズがある場合
        if (m_ScreenHeight > YSizeTextInScene - YSizeLine)
        {
            if (this->IsScrollInverted())
            {
                auto padding = (m_ScreenHeight - YSizeTextInScene);
                scrollPadding = (padding - m_ScrollPadding);
            }
            else
            {
                scrollPadding = m_ScrollPadding;
            }
        }

        // テキスト描画
        RecordablePosition position;
        position.Reset({ { PositionSceneText.x, PositionSceneText.y - scrollPadding }  });
        DrawImpl(position);

        m_ScreenHeight = position.GetHeight();

        // スクロールバーの描画
        if (m_ScreenHeight > YSizeTextInScene - YSizeLine + YSizeScroll)
        {
            s2d::Point2D startPoint = PositionScrollStart;
            s2d::Point2D endPoint = PositionScrollEnd;
            s2d::DrawLine(startPoint, endPoint, ColorSceneBorder, WidthBorderDefault);

            s2d::Point2D barStartPosition = PositionScrollStart;
            barStartPosition.x -= XSizeScrollBar / 2;
            float displayedScale = YSizeTextInScene / m_ScreenHeight;
            barStartPosition.y += (scrollPadding / m_ScreenHeight) * YSizeScrollBar;
            auto barLength = YSizeScrollBar * displayedScale;
            s2d::Rectangle bg = { { barStartPosition.x, barStartPosition.y, XSizeScrollBar, barLength } };
            s2d::FillRectangle(bg, ColorSceneBackground);
            s2d::DrawRectangle(bg, ColorSceneBorder, WidthBorderDefault);
        }
    };

    void DrawHeader() NN_NOEXCEPT
    {
        s2d::Point2D position = PositionHeader;
        s2d::Rectangle bg = { { position.x, position.y, HeaderSize.width, HeaderSize.height } };
        s2d::FillRectangle(bg, ColorSceneBackground);

        s2d::Point2D startPoint = position;
        startPoint.y += HeaderSize.height;
        s2d::Point2D endPoint = position;
        endPoint.x += PositionHeader.x + HeaderSize.width;
        endPoint.y = startPoint.y;
        s2d::DrawLine(startPoint, endPoint, ColorSceneBorder, WidthBorderDefault);
        s2d::SetTextColor(ColorSceneTextDefault);
        s2d::SetTextScale(TextScaleDefault);
        position = PositionHeader;
        position.x += XSizePadding * 2;
        position.y += YSizeHeaderPadding;
        s2d::DrawText(position, "%s", GetSceneName().c_str());
        s2d::Size size;
        s2d::GetTextDrawSize(&size, "%s", GetGuideString().c_str());
        position.x = PositionHeader.x + HeaderSize.width - (size.width + XSizePadding * 2);
        s2d::DrawText(position, "%s", GetGuideString().c_str());
    };

    void Scroll(bool up) NN_NOEXCEPT
    {
        // スクロールするほどのサイズがない場合
        if (m_ScreenHeight < YSizeTextInScene - YSizeLine)
        {
            m_ScrollPadding = 0.0f;
        }

        if (!up == this->IsScrollInverted())
        {
            m_ScrollPadding += YSizeScroll;
            if (this->IsScrollInverted())
            {
                if (m_ScreenHeight - YSizeTextInScene < m_ScrollPadding)
                {
                    m_ScrollPadding = m_ScreenHeight - YSizeTextInScene;
                }
            }
            else
            {
                if (PositionSceneText.y + m_ScreenHeight - ScreenSize.height < m_ScrollPadding)
                {
                    m_ScrollPadding -= YSizeScroll;
                }
            }

        }
        else
        {
            m_ScrollPadding -= YSizeScroll;
        }
        if (m_ScrollPadding < 0.0f)
        {
            m_ScrollPadding = 0.0f;
        }
    };

    virtual PackedButton Control(const PackedButton& button) NN_NOEXCEPT { return button; };
    virtual void OnSceneFocused() NN_NOEXCEPT { /* 何もしない */ };

protected:
    virtual void DrawImpl(RecordablePosition& position) NN_NOEXCEPT = 0;
    virtual std::string GetGuideString() NN_NOEXCEPT = 0;
    virtual std::string GetSceneName() NN_NOEXCEPT = 0;
    virtual bool IsScrollInverted() NN_NOEXCEPT { return false; };
};
