﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/system/hid_UniquePad.h>
#include <nn/hid/system/hid_FirmwareUpdate.h>

// 対応する NpadId
const ::nn::hid::NpadIdType NpadIds[] =
{
    ::nn::hid::NpadId::No1,
    ::nn::hid::NpadId::No2,
    ::nn::hid::NpadId::No3,
    ::nn::hid::NpadId::No4,
    ::nn::hid::NpadId::No5,
    ::nn::hid::NpadId::No6,
    ::nn::hid::NpadId::No7,
    ::nn::hid::NpadId::No8
};

const int NpadIdCountMax = NN_ARRAY_SIZE(NpadIds);

//!< サポートしているデバイスか判定します。
bool IsSupportedDevice(const nn::hid::system::UniquePadId& id) NN_NOEXCEPT;

const char* GetUniquePadTypeName(nn::hid::system::UniquePadType padType) NN_NOEXCEPT;
const char* GetUniquePadInterfaceName(nn::hid::system::UniquePadInterface interface) NN_NOEXCEPT;
const char* GetUpdateRequiredReasonString(
    ::nn::Result result,
    ::nn::hid::system::FirmwareUpdateRequiredReason reason) NN_NOEXCEPT;
const char* GetUpdateStageString(nn::hid::system::FirmwareUpdateStage stage) NN_NOEXCEPT;

//!< コントローラのファームウェアバージョンを読み込むクラスです。
class FirmwareVersionReader final
{
    NN_DISALLOW_COPY(FirmwareVersionReader);
    NN_DISALLOW_MOVE(FirmwareVersionReader);

public:
    FirmwareVersionReader() NN_NOEXCEPT :
        m_Mutex(true),
        m_UniquePadId(),
        m_Version(),
        m_IsValid(false),
        m_HasValidVersion(false)
    {
    }

    // 対象コントローラーの UniquePadId を設定
    void SetUniquePadId(const nn::hid::system::UniquePadId& id) NN_NOEXCEPT;

    // ファームウェアバージョンを無効化
    void Invalidate() NN_NOEXCEPT;

    // 有効なバージョンを取得済みか
    bool HasValidVersion() const NN_NOEXCEPT;

    // 読み込んだバージョン情報を取得
    bool Get(nn::hid::system::FirmwareVersion* pOutVersion) const NN_NOEXCEPT;

    // バージョン情報の更新
    void Update() NN_NOEXCEPT;

private:
    void InvalidateVersion() NN_NOEXCEPT;

private:
    mutable nn::os::Mutex m_Mutex;

    nn::hid::system::UniquePadId        m_UniquePadId;
    nn::hid::system::FirmwareVersion    m_Version;
    bool m_IsValid;
    bool m_HasValidVersion;
};
