﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/os.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/system/hid_UniquePad.h>

 /**
  * @brief   左コン操作に関する定義 (NpadController クラスのテンプレート引数として利用します)
  */
struct NpadLeftJoy
{
    typedef nn::hid::NpadStyleJoyLeft Style;
    typedef nn::hid::NpadJoyLeftState State;
};

 /**
  * @brief   右コン操作に関する定義 (NpadController クラスのテンプレート引数として利用します)
  */
struct NpadRightJoy
{
    typedef nn::hid::NpadStyleJoyRight Style;
    typedef nn::hid::NpadJoyRightState State;
};

/**
 * @brief   フルキー操作に関する定義 (NpadController クラスのテンプレート引数として利用します)
 */
struct NpadFullKey
{
    typedef nn::hid::NpadStyleFullKey Style;
    typedef nn::hid::NpadFullKeyState State;
};

 /**
  * @brief   携帯機操作に関する定義 (NpadController クラスのテンプレート引数として利用します)
  */
struct NpadHandheld
{
    typedef nn::hid::NpadStyleHandheld Style;
    typedef nn::hid::NpadHandheldState State;
};

 /**
  * @brief   コントローラーに対する処理を扱うインターフェースクラスです。
  */
class INpadController
{
    NN_DISALLOW_COPY(INpadController);
    NN_DISALLOW_MOVE(INpadController);

public:
    INpadController() NN_NOEXCEPT {}
    virtual ~INpadController() NN_NOEXCEPT {}
    virtual void Update() NN_NOEXCEPT = 0;
    virtual void SetPadType(nn::hid::system::UniquePadType padType) NN_NOEXCEPT = 0;

    virtual nn::hid::NpadIdType GetNpadId() const NN_NOEXCEPT = 0;

    /**
     * @brief   指定されたボタンの組み合わせが押されているか
     */
    virtual bool IsPressed(nn::hid::NpadButtonSet buttons) const NN_NOEXCEPT = 0;

    /**
     * @brief   指定されたボタンの組み合わせが新たに押されたか
     */
    virtual bool IsTriggered(nn::hid::NpadButtonSet buttons) const NN_NOEXCEPT = 0;

    /**
     * @brief   指定されたボタンの組み合わせが離されたか
     */
    virtual bool IsReleased(nn::hid::NpadButtonSet buttons) const NN_NOEXCEPT = 0;

    /**
     * @brief   更新開始ボタンが押されたか
     */
    virtual bool IsUpdateTriggered() const NN_NOEXCEPT = 0;

    /**
     * @brief   ローカル通信切り替えボタンが押されたか
     */
    virtual bool IsLdnTriggered() const NN_NOEXCEPT = 0;

};

/**
 * @brief   操作形態ごとの処理を記述した、 INpadController の派生クラスです。
 */
template<typename Npad>
class NpadController : public INpadController
{
    NN_DISALLOW_COPY(NpadController);
    NN_DISALLOW_MOVE(NpadController);

public:
    explicit NpadController(nn::hid::NpadIdType npadId) NN_NOEXCEPT :
        m_NpadId(npadId),
        m_PadType(),
        m_CurrentButtons(),
        m_PreviousButtons()
    {
    }

    virtual void SetPadType(nn::hid::system::UniquePadType padType) NN_NOEXCEPT NN_OVERRIDE
    {
        m_PadType = padType;
    }

    virtual nn::hid::NpadIdType GetNpadId() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_NpadId;
    }

    virtual bool IsPressed(nn::hid::NpadButtonSet buttons) const NN_NOEXCEPT NN_OVERRIDE
    {
        return (m_CurrentButtons & buttons) == buttons;
    }

    virtual bool IsTriggered(nn::hid::NpadButtonSet buttons) const NN_NOEXCEPT NN_OVERRIDE
    {
        return (m_CurrentButtons & buttons) == buttons &&
            (m_PreviousButtons & buttons) != buttons;
    }

    virtual bool IsReleased(nn::hid::NpadButtonSet buttons) const NN_NOEXCEPT NN_OVERRIDE
    {
        return (m_CurrentButtons & buttons) != buttons &&
            (m_PreviousButtons & buttons) == buttons;
    }

    virtual bool IsUpdateTriggered() const NN_NOEXCEPT NN_OVERRIDE
    {
        switch (m_PadType)
        {
        case nn::hid::system::UniquePadType_LeftController:
            return IsTriggered(nn::hid::NpadButton::L::Mask | nn::hid::NpadButton::ZL::Mask);

        case nn::hid::system::UniquePadType_RightController:
            return IsTriggered(nn::hid::NpadButton::R::Mask | nn::hid::NpadButton::ZR::Mask);

        case nn::hid::system::UniquePadType_FullKeyController:
            return IsTriggered(nn::hid::NpadButton::L::Mask | nn::hid::NpadButton::ZL::Mask) ||
                IsTriggered(nn::hid::NpadButton::R::Mask | nn::hid::NpadButton::ZR::Mask);

        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    virtual bool IsLdnTriggered() const NN_NOEXCEPT NN_OVERRIDE
    {
        switch (m_PadType)
        {
        case nn::hid::system::UniquePadType_LeftController:
            return IsTriggered(nn::hid::NpadButton::Up::Mask);

        case nn::hid::system::UniquePadType_RightController:
            return IsTriggered(nn::hid::NpadButton::X::Mask);

        case nn::hid::system::UniquePadType_FullKeyController:
            return IsTriggered(nn::hid::NpadButton::Up::Mask) ||
                IsTriggered(nn::hid::NpadButton::X::Mask);

        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    virtual void Update() NN_NOEXCEPT NN_OVERRIDE
    {
        UpdateButtons();
    }

private:
    void UpdateButtons() NN_NOEXCEPT
    {
        m_PreviousButtons = m_CurrentButtons;

        typename Npad::State state;
        nn::hid::GetNpadState(&state, m_NpadId);
        m_CurrentButtons = state.buttons;
    }

private:
    const nn::hid::NpadIdType      m_NpadId;            //!< Npad ID
    nn::hid::system::UniquePadType m_PadType;           //!< コントローラーの種類
    nn::hid::NpadButtonSet         m_CurrentButtons;    //!< 現在のボタン情報
    nn::hid::NpadButtonSet         m_PreviousButtons;   //!< 前回のボタン情報
};

/**
 * @brief   接続されているコントローラーを管理するクラスです。
 */
class PadManager final
{
    NN_DISALLOW_COPY(PadManager);
    NN_DISALLOW_MOVE(PadManager);

public:
    PadManager() NN_NOEXCEPT :
        m_Mutex(true),
        m_PadIds(),
        m_PadCount(0),
        m_pNpadControllers(),
        m_NpadControllersStorage()
    {
    }

    /**
     * @brief   接続されているコントローラーの台数を返します。
     */
    int GetPadCount() const NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        return m_PadCount;
    }

    /**
     * @brief   接続可能なコントローラーの台数を返します。
     */
    int GetPadCountMax() const NN_NOEXCEPT
    {
        return static_cast<int>(NN_ARRAY_SIZE(m_PadIds));
    }

    /**
     * @brief   コントローラーが 1 台以上接続されているか判定します。
     */
    bool IsDeviceConnected() const NN_NOEXCEPT
    {
        return GetPadCount() > 0;
    }

    /**
     * @brief   指定したコントローラーインデックスに対応する NpadId を取得します。
     */
    nn::Result GetNpadId(nn::hid::NpadIdType* pOutId, int padIndex) NN_NOEXCEPT;

    /**
     * @brief   指定したコントローラーインデックスに対応する UniquePadId を取得します。
     */
    nn::Result GetUniquePadId(nn::hid::system::UniquePadId* pOutId, int padIndex) NN_NOEXCEPT;

    /**
     * @brief   指定したコントローラーインデックスに対応する NpadController を取得します。
     */
    nn::Result GetNpadController(INpadController** pOutController, int padIndex) NN_NOEXCEPT;

    /**
     * @brief   コントローラー一覧を再取得します。
     */
    void Refresh() NN_NOEXCEPT;

    /**
     * @brief   コントローラー情報を更新します。
     */
    void Update() NN_NOEXCEPT;

private:
    /**
     * @brief   NpadController を格納できる十分なサイズ
     */
    static const size_t NpadControllerStorageSize = 256;

private:
    mutable nn::os::Mutex m_Mutex;

    nn::hid::system::UniquePadId m_PadIds[nn::hid::system::UniquePadIdCountMax];
    int m_PadCount;
    INpadController* m_pNpadControllers[nn::hid::system::UniquePadIdCountMax];
    char m_NpadControllersStorage[nn::hid::system::UniquePadIdCountMax][NpadControllerStorageSize];
};
