﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <mutex>
#include <memory>
#include <new>

#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/hid/system/hid_UniquePad.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/result/result_HandlingUtility.h>

#include "FirmwareUpdate_PadManager.h"
#include "FirmwareUpdate_Util.h"

nn::Result PadManager::GetNpadId(nn::hid::NpadIdType* pOutId, int padIndex) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutId);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (padIndex >= GetPadCount())
    {
        NN_RESULT_THROW(nn::hid::system::ResultUniquePadDisconnected());
    }

    auto padId = m_PadIds[padIndex];
    if (!IsSupportedDevice(padId))
    {
        // 非サポートデバイス
        NN_RESULT_THROW(nn::hid::system::ResultUniquePadDisconnected());
    }

    int controllerNumber;
    auto result = nn::hid::system::GetUniquePadControllerNumber(
        &controllerNumber,
        padId);
    if (nn::hid::system::ResultUniquePadNotAvailable::Includes(result) ||
        controllerNumber == 0)
    {
        // 割り当て無しは Handheld と見なす
        *pOutId = nn::hid::NpadId::Handheld;
    }
    else if (result.IsFailure())
    {
        NN_RESULT_THROW(result);
    }
    else
    {
        // コントローラー番号に対応する NpadId
        NN_ABORT_UNLESS_LESS_EQUAL(controllerNumber, NpadIdCountMax);
        *pOutId = NpadIds[controllerNumber - 1];
    }

    NN_RESULT_SUCCESS;
}

nn::Result PadManager::GetUniquePadId(nn::hid::system::UniquePadId* pOutId, int padIndex) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutId);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (padIndex >= GetPadCount())
    {
        NN_RESULT_THROW(nn::hid::system::ResultUniquePadDisconnected());
    }

    auto padId = m_PadIds[padIndex];
    if (!IsSupportedDevice(padId))
    {
        // 非サポートデバイス
        NN_RESULT_THROW(nn::hid::system::ResultUniquePadDisconnected());
    }

    *pOutId = padId;

    NN_RESULT_SUCCESS;
}

nn::Result PadManager::GetNpadController(INpadController** pOutController, int padIndex) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutController);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (padIndex >= GetPadCount() ||
        !IsSupportedDevice(m_PadIds[padIndex]) ||
        m_pNpadControllers[padIndex] == nullptr)
    {
        NN_RESULT_THROW(nn::hid::system::ResultUniquePadDisconnected());
    }

    *pOutController = m_pNpadControllers[padIndex];

    NN_RESULT_SUCCESS;
}

void PadManager::Refresh() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    m_PadCount = nn::hid::system::ListUniquePads(m_PadIds, nn::hid::system::UniquePadIdCountMax);

    for (int i = 0; i < m_PadCount; i++)
    {
        bool needsRollback = true;
        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                m_pNpadControllers[i] = nullptr;
            }
        };

        nn::hid::NpadIdType npadId;
        if (GetNpadId(&npadId, i).IsFailure())
        {
            continue;
        }

        auto id = m_PadIds[i];
        auto padType = nn::hid::system::GetUniquePadType(id);

        nn::hid::system::UniquePadInterface interface;
        if (nn::hid::system::GetUniquePadInterface(&interface, id).IsFailure())
        {
            continue;
        }
        else if (interface == nn::hid::system::UniquePadInterface_Rail)
        {
            m_pNpadControllers[i] = new(m_NpadControllersStorage[i]) NpadController<NpadHandheld>(npadId);
        }
        else
        {
            switch (padType)
            {
            case nn::hid::system::UniquePadType_LeftController:
                m_pNpadControllers[i] = new(m_NpadControllersStorage[i]) NpadController<NpadLeftJoy>(npadId);
                break;

            case nn::hid::system::UniquePadType_RightController:
                m_pNpadControllers[i] = new(m_NpadControllersStorage[i]) NpadController<NpadRightJoy>(npadId);
                break;

            case nn::hid::system::UniquePadType_FullKeyController:
                m_pNpadControllers[i] = new(m_NpadControllersStorage[i]) NpadController<NpadFullKey>(npadId);
                break;

            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }

        m_pNpadControllers[i]->SetPadType(padType);
        needsRollback = false;
    }
}

void PadManager::Update() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    for (int i = 0; i < m_PadCount; i++)
    {
        if (m_pNpadControllers[i] != nullptr)
        {
            m_pNpadControllers[i]->Update();
        }
    }
}
