﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// デバッグ用にローカル通信を有効にする
#define ENABLE_LDN_DEBUG

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/init/init_Malloc.h>
#include <nn/os.h>
#include <nn/oe.h>
#include <nn/oe/oe_HomeButtonControl.h>

#if defined(ENABLE_LDN_DEBUG)
#include <nn/ldn.h>
#endif

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#include <nv/nv_MemoryManagement.h>
#endif

#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
#include <nn/nn_Windows.h>
#endif

#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/hid/system/hid_UniquePad.h>
#include <nn/hid/system/hid_FirmwareUpdate.h>
#include <nn/hid/system/hid_Result.system.h>
#include <nn/result/result_HandlingUtility.h>

#include "FirmwareUpdate_PadManager.h"
#include "FirmwareUpdate_Util.h"

#include "GraphicSystem.h"
#include "GraphicUtil.h"

namespace
{

const size_t ApplicationHeapSize = 128 * 1024 * 1024;

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
const size_t GraphicsMemorySize = 8 * 1024 * 1024;
#endif  // if defined(NN_BUILD_TARGET_PLATFORM_NX)

const int FrameRate = 60;

#if defined(NN_BUILD_CONFIG_OS_WIN)

WNDPROC DefaultWndProc = 0;

LRESULT CALLBACK WndProc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
    switch (uMsg)
    {
    case WM_ACTIVATEAPP:
        break;

    default:
        break;
    }

    return ::CallWindowProc(DefaultWndProc, hwnd, uMsg, wParam, lParam);
}
#endif  // if defined(NN_BUILD_CONFIG_OS_WIN)

PadManager g_PadManager;
FirmwareVersionReader g_VersionReaders[::nn::hid::system::UniquePadIdCountMax];
bool g_IsUpdateSkipFlagEnabled = false;

// 各種動作フラグ
struct ModeFlags
{
    typedef ::nn::util::BitFlagSet<32, ModeFlags>::Flag<0> IsRunning;           // FW 更新中
    typedef ::nn::util::BitFlagSet<32, ModeFlags>::Flag<1> IsProgressDisplay;   // 進捗表示中
    typedef ::nn::util::BitFlagSet<32, ModeFlags>::Flag<2> IsLdnEnabled;        // ローカル通信有効
};

typedef ::nn::util::BitFlagSet<32, ModeFlags> ModeFlagSet;

// FW 更新の進捗情報
struct UpdateInfo
{
    ModeFlagSet                                   _mode;
    ::nn::hid::system::UniquePadId                targetId;
    ::nn::hid::system::FirmwareUpdateDeviceHandle deviceHandle;
    ::nn::hid::system::FirmwareUpdateState        state;
    ::nn::Result                                  resultForDisplay;
    ::nn::os::Tick                                startTick;
    ::nn::os::Tick                                endTick;

    // FW 更新を実行中か
    bool IsRunning()
    {
        return _mode.Test<ModeFlags::IsRunning>();
    }

    // 進捗表示中か
    bool IsProgressDisplay()
    {
        return _mode.Test<ModeFlags::IsProgressDisplay>();
    }

    // ローカル通信が有効か
    bool IsLdnEnabled()
    {
        return _mode.Test<ModeFlags::IsLdnEnabled>();
    }

    // メンバのクリア
    void Clear()
    {
        _mode.Reset();
        targetId         = ::nn::hid::system::UniquePadId();
        deviceHandle     = ::nn::hid::system::FirmwareUpdateDeviceHandle();
        state            = ::nn::hid::system::FirmwareUpdateState();
        resultForDisplay = ::nn::ResultSuccess();
        startTick        = ::nn::os::Tick();
        endTick          = ::nn::os::Tick();
    }

    // 更新開始
    void Start()
    {
        ::nn::oe::BeginBlockingHomeButton();
        startTick = ::nn::os::GetSystemTick();
        _mode.Set<ModeFlags::IsRunning>();
    }

    // 更新終了
    void Stop()
    {
        ::nn::oe::EndBlockingHomeButton();
        endTick = ::nn::os::GetSystemTick();
        _mode.Reset<ModeFlags::IsRunning>();
    }

    // 進捗情報の表示開始
    void StartDisplayProgress()
    {
        _mode.Set<ModeFlags::IsProgressDisplay>();
    }

    // ローカル通信 ON/OFF 切り替え
    void SetLdnEnabled(bool isEnabled)
    {
        _mode.Set<ModeFlags::IsLdnEnabled>(isEnabled);
    }

} g_UpdateInfo;

// ローカル通信の開始
void StartLocalNetwork()
{
#if defined(ENABLE_LDN_DEBUG)
    if (g_UpdateInfo.IsLdnEnabled())
    {
        return;
    }

    nn::ldn::Initialize();
    nn::ldn::OpenAccessPoint();

    nn::ldn::NetworkConfig network = {};
    network.intentId                  = nn::ldn::MakeIntentId(0x010000000000B22Cull, 0);
    network.channel                   = nn::ldn::AutoChannel;
    network.nodeCountMax              = 1;
    network.localCommunicationVersion = 0;

    nn::ldn::SecurityConfig security = {};
    security.securityMode   = nn::ldn::SecurityMode_Product;
    security.passphraseSize = nn::ldn::PassphraseSizeMin;

    nn::ldn::UserConfig user = {};
    user.userName[0] = 'A';

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::ldn::CreateNetwork(network, security, user));

    g_UpdateInfo.SetLdnEnabled(true);
#endif  // if defined(ENABLE_LDN_DEBUG)
}

// ローカル通信の終了
void StopLocalNetwork()
{
#if defined(ENABLE_LDN_DEBUG)
    if (!g_UpdateInfo.IsLdnEnabled())
    {
        return;
    }

    nn::ldn::DestroyNetwork();
    nn::ldn::CloseAccessPoint();
    nn::ldn::Finalize();

    g_UpdateInfo.SetLdnEnabled(false);
#endif  // if defined(ENABLE_LDN_DEBUG)
}

// FW 更新中かどうか
bool IsUpdating()
{
#if 0
    ::nn::hid::system::FirmwareUpdateState state;
    auto result = ::nn::hid::system::GetFirmwareUpdateState(&state, g_UpdateDeviceHandle);

    return !::nn::hid::system::ResultFirmwareUpdateNotStarted::Includes(result);
#else
    return g_UpdateInfo.IsRunning();
#endif
}

// UniquePad の再取得
void RefreshPadManager()
{
    g_PadManager.Refresh();

    // バージョンの再読み込み
    for (int i = 0; i < g_PadManager.GetPadCountMax(); i++)
    {
        nn::hid::system::UniquePadId uniquePadId;
        if (g_PadManager.GetUniquePadId(&uniquePadId, i).IsSuccess())
        {
            g_VersionReaders[i].SetUniquePadId(uniquePadId);
        }
        else
        {
            g_VersionReaders[i].Invalidate();
        }
    }
}

// Result の内容を表示
void PrintResult(::nn::Result result)
{
    if (result.IsSuccess())
    {
        return;
    }

    if (::nn::hid::system::ResultUniquePadDisconnected::Includes(result))
    {
        NN_LOG("Controller is not connected\n");
    }
    else if (::nn::hid::system::ResultFirmwareUpdateBusy::Includes(result))
    {
        NN_LOG("FirmwareUpdate already running\n");
    }
    else if (::nn::hid::system::ResultFirmwareUpdateNotNeeded::Includes(result))
    {
        NN_LOG("Already updated\n");
    }
    else
    {
        NN_LOG("Unhandled result: 0x%08X (Module=%d, Description=%d)\n",
            result.GetInnerValueForDebug(),
            result.GetModule(),
            result.GetDescription());
    }
}

// アップデート開始
::nn::Result StartUpdate(int padIndex)
{
    if (IsUpdating())
    {
        NN_RESULT_SUCCESS;
    }

    ::nn::hid::system::FirmwareVersion version;
    if (g_VersionReaders[padIndex].Get(&version))
    {
        NN_LOG(
            "FirmwareVersion: %s.%02X.%02X.%02X.%02X\n",
            version.identifier,
            version.major,
            version.minor,
            version.micro,
            version.revision);
    }

    ::nn::hid::system::UniquePadId id;
    NN_RESULT_DO(g_PadManager.GetUniquePadId(&id, padIndex));

    g_UpdateInfo.StartDisplayProgress();
    g_UpdateInfo.targetId  = id;

    auto result = ::nn::hid::system::StartFirmwareUpdate(
        &g_UpdateInfo.deviceHandle,
        id);
    g_UpdateInfo.resultForDisplay = result;
    g_UpdateInfo.startTick = g_UpdateInfo.endTick = ::nn::os::Tick();
    NN_RESULT_DO(result);

    NN_LOG("Update is started\n");
    g_UpdateInfo.Start();

    NN_RESULT_SUCCESS;
}

// アップデート中断
void AbortUpdate()
{
    if (!IsUpdating())
    {
        return;
    }

    ::nn::hid::system::AbortFirmwareUpdate();
    NN_LOG("Aborted\n");
}

// パッド入力処理
void UpdateController()
{
    // FW バージョン情報の更新
    for (auto& reader : g_VersionReaders)
    {
        reader.Update();
    }

    for (int i = 0; i < g_PadManager.GetPadCount(); i++)
    {
        INpadController* pController;
        if (g_PadManager.GetNpadController(&pController, i).IsFailure())
        {
            continue;
        }

        pController->Update();

        if (g_UpdateInfo.IsRunning())
        {
            // + または - で中断
            if (pController->IsTriggered(nn::hid::NpadButton::Plus::Mask) ||
                pController->IsTriggered(nn::hid::NpadButton::Minus::Mask))
            {
                AbortUpdate();
            }
        }
        else
        {
            // L+ZL または R+ZR で開始
            if (pController->IsUpdateTriggered())
            {
                PrintResult(StartUpdate(i));
            }

            // UP または X でローカル通信切り替え
            else if (pController->IsLdnTriggered())
            {
                g_UpdateInfo.IsLdnEnabled() ? StopLocalNetwork() : StartLocalNetwork();
            }

            // Yボタンで Hotfix 起因のファームウェア更新をスキップする
            else if (pController->IsTriggered(nn::hid::NpadButton::Y::Mask))
            {
                g_IsUpdateSkipFlagEnabled = !g_IsUpdateSkipFlagEnabled;
                ::nn::hid::system::SetFirmwareHotfixUpdateSkipEnabled(g_IsUpdateSkipFlagEnabled);
            }

        }
    }
}

// 現在のアップデート状態を更新
void UpdateCurrentState()
{
    if (!g_UpdateInfo.IsRunning())
    {
        return;
    }

    auto result = ::nn::hid::system::GetFirmwareUpdateState(&g_UpdateInfo.state, g_UpdateInfo.deviceHandle);
    if (result.IsFailure())
    {
        PrintResult(result);
        g_UpdateInfo.Stop();
    }

    g_UpdateInfo.resultForDisplay = result;

    if (g_UpdateInfo.state.progress >= 100)
    {
        g_UpdateInfo.Stop();
        NN_LOG("Complete!\n");
    }
}

// アップデート処理の Result を描画
void RenderUpdateResult(DisplayUnit display, nn::Result result)
{
    if (result.IsSuccess())
    {
        return;
    }

    display.SetCursor();
    display.pWriter->SetTextColor(Display_ColorRed);

    if (::nn::hid::system::ResultUniquePadDisconnected::Includes(result))
    {
        display.pWriter->Print("Controller is disconnected");
    }
    else if (::nn::hid::system::ResultFirmwareUpdateBusy::Includes(result))
    {
        display.pWriter->Print("FirmwareUpdate already running");
    }
    else if (::nn::hid::system::ResultFirmwareUpdateNotNeeded::Includes(result))
    {
        display.pWriter->Print("Already updated");
    }
    else if (::nn::hid::system::ResultFirmwareVersionReading::Includes(result))
    {
        display.pWriter->Print("Version reading");
    }
    else if (::nn::hid::system::ResultFirmwareUpdateHardwareError::Includes(result))
    {
        display.pWriter->Print("Hardware error");
    }
    else if (::nn::hid::system::ResultFirmwareUpdateRequireBluetooth::Includes(result))
    {
        display.pWriter->Print("Update on rail is not supported");
    }
    else if (::nn::hid::system::ResultFirmwareUpdateFailed::Includes(result))
    {
        display.pWriter->Print("Update failed");
    }
    else
    {
        display.pWriter->Print("Unhandled result: 0x%08X", result.GetInnerValueForDebug());
    }

    display.pWriter->SetTextColor(Display_ColorWhite);
}

// 進捗バーの描画
void RenderProgressBar(DisplayUnit display, uint8_t progress)
{
    for (int i = 0; i < 100; i++)
    {
        display.SetCursor();
        display.pWriter->SetTextColor(i < progress ? Display_ColorWhite : Display_ColorGray);
        display.pWriter->Print("|");
        display.x += 1.5f;
        display.SetCursor();
        display.pWriter->Print("|");
        display.x += 1.5f;
    }

    display.pWriter->SetTextColor(Display_ColorWhite);
    display.x += 24;
    display.SetCursor();
    display.pWriter->Print("%3d%%", progress);
}

// 進捗情報の描画
void RenderUpdateState(nn::gfx::util::DebugFontTextWriter* pWriter, float x, float y)
{
    if (!g_UpdateInfo.IsProgressDisplay())
    {
        return;
    }

    DisplayUnit display = { pWriter, x, y };
    display.SetCursor();
    display.pWriter->SetTextColor(Display_ColorWhite);
    display.pWriter->Print("Target UniquePad ID: %d", g_UpdateInfo.targetId);

    if (g_UpdateInfo.resultForDisplay.IsSuccess())
    {
        display.pWriter->Print(" (Handle: 0x%08X%08X)",
            static_cast<uint32_t>(g_UpdateInfo.deviceHandle._storage >> 32),
            static_cast<uint32_t>(g_UpdateInfo.deviceHandle._storage));
        display.y += DisplayYOffset_Line;
        display.SetCursor();

        // 進捗状況表示
        display.pWriter->SetTextColor(
            g_UpdateInfo.state.stage == nn::hid::system::FirmwareUpdateStage_Completed ?
            Display_ColorGreen :
            Display_ColorWhite);
        display.pWriter->Print(GetUpdateStageString(g_UpdateInfo.state.stage));

        display.x  = DisplayXOffset_ProgressBar;
        display.y += DisplayYOffset_Line;
        RenderProgressBar(display, g_UpdateInfo.state.progress);
    }
    else
    {
        // エラー表示
        display.x  = DisplayXOffset_UpdateResult;
        display.y += DisplayYOffset_Line * 2;
        RenderUpdateResult(display, g_UpdateInfo.resultForDisplay);
    }

    display.y += DisplayYOffset_Line;
    display.SetCursor();

    // 経過時間表示
    auto tick = IsUpdating() ? ::nn::os::GetSystemTick() : g_UpdateInfo.endTick;
    auto diffTime = (tick - g_UpdateInfo.startTick).ToTimeSpan();
    display.pWriter->Print(
        "Time: %02d:%02d.%03d",
        diffTime.GetMinutes(),
        diffTime.GetSeconds() % 60,
        diffTime.GetMilliSeconds() % 1000);
}

// 単一のコントローラー情報の描画
void RenderControllerInfo(DisplayUnit display, nn::hid::system::UniquePadId id)
{
    int uniquePadControllerNumber;
    auto resultControllerNumber = ::nn::hid::system::GetUniquePadControllerNumber(
        &uniquePadControllerNumber,
        id);

    ::nn::hid::system::UniquePadInterface uniquePadInterface;
    auto resultInterface = ::nn::hid::system::GetUniquePadInterface(
        &uniquePadInterface,
        id);

    auto uniquePadType = ::nn::hid::system::GetUniquePadType(id);

    display.SetCursor();

    if (resultControllerNumber.IsFailure() ||
        resultInterface.IsFailure())
    {
        display.pWriter->Print("???");
        return;
    }

    display.x = DisplayXOffset_UniquePadId;
    display.SetCursor();
    display.pWriter->Print("%6X", id._storage);
    display.x = DisplayXOffset_ControllerNumber;
    display.SetCursor();
    display.pWriter->Print("%4d", uniquePadControllerNumber);
    display.x = DisplayXOffset_InterfaceName;
    display.SetCursor();
    display.pWriter->Print("%s", GetUniquePadInterfaceName(uniquePadInterface));
    display.x = DisplayXOffset_TypeName;
    display.SetCursor();
    display.pWriter->Print("%s", GetUniquePadTypeName(uniquePadType));
    display.x = DisplayXOffset_Reason;
    display.SetCursor();

    nn::hid::system::FirmwareUpdateRequiredReason reason;
    auto reasonResult = nn::hid::system::CheckFirmwareUpdateRequired(&reason, id);
    if (nn::hid::system::ResultFirmwareUpdateHardwareError::Includes(reasonResult))
    {
        display.pWriter->SetTextColor(Display_ColorRed);
    }
    else if (reason != nn::hid::system::FirmwareUpdateRequiredReason_Nothing)
    {
        display.pWriter->SetTextColor(Display_ColorOrange);
    }
    else
    {
        display.pWriter->SetTextColor(Display_ColorWhite);
    }
    display.pWriter->Print("%s", GetUpdateRequiredReasonString(reasonResult, reason));
    display.x = DisplayXOffset_FwVersion;
    display.SetCursor();

    nn::hid::system::FirmwareVersion version;
    auto versionResult = nn::hid::system::GetFirmwareVersion(&version, id);
    if (versionResult.IsSuccess())
    {
        bool isAvailable;
        auto result = nn::hid::system::IsFirmwareUpdateAvailable(&isAvailable, id);
        if (result.IsSuccess())
        {
            // 更新可能なコントローラーを強調
            display.pWriter->SetTextColor(isAvailable ? Display_ColorOrange : Display_ColorWhite);
        }
        else if (nn::hid::system::ResultFirmwareUpdateRequireBluetooth::Includes(result))
        {
            display.pWriter->SetTextColor(Display_ColorOrange);
        }
        else
        {
            isAvailable = false;
            display.pWriter->SetTextColor(
                nn::hid::system::ResultFirmwareUpdateHardwareError::Includes(result) ?
                Display_ColorRed :
                Display_ColorWhite);
        }

        display.pWriter->Print(
            "%s.%02x.%02x.%02x.%02x",
            version.identifier,
            version.major,
            version.minor,
            version.micro,
            version.revision);

        if (nn::hid::system::ResultFirmwareUpdateHardwareError::Includes(result))
        {
            display.pWriter->Print("  (HW error)");
        }
        else if (nn::hid::system::ResultFirmwareUpdateRequireBluetooth::Includes(result))
        {
            display.pWriter->Print("  (Update on rail is not supported)");
        }
        else if (isAvailable)
        {
            nn::hid::system::FirmwareVersion newVersion;
            nn::hid::system::GetAvailableFirmwareVersion(&newVersion, id);
            display.pWriter->Print(
                "  (→ %s.%02x.%02x.%02x.%02x)",
                newVersion.identifier,
                newVersion.major,
                newVersion.minor,
                newVersion.micro,
                newVersion.revision);
        }
        else
        {
            display.pWriter->Print("  (Latest)");
        }
    }
    else if (nn::hid::system::ResultFirmwareVersionReading::Includes(versionResult))
    {
        display.pWriter->Print("Reading...");
    }

    display.pWriter->SetTextColor(Display_ColorWhite);

}  // NOLINT(impl/function_size)

// コントローラー一覧の描画
void RenderControllerList(nn::gfx::util::DebugFontTextWriter* pWriter, float x, float y)
{
    DisplayUnit display = { pWriter, x, y };

    // ヘッダ
    display.x = DisplayXOffset_UniquePadId;
    display.SetCursor();
    display.pWriter->Print("Unique\nPad ID");
    display.x  = DisplayXOffset_ControllerNumber;
    display.y += DisplayYOffset_Line * 0.5f;
    display.SetCursor();
    display.pWriter->Print("Cnt #");
    display.x = DisplayXOffset_InterfaceName;
    display.SetCursor();
    display.pWriter->Print("Interface");
    display.x = DisplayXOffset_TypeName;
    display.SetCursor();
    display.pWriter->Print("Type");
    display.x = DisplayXOffset_Reason;
    display.SetCursor();
    display.pWriter->Print("Req. reason");
    display.x = DisplayXOffset_FwVersion;
    display.SetCursor();
    display.pWriter->Print("FW Ver.");

    display.x = x;
    display.y += DisplayYOffset_Line * 1.5f;

    // 各コントローラーの情報
    for (int i = 0; i < g_PadManager.GetPadCount(); i++)
    {
        nn::hid::system::UniquePadId id;
        if (g_PadManager.GetUniquePadId(&id, i).IsFailure())
        {
            continue;
        }

        RenderControllerInfo(display, id);
        display.y += DisplayYOffset_Line;
    }
}

// ローカル通信状態の描画
void RenderLocalNetworkStatus(nn::gfx::util::DebugFontTextWriter* pWriter, float x, float y)
{
#if defined(ENABLE_LDN_DEBUG)
    DisplayUnit display = { pWriter, x, y };

    display.SetCursor();
    display.pWriter->SetTextColor(Display_ColorWhite);
    display.pWriter->Print("Local network : ");

    if (g_UpdateInfo.IsLdnEnabled())
    {
        display.pWriter->Print("ON");
    }
    else
    {
        display.pWriter->SetTextColor(Display_ColorGray);
        display.pWriter->Print("OFF");
        display.pWriter->SetTextColor(Display_ColorWhite);
    }
#else
    NN_UNUSED(pWriter);
    NN_UNUSED(x);
    NN_UNUSED(y);
#endif  // if defined(ENABLE_LDN_DEBUG)
}

// 操作ヘルプの描画
void RenderHelp(nn::gfx::util::DebugFontTextWriter* pWriter, float x, float y)
{
    DisplayUnit display = { pWriter, x, y };

    display.SetCursor();
    display.pWriter->SetTextColor(Display_ColorWhite);

    if (g_UpdateInfo.IsRunning())
    {
        display.pWriter->Print("Press + or - to abort");
    }
    else
    {
        display.pWriter->Print("Press L+ZL or R+ZR to update");

#if defined(ENABLE_LDN_DEBUG)
        display.y -= DisplayYOffset_Line;
        display.SetCursor();
        display.pWriter->Print(
            "Press UP or X to <%s> local network",
            g_UpdateInfo.IsLdnEnabled() ? "disable" : "enable");
#endif  // if defined(ENABLE_LDN_DEBUG)

        display.y -= DisplayYOffset_Line;
        display.SetCursor();
        display.pWriter->Print("Press Y to <%s> UpdateSkipMode",
            g_IsUpdateSkipFlagEnabled ? "disable" : "enable");
    }
}

// 画面描画
void Render(nn::gfx::util::DebugFontTextWriter* pWriter)
{
    NN_ASSERT_NOT_NULL(pWriter);

    pWriter->SetScale(1.8f, 1.8f);
    pWriter->SetCursor(24, 24);
    pWriter->SetTextColor(Display_ColorWhite);
    pWriter->Print("Controller Firmware Update Test Tool");

    pWriter->SetScale(1.3f, 1.3f);
    RenderControllerList(pWriter, 32, 96);
    RenderUpdateState(pWriter, DisplayXOffset_UpdateStage, DisplayYOffset_UpdateInfo);
    RenderLocalNetworkStatus(pWriter, DisplayXOffset_LocalNetwork, DisplayYOffset_LocalNetwork);
    RenderHelp(pWriter, DisplayXOffset_Help, DisplayYOffset_Help);
}

// 終了リクエスト判定
bool IsExitRequested()
{
#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
    MSG msg;
    while (::PeekMessageA(&msg, NULL, 0, 0, PM_REMOVE))
    {
        if (msg.message == WM_QUIT)
        {
            return true;
        }

        ::TranslateMessage(&msg);
        ::DispatchMessageA(&msg);
    }

#elif defined(NN_BUILD_TARGET_PLATFORM_NX)
    ::nn::oe::Message message;
    if (!::nn::oe::TryPopNotificationMessage(&message))
    {
        return false;
    }

    switch (message)
    {
    case ::nn::oe::MessageExitRequest:
        return true;

    case ::nn::oe::MessageFocusStateChanged:
        if (::nn::oe::GetCurrentFocusState() == ::nn::oe::FocusState_InFocus)
        {
            // ツール外で割り当てモードが変わることがあるので、アクティブになったら元に戻す
            for (auto id : NpadIds)
            {
                ::nn::hid::SetNpadJoyAssignmentModeSingle(id);
            }
        }
        break;

    default:
        break;
    }
#endif

    return false;
}

} //namespace

extern "C" void nnMain()
{
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    nv::SetGraphicsAllocator(NvAllocate, NvFree, NvReallocate, nullptr);
    nv::SetGraphicsDevtoolsAllocator(NvAllocate, NvFree, NvReallocate, nullptr);
    nv::InitializeGraphics(std::malloc(GraphicsMemorySize), GraphicsMemorySize);
#endif  // if defined(NN_BUILD_TARGET_PLATFORM_NX)

    ApplicationHeap applicationHeap;
    applicationHeap.Initialize(ApplicationHeapSize);

    auto* pGraphicsSystem = new ::GraphicsSystem();
    pGraphicsSystem->SetApplicationHeap(&applicationHeap);
    pGraphicsSystem->Initialize();

#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
    HWND hwnd = reinterpret_cast<HWND>(pGraphicsSystem->GetNativeWindowHandle());

    DefaultWndProc = reinterpret_cast<WNDPROC>(::GetWindowLongPtr(hwnd, GWLP_WNDPROC));
    ::SetWindowLongPtr(hwnd, GWLP_WNDPROC, reinterpret_cast<LONG_PTR>(WndProc));

    ::RegisterTouchWindow(hwnd, 0);
#endif  // if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)

    NN_LOG("Firmware Update Test Tool Start.\n");

    ::nn::hid::InitializeNpad();
    ::nn::hid::SetSupportedNpadIdType(NpadIds, NpadIdCountMax);
    ::nn::hid::SetSupportedNpadStyleSet(
        ::nn::hid::NpadStyleFullKey::Mask |
        ::nn::hid::NpadStyleJoyLeft::Mask |
        ::nn::hid::NpadStyleJoyRight::Mask |
        ::nn::hid::NpadStyleHandheld::Mask);

    // 1 本横持ちにする
    for (auto id : NpadIds)
    {
        ::nn::hid::SetNpadJoyAssignmentModeSingle(id);
    }
    ::nn::hid::SetNpadHandheldActivationMode(::nn::hid::NpadHandheldActivationMode_Single);

    ::nn::os::SystemEventType connectionEvent;
    ::nn::hid::system::BindUniquePadConnectionEvent(
        &connectionEvent,
        ::nn::os::EventClearMode_ManualClear);

    ::nn::hid::system::SetFirmwareHotfixUpdateSkipEnabled(false);
    ::nn::hid::system::InitializeFirmwareUpdate();

    // 終了通知を受け取る
    ::nn::oe::SetFocusHandlingMode(::nn::oe::FocusHandlingMode_Notify);
    ::nn::oe::EnterExitRequestHandlingSection();

    g_UpdateInfo.Clear();

    // XXX: 少し待たないと UniquePad の情報が正しく取れないことがあるので W/A
    ::nn::os::SleepThread(::nn::TimeSpan::FromMilliSeconds(100));

    while (!IsExitRequested())
    {
        if (::nn::os::TryWaitSystemEvent(&connectionEvent))
        {
            // コントローラーの接続状態が変化した
            ::nn::os::ClearSystemEvent(&connectionEvent);
            RefreshPadManager();
        }

        UpdateController();
        UpdateCurrentState();

        auto* pTextWriter = &pGraphicsSystem->GetDebugFont();
        Render(pTextWriter);

        pGraphicsSystem->BeginDraw();
        pTextWriter->Draw(&pGraphicsSystem->GetCommandBuffer());
        pGraphicsSystem->EndDraw();
        pGraphicsSystem->Synchronize(
            nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / FrameRate));
    }

    StopLocalNetwork();
    ::nn::hid::system::SetFirmwareHotfixUpdateSkipEnabled(false);

    ::nn::oe::EndBlockingHomeButton();
    ::nn::oe::LeaveExitRequestHandlingSection();
    ::nn::os::DestroySystemEvent(&connectionEvent);

    pGraphicsSystem->Finalize();
    delete pGraphicsSystem;

    applicationHeap.Finalize();

    NN_LOG("Firmware Update Test Tool Done.\n");
}
