﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "frm_Npad.h"

namespace frm {
namespace hid {

    /* Singleton パターン */
    Npad& Npad::GetInstance() NN_NOEXCEPT
    {
        static Npad Instance;
        return Instance;
    }

    const int PressCnt = 30;
    nn::hid::NpadButtonSet BtnBuf[PressCnt] = {};

    int BeatCnt = 0;

    /* ------------------------------------------------------------ */
    // PRIVATE関数
    /* ------------------------------------------------------------ */
    void Npad::GetBtnState(NPadState* st) NN_NOEXCEPT
    {
        nn::hid::NpadHandheldState  stHand[NpadIdCountMax];
        nn::hid::NpadFullKeyState   stFull[NpadIdCountMax];
        nn::hid::NpadJoyDualState   stDual[NpadIdCountMax];
        nn::hid::NpadJoyLeftState   stLeft[NpadIdCountMax];
        nn::hid::NpadJoyRightState  stRight[NpadIdCountMax];
        nn::hid::NpadHandheldState  stHandOld[NpadIdCountMax];
        nn::hid::NpadFullKeyState   stFullOld[NpadIdCountMax];
        nn::hid::NpadJoyDualState   stDualOld[NpadIdCountMax];
        nn::hid::NpadJoyLeftState   stLeftOld[NpadIdCountMax];
        nn::hid::NpadJoyRightState  stRightOld[NpadIdCountMax];

        // 前回値の保存
        st->btnOld = st->btn;
        // ボタン状態のリセット
        st->btn.Reset();

        for (int i = 0; i < NpadIdCountMax; i++)
        {
            //現在有効な操作形態(NpadStyleSet)を取得
            nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(NpadIds[i]);

            // フルキー操作が有効な場合
            if (style.Test<nn::hid::NpadStyleFullKey>() == true)
            {
                // 前回の状態を保存
                stFullOld[i] = stFull[i];
                // ボタン入力状態の取得
                nn::hid::GetNpadState(&(stFull[i]), NpadIds[i]);
                st->btn |= stFull[i].buttons;

                // サンプリング値の取得
                st->smpl = stFull[i].samplingNumber;

                // アナログスティックのアナログ値取得
                st->anlgL = stFull[i].analogStickL;
                st->anlgR = stFull[i].analogStickR;
            }
            // Joy-Con 操作が有効な場合
            else if (style.Test<nn::hid::NpadStyleJoyDual>() == true)
            {
                // 前回の状態を保存
                stDualOld[i] = stDual[i];
                // ボタン入力状態の取得
                nn::hid::GetNpadState(&(stDual[i]), NpadIds[i]);
                st->btn |= stDual[i].buttons;

                // サンプリング値の取得
                st->smpl = stDual[i].samplingNumber;

                // アナログスティックのアナログ値取得
                st->anlgL = stDual[i].analogStickL;
                st->anlgR = stDual[i].analogStickR;
            }
            // 携帯機コントローラー操作が有効な場合
            else if (style.Test<nn::hid::NpadStyleHandheld>() == true)
            {
                // 前回の状態を保存
                stHandOld[i] = stHand[i];
                // ボタン入力状態の取得
                nn::hid::GetNpadState(&(stHand[i]), NpadIds[i]);
                st->btn |= stHand[i].buttons;

                // サンプリング値の取得
                st->smpl = stHand[i].samplingNumber;

                // アナログスティックのアナログ値取得
                st->anlgL = stHand[i].analogStickL;
                st->anlgR = stHand[i].analogStickR;
            }
            // JoyCon片手持ち操作(Left)が有効な場合
            else if (style.Test<nn::hid::NpadStyleJoyLeft>() == true)
            {
                // 前回の状態を保存
                stLeftOld[i] = stLeft[i];
                // ボタン入力状態の取得
                nn::hid::GetNpadState(&(stLeft[i]), NpadIds[i]);

                if (nn::hid::GetNpadJoyHoldType() == nn::hid::NpadJoyHoldType_Horizontal)
                {
                    const nn::hid::NpadButtonSet temp = stLeft[i].buttons;

                    stLeft[i].buttons.Set<nn::hid::NpadButton::A>(temp.Test<nn::hid::NpadButton::Down>());
                    stLeft[i].buttons.Set<nn::hid::NpadButton::B>(temp.Test<nn::hid::NpadButton::Left>());
                    stLeft[i].buttons.Set<nn::hid::NpadButton::X>(temp.Test<nn::hid::NpadButton::Right>());
                    stLeft[i].buttons.Set<nn::hid::NpadButton::Y>(temp.Test<nn::hid::NpadButton::Up>());

                    stLeft[i].buttons.Set<nn::hid::NpadButton::Right>(stLeft[i].buttons.Test<nn::hid::NpadButton::StickLDown>());
                    stLeft[i].buttons.Set<nn::hid::NpadButton::Down>(stLeft[i].buttons.Test<nn::hid::NpadButton::StickLLeft>());
                    stLeft[i].buttons.Set<nn::hid::NpadButton::Up>(stLeft[i].buttons.Test<nn::hid::NpadButton::StickLRight>());
                    stLeft[i].buttons.Set<nn::hid::NpadButton::Left>(stLeft[i].buttons.Test<nn::hid::NpadButton::StickLUp>());

                    stLeft[i].buttons.Set<nn::hid::NpadButton::L>(stLeft[i].buttons.Test<nn::hid::NpadJoyButton::L>() | stLeft[i].buttons.Test<nn::hid::NpadJoyButton::LeftSL>());
                    stLeft[i].buttons.Set<nn::hid::NpadButton::R>(stLeft[i].buttons.Test<nn::hid::NpadJoyButton::LeftSR>());

                    stLeft[i].buttons.Set<nn::hid::NpadButton::StickLRight>(temp.Test<nn::hid::NpadButton::StickLDown>());
                    stLeft[i].buttons.Set<nn::hid::NpadButton::StickLDown>(temp.Test<nn::hid::NpadButton::StickLLeft>());
                    stLeft[i].buttons.Set<nn::hid::NpadButton::StickLUp>(temp.Test<nn::hid::NpadButton::StickLRight>());
                    stLeft[i].buttons.Set<nn::hid::NpadButton::StickLLeft>(temp.Test<nn::hid::NpadButton::StickLUp>());

                    stLeft[i].buttons.Set<nn::hid::NpadJoyButton::LeftSL>(false);
                    stLeft[i].buttons.Set<nn::hid::NpadJoyButton::LeftSR>(false);
                }

                st->btn |= stLeft[i].buttons;

                // サンプリング値の取得
                st->smpl = stLeft[i].samplingNumber;

                // アナログスティックのアナログ値取得
                st->anlgL = stLeft[i].analogStickL;
            }
            // JoyCon片手持ち操作(Right)が有効な場合
            else if (style.Test<nn::hid::NpadStyleJoyRight>() == true)
            {
                // 前回の状態を保存
                stRightOld[i] = stRight[i];
                // ボタン入力状態の取得
                nn::hid::GetNpadState(&(stRight[i]), NpadIds[i]);

                if (nn::hid::GetNpadJoyHoldType() == nn::hid::NpadJoyHoldType_Horizontal)
                {
                    const nn::hid::NpadButtonSet temp = stRight[i].buttons;

                    stRight[i].buttons.Set<nn::hid::NpadButton::A>(temp.Test<nn::hid::NpadButton::X>());
                    stRight[i].buttons.Set<nn::hid::NpadButton::B>(temp.Test<nn::hid::NpadButton::A>());
                    stRight[i].buttons.Set<nn::hid::NpadButton::X>(temp.Test<nn::hid::NpadButton::Y>());
                    stRight[i].buttons.Set<nn::hid::NpadButton::Y>(temp.Test<nn::hid::NpadButton::B>());

                    stRight[i].buttons.Set<nn::hid::NpadButton::Right>(stRight[i].buttons.Test<nn::hid::NpadButton::StickRUp>());
                    stRight[i].buttons.Set<nn::hid::NpadButton::Down>(stRight[i].buttons.Test<nn::hid::NpadButton::StickRRight>());
                    stRight[i].buttons.Set<nn::hid::NpadButton::Up>(stRight[i].buttons.Test<nn::hid::NpadButton::StickRLeft>());
                    stRight[i].buttons.Set<nn::hid::NpadButton::Left>(stRight[i].buttons.Test<nn::hid::NpadButton::StickRDown>());

                    stRight[i].buttons.Set<nn::hid::NpadButton::L>(stRight[i].buttons.Test<nn::hid::NpadJoyButton::RightSL>());
                    stRight[i].buttons.Set<nn::hid::NpadButton::R>(stRight[i].buttons.Test<nn::hid::NpadJoyButton::R>() | stRight[i].buttons.Test<nn::hid::NpadJoyButton::RightSR>());

                    stRight[i].buttons.Set<nn::hid::NpadButton::StickRRight>(temp.Test<nn::hid::NpadButton::StickRUp>());
                    stRight[i].buttons.Set<nn::hid::NpadButton::StickRDown>(temp.Test<nn::hid::NpadButton::StickRRight>());
                    stRight[i].buttons.Set<nn::hid::NpadButton::StickRUp>(temp.Test<nn::hid::NpadButton::StickRLeft>());
                    stRight[i].buttons.Set<nn::hid::NpadButton::StickRLeft>(temp.Test<nn::hid::NpadButton::StickRDown>());

                    stRight[i].buttons.Set<nn::hid::NpadJoyButton::RightSL>(false);
                    stRight[i].buttons.Set<nn::hid::NpadJoyButton::RightSR>(false);
                }

                st->btn |= stRight[i].buttons;

                // サンプリング値の取得
                st->smpl = stRight[i].samplingNumber;

                // アナログスティックのアナログ値取得
                st->anlgR = stRight[i].analogStickR;
            }
            else
            {
                /* 何もしない */
            }
        }
    } // NOLINT(impl/function_size)

    //!< ボタンのトリガー入力を取得します。
    nn::hid::NpadButtonSet Npad::GetBtnTrg(const nn::hid::NpadButtonSet current,
                                           const nn::hid::NpadButtonSet previous)
    {
        return (current ^ previous) & current;
    }

    //!< ボタンの長押しを取得します。
    nn::hid::NpadButtonSet Npad::GetBtnLongPress(const nn::hid::NpadButtonSet current)
    {
        nn::hid::NpadButtonSet result;
        nn::hid::NpadButtonSet t_BtnBuf[PressCnt];

        // バッファリング
        memcpy(t_BtnBuf, BtnBuf, sizeof(t_BtnBuf) * 8);
        for (int i = 1; i < PressCnt; i++)
        {
            BtnBuf[i] = t_BtnBuf[i - 1];
        }
        BtnBuf[0] = current;

        // 全てのビットの論理積を返す
        result = BtnBuf[0];
        for (int i = 1; i < PressCnt; i++)
        {
            result &= BtnBuf[i];
        }
        return result;
    }

    //!< ボタン長押しによる周期入力を取得します。
    nn::hid::NpadButtonSet Npad::GetBtnBeat(const nn::hid::NpadButtonSet current)
    {
        nn::hid::NpadButtonSet result;

        if (current.IsAnyOn())
        {
            BeatCnt++;
            if (BeatCnt > 10)
            {
                result = current;
                BeatCnt = 0;
            }
            else {
                result.Reset();
            }
        }
        else
        {
            result.Reset();
            BeatCnt = 0;
        }
        return result;
    }

    /* ------------------------------------------------------------ */
    // PUBLIC関数
    /* ------------------------------------------------------------ */
    void Npad::Initialize() NN_NOEXCEPT
    {
        nn::hid::InitializeNpad();
        // 使用する Npad を設定
        nn::hid::SetSupportedNpadIdType(NpadIds, NpadIdCountMax);
        //使用する操作形態を設定
        nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask
                                        | nn::hid::NpadStyleJoyDual::Mask
                                        | nn::hid::NpadStyleHandheld::Mask
                                        | nn::hid::NpadStyleJoyLeft::Mask
                                        | nn::hid::NpadStyleJoyRight::Mask
                                        );

        // 片手持ちの場合は全てHoldType_Horizontalにする
        nn::hid::SetNpadJoyHoldType(nn::hid::NpadJoyHoldType_Horizontal);
        // 常に操作形態の変更を行えるようにする
        nn::hid::StartLrAssignmentMode();

        Npad::ChangeActivationMode(0);
    }

    void Npad::Update(NPadState* st) NN_NOEXCEPT
    {
        GetBtnState(st);
        st->trg = GetBtnTrg(st->btn, st->btnOld);
        st->longpress = GetBtnLongPress(st->btn);
        st->beat = GetBtnBeat(st->longpress);
    }

    int Npad::ChangeActivationMode(int md) NN_NOEXCEPT
    {
        if (md == 0)
        {
            nn::hid::SetNpadHandheldActivationMode(nn::hid::NpadHandheldActivationMode_Dual);
        }
        else if (md == 1)
        {
            nn::hid::SetNpadHandheldActivationMode(nn::hid::NpadHandheldActivationMode_Single);
        }
        else if(md == 2)
        {
            nn::hid::SetNpadHandheldActivationMode(nn::hid::NpadHandheldActivationMode_None);
        }
        else
        {
            nn::hid::SetNpadHandheldActivationMode(nn::hid::NpadHandheldActivationMode_Dual);
            md = 0;
        }
        return md;
    }
}
}
// [EOF]
