﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/gfx.h>
#include <nn/vi.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>

#include "AppHeap.h"


class GraphicsSystem
{
    enum { ViewPortCount = 5 };

    NN_DISALLOW_COPY(GraphicsSystem);
    NN_DISALLOW_MOVE(GraphicsSystem);

private:
    nn::mem::StandardAllocator* m_pApplicationHeap;

    int m_Width;
    int m_Height;

    nn::gfx::MemoryPool m_MemoryPool;
    void* m_pMemoryPool;
    ptrdiff_t m_pMemoryPoolOffset;
    ptrdiff_t m_pMemoryPoolCommandOffset;

    nn::gfx::MemoryPool m_MemoryPoolInvisible;
    void* m_pMemoryPoolInvisible;
    ptrdiff_t m_pMemoryPoolInvisibleOffset;

    nn::gfx::DescriptorPool m_TextureDescriptorPool;
    int m_TextureDescriptorBaseIndex;

    nn::gfx::DescriptorPool m_SamplerDescriptorPool;
    int m_SamplerDescriptorBaseIndex;

    nn::vi::Display* m_pDisplay;
    nn::vi::Layer* m_pLayer;
    nn::vi::NativeWindowHandle m_NativeWindowHandle;
    nn::gfx::Device m_Device;
    nn::gfx::SwapChain m_SwapChain;
    nn::gfx::Queue m_Queue;
    nn::gfx::Fence m_Fence;
    nn::gfx::Texture m_ColorTargetTexture;
    nn::gfx::Texture m_DepthStencilTexture;
    nn::gfx::DepthStencilView m_DepthStencilView;
    nn::gfx::CommandBuffer m_CommandBuffer;
    nn::gfx::ViewportScissorState m_ViewportScissorState[ViewPortCount];
    nn::gfx::BlendState m_BlendState;
    nn::gfx::DepthStencilState m_DepthStencilState;
    nn::gfx::RasterizerState m_RasterizerState;

    nns::gfx::PrimitiveRenderer::Renderer* m_pPrimitiveRenderer;

    std::vector<void*> m_CommandBufferControlMemoryChunks;

public:
    GraphicsSystem() NN_NOEXCEPT;
    ~GraphicsSystem() NN_NOEXCEPT;

    nn::gfx::DescriptorPool& GetTextureDescriptorPool() NN_NOEXCEPT;

    int GetTextureDescriptorBaseIndex() const NN_NOEXCEPT;

    nn::gfx::DescriptorPool& GetSamplerDescriptorPool() NN_NOEXCEPT;

    int GetSamplerDescriptorBaseIndex() const NN_NOEXCEPT;

    int GetWidth() const NN_NOEXCEPT;

    int GetHeight() const NN_NOEXCEPT;

    nn::vi::NativeWindowHandle GetNativeWindowHandle() const NN_NOEXCEPT;

    nn::gfx::Device& GetDevice() NN_NOEXCEPT;

    nn::gfx::CommandBuffer& GetCommandBuffer() NN_NOEXCEPT;

    nn::gfx::ViewportScissorState& GetViewportScissor(int i) NN_NOEXCEPT;

    nns::gfx::PrimitiveRenderer::Renderer& GetPrimitiveRenderer() NN_NOEXCEPT;

    void Initialize(
        nn::mem::StandardAllocator* pApplicationHeap, int width, int height) NN_NOEXCEPT;

    void Finalize() NN_NOEXCEPT;

    void BeginDraw() NN_NOEXCEPT;

    void EndDraw() NN_NOEXCEPT;

    void Synchronize(const nn::TimeSpan& timeout) NN_NOEXCEPT;

private:
    static void OutOfCommandMemoryEventCallback(
        nn::gfx::CommandBuffer* pCommandBuffer,
        const nn::gfx::OutOfMemoryEventArg& arg) NN_NOEXCEPT;

    static void OutOfControlMemoryEventCallback(
        nn::gfx::CommandBuffer* pCommandBuffer,
        const nn::gfx::OutOfMemoryEventArg& arg) NN_NOEXCEPT;

    static void* AllocateFunction(size_t size,
        size_t alignment,
        void* pUserData) NN_NOEXCEPT;

    static void FreeFunction(void* ptr, void* pUserData) NN_NOEXCEPT;

    void InitializeDisplay() NN_NOEXCEPT;

    void InitializeDevice() NN_NOEXCEPT;

    void InitializeMemoryPool() NN_NOEXCEPT;

    void InitializeMemoryPoolInvisible() NN_NOEXCEPT;

    void InitializeSwapChain() NN_NOEXCEPT;

    void InitializeQueue() NN_NOEXCEPT;

    void InitializeFence() NN_NOEXCEPT;

    void InitializeColorTargetTexture() NN_NOEXCEPT;

    void InitializeDepthStencilTexture() NN_NOEXCEPT;

    void InitializeDepthStencilView() NN_NOEXCEPT;

    void InitializeTextureDescriptorPool() NN_NOEXCEPT;

    void InitializeSamplerDescriptorPool() NN_NOEXCEPT;

    void InitializeCommandBuffer() NN_NOEXCEPT;

    void InitializeViewportScissorState(
        nn::gfx::ViewportScissorState* pViewportScissorState,
        const int originX, const int originY, const int width, const int height) NN_NOEXCEPT;

    void InitializeViewportScissorState() NN_NOEXCEPT;

    void InitializeBlendState() NN_NOEXCEPT;

    void InitializeDepthStencilState() NN_NOEXCEPT;

    void InitializeRasterizerState() NN_NOEXCEPT;

    void InitializePrimitiveRenderer() NN_NOEXCEPT;

    void FinalizeMemoryPool() NN_NOEXCEPT;

    void FinalizeMemoryPoolInvisible() NN_NOEXCEPT;

    void FinalizeViewportScissorState() NN_NOEXCEPT;

    void FinalizeBlendState() NN_NOEXCEPT;

    void FinalizePrimitiveRenderer() NN_NOEXCEPT;

    void ResetCommandBuffer() NN_NOEXCEPT;

    void AddCommandMemoryToCommandBuffer(nn::gfx::CommandBuffer* pCommandBuffer
                                         ) NN_NOEXCEPT;

    void AddControlMemoryToCommandBuffer(nn::gfx::CommandBuffer* pCommandBuffer
                                         ) NN_NOEXCEPT;
};
