﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @brief
    本体 6 軸センサーの値を波形表示します
*/

#include "Main_DrawWave.h"

namespace main{

    //!< 座標軸を描画します。
    void DrawWave::WriteAxis(GraphicsSystem* pGraphicsSystem, float offsetX, float offsetY) NN_NOEXCEPT
    {
        // レンダラの生成
        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();
        // 線の幅を指定
        pPrimitiveRenderer->SetLineWidth(1.f);
        // 描画カラーの指定
        pPrimitiveRenderer->SetColor(HalfColor::Gray);
        // 縦軸を描画する
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DLine(&pGraphicsSystem->GetCommandBuffer(), offsetX, offsetY - 120, offsetX, offsetY + 120);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DLine(&pGraphicsSystem->GetCommandBuffer(), offsetX, offsetY - 120, offsetX - 10, offsetY - 110);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DLine(&pGraphicsSystem->GetCommandBuffer(), offsetX, offsetY - 120, offsetX + 10, offsetY - 110);
        // 横軸を描画する
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DLine(&pGraphicsSystem->GetCommandBuffer(), offsetX - 20, offsetY, offsetX + 300, offsetY);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DLine(&pGraphicsSystem->GetCommandBuffer(), offsetX + 300, offsetY, offsetX + 290, offsetY - 10);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DLine(&pGraphicsSystem->GetCommandBuffer(), offsetX + 300, offsetY, offsetX + 290, offsetY + 10);
        // 枠を描画する
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), offsetX - 30, offsetY - 140, 350, 280);
    }

    // センサデータをバッファリングします。
    void DrawWave::BufferSensorDataAxis(frm::hid::SixAxisState* pState, SensorBuffer* pAccelBuffer, SensorBuffer* pAngulerRateBuffer) NN_NOEXCEPT
    {
        SensorBuffer tempAccelBuffer[SensorBufferNum];
        SensorBuffer tempAngulerRateBuffer[SensorBufferNum];

        memcpy(tempAccelBuffer, pAccelBuffer, sizeof(float) * 3 * SensorBufferNum);
        memcpy(tempAngulerRateBuffer, pAngulerRateBuffer, sizeof(float) * 3 * SensorBufferNum);
        for (int i = 1; i < SensorBufferNum; i++)
        {
            pAccelBuffer[i].X = tempAccelBuffer[i - 1].X;
            pAccelBuffer[i].Y = tempAccelBuffer[i - 1].Y;
            pAccelBuffer[i].Z = tempAccelBuffer[i - 1].Z;
            pAngulerRateBuffer[i].X = tempAngulerRateBuffer[i - 1].X;
            pAngulerRateBuffer[i].Y = tempAngulerRateBuffer[i - 1].Y;
            pAngulerRateBuffer[i].Z = tempAngulerRateBuffer[i - 1].Z;

        }
        pAccelBuffer[0].X = pState->acceleration.x;
        pAccelBuffer[0].Y = pState->acceleration.y;
        pAccelBuffer[0].Z = pState->acceleration.z;
        pAngulerRateBuffer[0].X = pState->angularVelocity.x;
        pAngulerRateBuffer[0].Y = pState->angularVelocity.y;
        pAngulerRateBuffer[0].Z = pState->angularVelocity.z;
    }

    //　センサデータを波形表示します。
    void DrawWave::DrawWaveData(GraphicsSystem* pGraphicsSystem, SensorBuffer* value, float offsetX, float offsetY) NN_NOEXCEPT
    {
        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();

        pPrimitiveRenderer->SetLineWidth(1.5f);

        const float Gain = 28.f;
        const float Coef = 5.f;

        for (int i = 1; i < SensorBufferNum; i++)
        {
            pPrimitiveRenderer->SetColor(Color::Red);
            float x = offsetX + i * Coef;
            float y = offsetY - value[i].X * Gain;
            float x_old = offsetX + (i - 1) * Coef;
            float y_old = offsetY - value[i - 1].X * Gain;
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DLine(&pGraphicsSystem->GetCommandBuffer(), x, y, x_old, y_old);

            pPrimitiveRenderer->SetColor(Color::Green);
            x = offsetX + i * Coef;
            y = offsetY - value[i].Y * Gain;
            x_old = offsetX + (i - 1) * Coef;
            y_old = offsetY - value[i - 1].Y * Gain;
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DLine(&pGraphicsSystem->GetCommandBuffer(), x, y, x_old, y_old);

            pPrimitiveRenderer->SetColor(Color::Indigo);
            x = offsetX + i * Coef;
            y = offsetY - value[i].Z * Gain;
            x_old = offsetX + (i - 1) * Coef;
            y_old = offsetY - value[i - 1].Z * Gain;
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DLine(&pGraphicsSystem->GetCommandBuffer(), x, y, x_old, y_old);
        }
    }

    void DrawWave::Draw( nn::gfx::util::DebugFontTextWriter* pTextWriter,
                                GraphicsSystem* pGraphicsSystem,
                                frm::hid::SixAxisState sixAxisState,
                                SensorBuffer* pAccelBuffer,
                                SensorBuffer* pAngulerRateBuffer,
                                const float offsetX,
                                const float accelOffsetY) NN_NOEXCEPT
    {
        const float rateOffsetY = accelOffsetY + 300;

        pTextWriter->SetScale(0.8f, 0.8f);

        pTextWriter->SetTextColor(Color::Gray);
        pTextWriter->SetCursor(offsetX - 20, accelOffsetY - 140);
        pTextWriter->Print("加速度");
        pTextWriter->SetCursor(offsetX + 140, accelOffsetY - 140);
        pTextWriter->Print("凡例");
        pTextWriter->SetTextColor(Color::Red);
        pTextWriter->SetCursor(offsetX + 180, accelOffsetY - 140);
        pTextWriter->Print("赤：X");
        pTextWriter->SetTextColor(Color::Green);
        pTextWriter->SetCursor(offsetX + 220, accelOffsetY - 140);
        pTextWriter->Print("緑：Y");
        pTextWriter->SetTextColor(Color::Indigo);
        pTextWriter->SetCursor(offsetX + 260, accelOffsetY - 140);
        pTextWriter->Print("青：Z");

        pTextWriter->SetTextColor(Color::Gray);
        pTextWriter->SetCursor(offsetX - 20, accelOffsetY - 140);
        pTextWriter->Print("角速度");
        pTextWriter->SetCursor(offsetX + 140, accelOffsetY - 140);
        pTextWriter->Print("凡例");
        pTextWriter->SetTextColor(Color::Red);
        pTextWriter->SetCursor(offsetX + 180, accelOffsetY - 140);
        pTextWriter->Print("赤：X");
        pTextWriter->SetTextColor(Color::Green);
        pTextWriter->SetCursor(offsetX + 220, accelOffsetY - 140);
        pTextWriter->Print("緑：Y");
        pTextWriter->SetTextColor(Color::Indigo);
        pTextWriter->SetCursor(offsetX + 260, accelOffsetY - 140);
        pTextWriter->Print("青：Z");

        BufferSensorDataAxis(&sixAxisState, pAccelBuffer, pAngulerRateBuffer);

        DrawWaveData(pGraphicsSystem, pAccelBuffer, offsetX, accelOffsetY);
        DrawWaveData(pGraphicsSystem, pAngulerRateBuffer, offsetX, rateOffsetY);

        WriteAxis(pGraphicsSystem, offsetX, accelOffsetY);
        WriteAxis(pGraphicsSystem, offsetX, rateOffsetY);
    }


}
// [EOF]
