﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @brief
    本体 6 軸センサーを使って SixAxisSensor の状態を取得するためのサンプルプログラム
*/

#include <cstdlib>
#include <string>
#include <vector>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nn/hid.h>
#include <nn/hid/hid_NpadSixAxisSensor.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/hid/hid_SevenSixAxisSensor.h>
#include <nn/TargetConfigs/build_Platform.h>
#include <nn/oe.h>
#include <nn/err.h>
#include <nn/err/err_ShowErrorApiForSystem.h>
#include <nn/err/err_LibraryApi.h>


#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#include <nv/nv_MemoryManagement.h>
#endif

#include <AppHeap.h>
#include <FontSys.h>
#include <GraphicsSys.h>
#include <WindowMessage.h>

#include <Color.h>
#include <frm_Layout.h>
#include <frm_Npad.h>
#include <frm_SixAxis.h>
#include <frm_Applet.h>
#include <frm_VrMd.h>
#include <frm_Touch.h>

#include "Main_Menu.h"
#include "Main_DrawWave.h"
#include "Main_DrawCube.h"

namespace {

const size_t ApplicationHeapSize = 128 * 1024 * 1024;

const int FrameBufferWidth = 1280;

const int FrameBufferHeight = 720;

const int FrameRate = 60;

template<typename T, std::size_t U>
std::size_t GetArrayLength(const T (&staticArray)[U])
{
    NN_UNUSED(staticArray);
    return U;
}

bool vrMode = false;
bool sevenSixAxisSnsMd = false;
bool npadSixAxisSnsMd = false;
bool sevenSixAxisSnsGetMd = false;
bool npadSixAxisSnsGetMd = false;
int NpadActvMd;

bool isInitializeNpad = false;
bool isInitializeSeven = false;

int counter = 0;

int CursolPosition = 0;
int DesidePosition = 0;

frm::hid::TapState TouchBtnSt;
frm::hid::TapState TouchBtnStDummy;

// 通常メニューリスト
std::string MenuListNow[] = {  "図形初期位置設定"
                              ,"HandHeldActivate切替"
                              ,"Npad6軸Initialize"
                              ,"本体6軸Initialize"
                              ,"本体6軸Finalize"
                              ,"本体6軸ON/OFF"
                              ,"本体6軸データ取得開始/停止"
                              ,"Npad6軸ON/OFF"
                              ,"Npad6軸データ取得開始/停止"
                              ,"VRモードON/OFF"
                              ,"C-Supアプレット呼出"
                              ,"U-Selアプレット呼出"
                              ,"終了(NULL)"
                              ,"終了(abort)"
                              ,"表示切替(波形/図形)"
};
// 通常メニューリストの数
int MenuNumNow = sizeof(MenuListNow) / sizeof(std::string);

// レガシーメニューリスト(カーソル選択不可のもの)
std::string CommonMenuListNow[] = {  "(HOME)→(X)→(A) : 正常終了"
                                    ,"方向キーorスティック：メニュー選択"
                                    ,"(A)：メニューの決定"
};
// レガシーメニューリストの数
int CommonMenuNumNow = sizeof(CommonMenuListNow) / sizeof(std::string);

//!< 画面上部に表示する基本情報を描画します。
void WriteBasicInfo(nn::gfx::util::DebugFontTextWriter* pTextWriter) NN_NOEXCEPT
{
    float posX = 10.f;
    float posY = 5.f;

    nn::oe::DisplayVersion version;
    nn::oe::GetDisplayVersion(&version);

    pTextWriter->SetScale(1, 1);
    pTextWriter->SetTextColor(Color::White);
    pTextWriter->SetCursor(posX, posY);
    pTextWriter->Print("APP VER：%s", version.value);

    pTextWriter->SetTextColor(Color::Orange);
    pTextWriter->SetCursor(posX + 50, posY + 50);
    pTextWriter->Print("Old");

    pTextWriter->SetTextColor(Color::Violet);
    pTextWriter->SetCursor(posX + 470, posY + 50);
    pTextWriter->Print("New");
}

//!< 6軸センサーの状態を描画します。
void WriteState(nn::gfx::util::DebugFontTextWriter* pTextWriter,
                GraphicsSystem* pGraphicsSystem,
                const frm::hid::SixAxisState& sixst,
                const frm::hid::SixAxisState& sevenSixst,
                const float offsetX,
                const float offsetY) NN_NOEXCEPT
{
    float posX = offsetX + 5;
    float posY = offsetY + 5;
    float statusPosX = offsetX + 180;
    float sevenstatusPosX = statusPosX + 110;

    const float DeltaY = 25.0f;

    // レンダラの生成
    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();
    // 線の幅を指定
    pPrimitiveRenderer->SetLineWidth(1.f);
    // 描画カラーの指定
    pPrimitiveRenderer->SetColor(Color::White);

    pTextWriter->SetScale(0.9f, 0.9f);
    pTextWriter->SetTextColor(Color::White);
    pTextWriter->SetCursor(posX, posY - 30);
    pTextWriter->Print("Parameter");
    pTextWriter->SetCursor(statusPosX, posY - 30);
    pTextWriter->Print("Old(Npad)");
    pTextWriter->SetCursor(sevenstatusPosX, posY - 30);
    pTextWriter->Print("New(Seven)");

    for (int i = 0; i < 16; i++)
    {
        pPrimitiveRenderer->SetColor(HalfColor::White);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), offsetX, (i * DeltaY) + offsetY, 400, DeltaY);
        pPrimitiveRenderer->SetColor(HalfColor::Gray);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DLine(&pGraphicsSystem->GetCommandBuffer(), statusPosX, (i * DeltaY) + offsetY, statusPosX, (i * 30) + offsetY + DeltaY);
        if (i < 13)
        {
            pGraphicsSystem->GetPrimitiveRenderer().Draw2DLine(&pGraphicsSystem->GetCommandBuffer(), sevenstatusPosX, (i * DeltaY) + offsetY, sevenstatusPosX, (i * DeltaY) + offsetY + DeltaY);
        }

    }
    pTextWriter->SetScale(0.8f, 0.8f);
    pTextWriter->SetTextColor(Color::White);
    pTextWriter->SetCursor(posX, posY);
    pTextWriter->Print("SamplingNum [count]");
    pTextWriter->SetCursor(statusPosX + 10, posY);
    pTextWriter->Print("%lld", sixst.samplingNum);
    pTextWriter->SetCursor(sevenstatusPosX + 10, posY);
    pTextWriter->Print("%lld", sevenSixst.samplingNum);

    posY += DeltaY;

    pTextWriter->SetTextColor(Color::White);
    pTextWriter->SetCursor(posX, posY);
    pTextWriter->Print("CoordinationNum [count]");
    pTextWriter->SetCursor(statusPosX + 10, posY);
    pTextWriter->Print("－");
    pTextWriter->SetCursor(sevenstatusPosX + 10, posY);
    pTextWriter->Print("%lld", sevenSixst.coordinationNum);

    posY += DeltaY;

    pTextWriter->SetTextColor(Color::White);
    pTextWriter->SetCursor(posX, posY);
    pTextWriter->Print("Delta Time [msec]");
    pTextWriter->SetCursor(statusPosX + 10, posY);
    pTextWriter->Print("%lld", sixst.timeStamp.GetMilliSeconds());
    pTextWriter->SetCursor(sevenstatusPosX + 10, posY);
    pTextWriter->Print("%lld", sevenSixst.timeStamp.GetMilliSeconds());

    posY += DeltaY;
    pTextWriter->SetCursor(posX, posY);
    pTextWriter->Print("X 加速度[G]");
    pTextWriter->SetCursor(statusPosX + 10, posY);
    pTextWriter->Print("%.5f", sixst.acceleration.x);
    pTextWriter->SetCursor(sevenstatusPosX + 10, posY);
    pTextWriter->Print("%.5f", sevenSixst.acceleration.x);

    posY += DeltaY;
    pTextWriter->SetCursor(posX, posY);
    pTextWriter->Print("Y 加速度[G]");
    pTextWriter->SetCursor(statusPosX + 10, posY);
    pTextWriter->Print("%.5f", sixst.acceleration.y);
    pTextWriter->SetCursor(sevenstatusPosX + 10, posY);
    pTextWriter->Print("%.5f", sevenSixst.acceleration.y);

    posY += DeltaY;
    pTextWriter->SetCursor(posX, posY);
    pTextWriter->Print("Z 加速度[G]");
    pTextWriter->SetCursor(statusPosX + 10, posY);
    pTextWriter->Print("%.5f", sixst.acceleration.z);
    pTextWriter->SetCursor(sevenstatusPosX + 10, posY);
    pTextWriter->Print("%.5f", sevenSixst.acceleration.z);

    posY += DeltaY;
    pTextWriter->SetCursor(posX, posY);
    pTextWriter->Print("X 角速度[1/360dps]");
    pTextWriter->SetCursor(statusPosX + 10, posY);
    pTextWriter->Print("%.5f", sixst.angularVelocity.x);
    pTextWriter->SetCursor(sevenstatusPosX + 10, posY);
    pTextWriter->Print("%.5f", sevenSixst.angularVelocity.x);

    posY += DeltaY;
    pTextWriter->SetCursor(posX, posY);
    pTextWriter->Print("Y 角速度[1/360dps]");
    pTextWriter->SetCursor(statusPosX + 10, posY);
    pTextWriter->Print("%.5f", sixst.angularVelocity.y);
    pTextWriter->SetCursor(sevenstatusPosX + 10, posY);
    pTextWriter->Print("%.5f", sevenSixst.angularVelocity.y);

    posY += DeltaY;
    pTextWriter->SetCursor(posX, posY);
    pTextWriter->Print("Z 角速度[1/360dps]");
    pTextWriter->SetCursor(statusPosX + 10, posY);
    pTextWriter->Print("%.5f", sixst.angularVelocity.z);
    pTextWriter->SetCursor(sevenstatusPosX + 10, posY);
    pTextWriter->Print("%.5f", sevenSixst.angularVelocity.z);

    posY += DeltaY;
    pTextWriter->SetCursor(posX, posY);
    pTextWriter->Print("静止判定");
    pTextWriter->SetCursor(statusPosX + 10, posY);

    if (sixst.isRest != false)
    {
        pTextWriter->SetTextColor(HalfColor::White);
        pTextWriter->Print("静止");
    }
    else
    {
        pTextWriter->SetTextColor(Color::White);
        pTextWriter->Print("動作");
    }

    pTextWriter->SetCursor(sevenstatusPosX + 10, posY);

    if (sevenSixst.isRest != false)
    {
        pTextWriter->SetTextColor(HalfColor::White);
        pTextWriter->Print("静止");
    }
    else
    {
        pTextWriter->SetTextColor(Color::White);
        pTextWriter->Print("動作");
    }

    posY += DeltaY;
    pTextWriter->SetCursor(posX, posY);
    pTextWriter->SetTextColor(Color::Yellow);
    pTextWriter->Print("6軸Init状況");
    pTextWriter->SetCursor(statusPosX + 10, posY);
    if (isInitializeNpad == false)
    {
        pTextWriter->SetTextColor(HalfColor::Yellow);
        pTextWriter->Print("未初期化");
    }
    else
    {
        pTextWriter->SetTextColor(Color::Yellow);
        pTextWriter->Print("初期化済");
    }
    pTextWriter->SetCursor(sevenstatusPosX + 10, posY);
    if (isInitializeSeven == false)
    {
        pTextWriter->SetTextColor(HalfColor::Yellow);
        pTextWriter->Print("未初期化");
    }
    else
    {
        pTextWriter->SetTextColor(Color::Yellow);
        pTextWriter->Print("初期化済");
    }

    posY += DeltaY;
    pTextWriter->SetCursor(posX, posY);
    pTextWriter->SetTextColor(Color::Yellow);
    pTextWriter->Print("6軸Start状況");
    pTextWriter->SetCursor(statusPosX + 10, posY);

    if (npadSixAxisSnsMd == false)
    {
        pTextWriter->SetTextColor(HalfColor::Yellow);
        pTextWriter->Print("STOP");
    }
    else
    {
        pTextWriter->SetTextColor(Color::Yellow);
        pTextWriter->Print("START");
    }
    pTextWriter->SetCursor(sevenstatusPosX + 10, posY);
    if (sevenSixAxisSnsMd == false)
    {
        pTextWriter->SetTextColor(HalfColor::Yellow);
        pTextWriter->Print("STOP");
    }
    else
    {
        pTextWriter->SetTextColor(Color::Yellow);
        pTextWriter->Print("START");
    }

    posY += DeltaY;
    pTextWriter->SetCursor(posX, posY);
    pTextWriter->SetTextColor(Color::Yellow);
    pTextWriter->Print("6軸Get状況");
    pTextWriter->SetCursor(statusPosX + 10, posY);
    if (npadSixAxisSnsGetMd == false)
    {
        pTextWriter->SetTextColor(HalfColor::Yellow);
        pTextWriter->Print("Getしていない");
    }
    else
    {
        pTextWriter->SetTextColor(Color::Yellow);
        pTextWriter->Print("Getしている");
    }
    pTextWriter->SetCursor(sevenstatusPosX + 10, posY);
    if (sevenSixAxisSnsGetMd == false)
    {
        pTextWriter->SetTextColor(HalfColor::Yellow);
        pTextWriter->Print("Getしていない");
    }
    else
    {
        pTextWriter->SetTextColor(Color::Yellow);
        pTextWriter->Print("Getしている");
    }

    posY += DeltaY;
    posY += DeltaY;
    pTextWriter->SetTextColor(Color::Yellow);
    pTextWriter->SetCursor(posX, posY);
    pTextWriter->Print("HandheldActivate");
    pTextWriter->SetCursor(statusPosX + 10, posY);

    if (NpadActvMd == 0)
    {
        pTextWriter->Print("Dual");
    }
    else if (NpadActvMd == 1)
    {
        pTextWriter->Print("Single");
    }
    else /* NpadActvMd == 2 */
    {
        pTextWriter->Print("None");
    }

    posY += DeltaY;
    pTextWriter->SetCursor(posX, posY);
    pTextWriter->SetTextColor(Color::Yellow);
    pTextWriter->Print("VRモード");
    pTextWriter->SetCursor(statusPosX + 10, posY);

    if (vrMode != false)
    {
        pTextWriter->Print("ON");
    }
    else
    {
        pTextWriter->SetTextColor(HalfColor::Yellow);
        pTextWriter->Print("OFF");
    }

    pTextWriter->SetTextColor(Color::Yellow);
    posY += DeltaY;

} // NOLINT(impl/function_size)

//!< 6軸センサーの入力状態を出力します。
void PrintSixAxisSensorState(frm::hid::SixAxisState& pState)
{
    NN_LOG("Frequency: %d\n",(pState.samplingNum - pState.samplingNumPrev));
    NN_LOG("DataCount: %d\n",pState.samplingCount);
    NN_LOG("Accl[%.5f, %.5f, %.5f]\n", pState.acceleration.x, pState.acceleration.y, pState.acceleration.z);
    NN_LOG("Rate[%.5f, %.5f, %.5f]\n", pState.angularVelocity.x, pState.angularVelocity.y, pState.angularVelocity.z);
    pState.samplingNumPrev = pState.samplingNum;
}

//!< ConsoleSixAxisSensor 向けの処理を記述したクラスです。
class ConsoleSixAxisSensor
{
    NN_DISALLOW_COPY(ConsoleSixAxisSensor);
    NN_DISALLOW_MOVE(ConsoleSixAxisSensor);

private:
    int jdgTouch = 0;

    frm::hid::NPadState          m_ButtonState;
    frm::hid::SixAxisState       m_SixAxisState;
    frm::hid::SixAxisState       m_SevenSixAxisState;
    nn::util::Quaternion         m_Quaternion;
    nn::util::Quaternion         m_SevenQuaternion;
    const nn::hid::NpadIdType*   m_pId;

    uint32_t        m_FramerateCounter;
    nn::os::Tick    m_FramerateFirstTick;
    int64_t         m_FramerateFirstSample;
    float           m_FramerateComputation;
    float           m_PacketDropPercentage;
    main::Menu      menu;
    main::DrawWave  wave;
    main::DrawCube  cube;
    main::SensorBuffer AccelBuffer[main::SensorBufferNum];
    main::SensorBuffer AngularRateBuffer[main::SensorBufferNum];
    main::SensorBuffer SevenAccelBuffer[main::SensorBufferNum];
    main::SensorBuffer SevenAngularRateBuffer[main::SensorBufferNum];
    main::Menu::Menustate menuSt;

    bool isViewCube = false;

public:

    explicit ConsoleSixAxisSensor(const nn::hid::NpadIdType& id) NN_NOEXCEPT
        : m_pId(&id)
    {
        m_Quaternion = nn::util::Quaternion::Identity();
        m_SevenQuaternion = nn::util::Quaternion::Identity();
    }

    virtual ~ConsoleSixAxisSensor() NN_NOEXCEPT { /* 何もしない */ };

    void Initialize() NN_NOEXCEPT
    {
        // VRモードOFF
        gVrMd.Stop();
        vrMode = gVrMd.UpdateState();

        gTouchScreen.Initialize();

        gNpadSixAxisSns.GetInitialValue(&m_SixAxisState);
        gSevenSixAxisSns.GetInitialValue(&m_SevenSixAxisState);

        for (auto i = 0; i < main::SensorBufferNum; i++)
        {
            AccelBuffer[i].X = 0.f;
            AccelBuffer[i].Y = 0.f;
            AccelBuffer[i].Z = 0.f;
            AngularRateBuffer[i].X = 0.f;
            AngularRateBuffer[i].Y = 0.f;
            AngularRateBuffer[i].Z = 0.f;
            SevenAccelBuffer[i].X = 0.f;
            SevenAccelBuffer[i].Y = 0.f;
            SevenAccelBuffer[i].Z = 0.f;
            SevenAngularRateBuffer[i].X = 0.f;
            SevenAngularRateBuffer[i].Y = 0.f;
            SevenAngularRateBuffer[i].Z = 0.f;
        }
    }

    bool CanPrint() NN_NOEXCEPT
    {
        bool jdg = false;

        nn::oe::FocusState st;

        st = nn::oe::GetCurrentFocusState();

        jdg = true;
        return (jdg);
    }

    void Print() NN_NOEXCEPT
    {
        if (counter < 60)
        {
            counter++;
        }
        else
        {
            if (nn::oe::GetCurrentFocusState() == nn::oe::FocusState_OutOfFocus
                || nn::oe::GetCurrentFocusState() == nn::oe::FocusState_Background)
            {
                NN_LOG("[ApplicationState] : BG\n");
            }
            else
            {
                NN_LOG("[ApplicationState] : FG\n");
            }
            NN_LOG("[Old]--------------------------------\n");
            NN_LOG("Start or Stop : %s\n", npadSixAxisSnsMd ? "START" : "STOP");
            NN_LOG("GetData       : %s\n", npadSixAxisSnsGetMd ? "ENABLE" : "DISABLE");
            PrintSixAxisSensorState(m_SixAxisState);
            NN_LOG("Angle[%.5f, %.5f, %.5f]\n", m_SixAxisState.angle.x, m_SixAxisState.angle.y, m_SixAxisState.angle.z);
            NN_LOG("-------------------------------------\n");
            NN_LOG("[New]--------------------------------\n");
            NN_LOG("Start or Stop : %s\n", sevenSixAxisSnsMd ? "START" : "STOP");
            NN_LOG("GetData       : %s\n", sevenSixAxisSnsGetMd ? "ENABLE" : "DISABLE");
            PrintSixAxisSensorState(m_SevenSixAxisState);
            NN_LOG("Rotation[%.5f, %.5f, %.5f]\n", m_SevenSixAxisState.quaternion.GetX(), m_SevenSixAxisState.quaternion.GetY(), m_SevenSixAxisState.quaternion.GetZ());
            NN_LOG("-------------------------------------\n");
            NN_LOG("\n");
            counter = 0;
        }
    }

    void Update() NN_NOEXCEPT
    {
        gNPad.Update(&m_ButtonState);

        if (npadSixAxisSnsGetMd != false)
        {
            gNpadSixAxisSns.GetState(&m_SixAxisState);
        }

        if (sevenSixAxisSnsGetMd != false)
        {
            gSevenSixAxisSns.GetState(&m_SevenSixAxisState);
        }

        // 現在のVRモード取得
        vrMode = gVrMd.UpdateState();

        gTouchScreen.Update();

        menuSt = menu.GetMenuState(m_ButtonState);

        m_FramerateCounter++;
    }

    void OnOff() NN_NOEXCEPT
    {
        bool isNullAccess = false;

        if ((m_ButtonState.trg.Test<nn::hid::NpadButton::A>() && menuSt == menu.Menustate_Do && DesidePosition == 1) || TouchBtnSt.trgOffOn == true)
        {
            switch (CursolPosition)
            {
                case 0:
                    m_Quaternion = m_SixAxisState.quaternion;
                    m_SevenQuaternion = m_SevenSixAxisState.quaternion;
                    break;
                case 1:
                    if (NpadActvMd == 0)
                    {
                        NpadActvMd = gNPad.ChangeActivationMode(1);
                    }
                    else if (NpadActvMd == 1)
                    {
                        NpadActvMd = gNPad.ChangeActivationMode(2);
                    }
                    else /* NpadActvMd == 2 */
                    {
                        NpadActvMd = gNPad.ChangeActivationMode(0);
                    }
                    break;
                case 2:
                    if (!isInitializeNpad)
                    {
                        gNpadSixAxisSns.Initialize();
                        NN_LOG("NpadSixAxisSensor is Initialized.\n");
                        isInitializeNpad = true;
                    }
                    else
                    {
                        NN_LOG("Already Initialized.\n");
                    }

                    break;
                case 3:
                    if (!isInitializeSeven)
                    {
                        gSevenSixAxisSns.Initialize();
                        NN_LOG("ConsoleSixAxisSensor is Initialized.\n");
                        isInitializeSeven = true;
                    }
                    else
                    {
                        NN_LOG("Already Initialized.\n");
                    }
                    break;
                case 4:
                    gSevenSixAxisSns.Finalize();
                    // 差し戻し
                    isInitializeSeven = false;
                    sevenSixAxisSnsMd = false;
                    sevenSixAxisSnsGetMd = false;
                    NN_LOG("ConsoleSixAxisSensor is Finalized.\n");
                    break;
                case 5:
                    sevenSixAxisSnsMd = (sevenSixAxisSnsMd == false) ? gSevenSixAxisSns.Start() : gSevenSixAxisSns.Stop();
                    break;
                case 6:
                    sevenSixAxisSnsGetMd = (sevenSixAxisSnsGetMd == false) ? true : false;
                    break;
                case 7:
                    npadSixAxisSnsMd = (npadSixAxisSnsMd == false) ? gNpadSixAxisSns.Start(&m_SixAxisState) : gNpadSixAxisSns.Stop(&m_SixAxisState);
                    break;
                case 8:
                    npadSixAxisSnsGetMd = (npadSixAxisSnsGetMd == false) ? true : false;
                    break;
                case 9:
                    (vrMode == false) ? gVrMd.Start() : gVrMd.Stop();
                    break;
                case 10:
                    gApplet.CtlrSupThred();
                    break;
                case 11:
                    gApplet.AcntMngThred();
                    break;
                case 12:
                    isNullAccess = true;
                    break;
                case 13:
                    // アボート
                    NN_ABORT_UNLESS_NOT_NULL(nullptr);
                    break;
                case 14:
                    // アボート
                    isViewCube = (isViewCube == false) ? true : false;
                    break;
                default:
                    break;
            }
            if (isNullAccess != false)
            {
                // NULLポインタアクセス
                int* error = nullptr;
                *error = 0xFFFFFFFF;
            }
        }
        else if ((m_ButtonState.trg.Test<nn::hid::NpadButton::A>() && menuSt == menu.Menustate_Do && DesidePosition == 0))
        {
            menuSt = menu.ForceSetMenuState(menu.Menustate_Set);
        }
        else
        {
            gApplet.ThredDestroy();
        }

        if (m_ButtonState.trg.Test<nn::hid::NpadButton::Plus>() || m_ButtonState.trg.Test<nn::hid::NpadButton::Minus>())
        {
            m_Quaternion = m_SixAxisState.quaternion;
            m_SevenQuaternion = m_SevenSixAxisState.quaternion;
        }
    } // NOLINT(impl/function_size)

    bool Quit()
    {
        return false;
    }

    void Draw(nn::gfx::util::DebugFontTextWriter* pTextWriter, GraphicsSystem* pGraphicsSystem) NN_NOEXCEPT
    {
        WriteBasicInfo(pTextWriter);

        // 状態の表示
        WriteState(pTextWriter, pGraphicsSystem,m_SixAxisState,m_SevenSixAxisState,870, 280);

        jdgTouch = gTouchScreen.CalRelativeAxis(pTextWriter);
        menu.SetTouchPosition(jdgTouch);

        // メニューリストをセット
        menu.SetList(&MenuListNow[0], MenuNumNow, &CommonMenuListNow[0], CommonMenuNumNow);

        if (menuSt == menu.Menustate_Set)
        {
            // カーソルの選択
            CursolPosition = menu.Select(m_ButtonState);
        }
        else if (menuSt == menu.Menustate_Ready)
        {
            DesidePosition = menu.Deside(m_ButtonState);
        }
        else
        {
            // 何もしない
        }

        // 選択式メニューの表示
        menu.DrawRing(pTextWriter, pGraphicsSystem, 880, 40);
        // メニューリストを解放
        menu.ResetList();

        if (isViewCube == false)
        {
            // グラフの波形表示
            wave.Draw(pTextWriter, pGraphicsSystem, m_SixAxisState, &AccelBuffer[0], &AngularRateBuffer[0], 80, 220);
            wave.Draw(pTextWriter, pGraphicsSystem, m_SevenSixAxisState, &SevenAccelBuffer[0], &SevenAngularRateBuffer[0], 500, 220);
        }
        else
        {
            // 図形の描画
            cube.Draw(pGraphicsSystem, (m_SixAxisState.quaternion / m_Quaternion), 1);
            cube.Draw(pGraphicsSystem, (m_SevenSixAxisState.quaternion / m_SevenQuaternion), 2);
        }

        gTouchScreen.WritePoints(pTextWriter, gTouchScreen.GetPoints());

        gTouchScreen.makeTouchButton(pTextWriter, pGraphicsSystem, "決定", &TouchBtnSt, 1120, 180, 150, 60);
        gTouchScreen.makeTouchButton(pTextWriter, pGraphicsSystem, "戻る", &TouchBtnStDummy, 870, 180, 150, 60);

        if (menuSt == menu.Menustate_Ready)
        {
            if (DesidePosition == 1)
            {
                gTouchScreen.makeTouchButtonFrame(pGraphicsSystem, 1120, 180, 150, 60);
            }
            else
            {
                gTouchScreen.makeTouchButtonFrame(pGraphicsSystem, 870, 180, 150, 60);
            }
        }
        else if (menuSt == menu.Menustate_Set)
        {
            gTouchScreen.makeTouchButtonFrame(pGraphicsSystem, 870, 38, 400, 132);
        }
    }

    bool IsConnected() const NN_NOEXCEPT
    {
        return true;
    }
};

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
const size_t GraphicsMemorySize = 8 * 1024 * 1024;

void* NvAllocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, nn::util::align_up(size, alignment));
}

void NvFree(void* addr, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(addr);
}

void* NvReallocate(void* addr, size_t newSize, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}
#endif

} // namespace

extern "C" void nnMain()
{
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    nv::SetGraphicsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
    nv::SetGraphicsDevtoolsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
    nv::InitializeGraphics(std::malloc(GraphicsMemorySize), GraphicsMemorySize);
#endif

    nn::mem::StandardAllocator appAllocator;
    nn::Bit8* pAppMemory = new nn::Bit8[ApplicationHeapSize];
    appAllocator.Initialize(pAppMemory, ApplicationHeapSize);

    gNPad.Initialize();
    NpadActvMd = gNPad.ChangeActivationMode(0);

    gNpadSixAxisSns.Initialize();
    gSevenSixAxisSns.Initialize();

    isInitializeNpad = true;
    isInitializeSeven = true;

    NN_LOG("NpadSixAxisSensor is Initialized.\n");
    NN_LOG("ConsoleSixAxisSensor(New) is Initialized.\n");

    GraphicsSystem* pGraphicsSystem = new GraphicsSystem();
    pGraphicsSystem->Initialize(&appAllocator, FrameBufferWidth, FrameBufferHeight);

    EnableWindowMessage(pGraphicsSystem->GetNativeWindowHandle());

    FontSystem* pFontSystem = new FontSystem();
    pFontSystem->Initialize(&appAllocator, pGraphicsSystem);

    nn::gfx::util::DebugFontTextWriter& textWriter = pFontSystem->GetDebugFontTextWriter();

    nn::oe::Initialize();

    nn::oe::SetFocusHandlingMode(nn::oe::FocusHandlingMode_Notify);

    gApplet.Initialize();

    ConsoleSixAxisSensor* pConSixAxis = new ConsoleSixAxisSensor(frm::hid::NpadIds[0]);

    pConSixAxis->Initialize();

    while (NN_STATIC_CONDITION(true))
    {
        bool quits = false;

        switch (GetWindowMessage(pGraphicsSystem->GetNativeWindowHandle()))
        {
        case WindowMessage_Close:
            quits = true;
            break;
        default:
            break;
        }

        if (!pConSixAxis->IsConnected())
        {
            continue;
        }
        pConSixAxis->Update();

        if(pConSixAxis->CanPrint())
        {
            pConSixAxis->Print();
        }

        quits = pConSixAxis->Quit();

        pGraphicsSystem->BeginDraw();

        gLayout.Draw(pGraphicsSystem);

        pConSixAxis->OnOff();
        pConSixAxis->Draw(&textWriter, pGraphicsSystem);

        pFontSystem->Draw();

        pGraphicsSystem->EndDraw();

        pGraphicsSystem->Synchronize(nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / FrameRate));

        if (quits)
        {
            break;
        }
    }

    pFontSystem->Finalize();
    delete pFontSystem;

    pGraphicsSystem->Finalize();
    delete pGraphicsSystem;

    appAllocator.Free(pAppMemory);
    appAllocator.Finalize();

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    nv::FinalizeGraphics();
#endif

}
