﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <string>
#include <vector>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_NpadHandheld.h>
#include <nn/hid/hid_NpadSixAxisSensor.h>
#include <nn/hid/hid_SevenSixAxisSensor.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/hid/debug/hid_ConsoleSixAxisSensor.h>
#include <nn/hid/tmp/hid_ConsoleSixAxisSensor.h>
#include <nn/hid/system/hid_SixAxisSensorCalibration.h>
#include <nn/hid/system/hid_SixAxisSensorAccurateCalibration.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/TargetConfigs/build_Platform.h>

#include <nn/settings/system/settings_SixAxisSensor.h>
#include <nn/util/util_MathTypes.h>
#include <nn/util/util_Matrix.h>
#include <nn/util/util_MatrixApi.h>

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#include <nv/nv_MemoryManagement.h>
#endif

#include "../Common/ApplicationHeap.h"
#include "../Common/Color.h"
#include "../Common/FontSystem.h"
#include "../Common/GraphicsSystem.h"
#include "../Common/WindowMessage.h"

#include "INpadStyleSixAxisSensor.h"
#include "JoyDualSixAxisSensor.h"
#include "ConsoleSixAxisSensorSampler.h"
#include "CommandHandler.h"
#include "SixAxisSensorLogger.h"
#include "SixAxisSensorPointer.h"

namespace nn { namespace hid {

float GetSensorFusionErrorForDebug() NN_NOEXCEPT;

::nn::util::Float3 GetGyroBiasForDebug() NN_NOEXCEPT;

}}


namespace {

bool g_Quits = false;

SixAxisSensorLogger g_Logger;
ConsoleSixAxisSensorSampler g_ConsoleSixAxisSensorSampler;
CommandHandler g_CommandHandler;

const size_t LoggerThreadStackSize = 8192;
NN_OS_ALIGNAS_THREAD_STACK char  g_LoggerThreadStack[LoggerThreadStackSize];
::nn::os::ThreadType  g_LoggerThread;

const size_t SamplerThreadStackSize = 8192 * 6;
NN_OS_ALIGNAS_THREAD_STACK char  g_SamplerThreadStack[SamplerThreadStackSize];
::nn::os::ThreadType  g_SamplerThread;

const char* DumpFileName = "SixAxisSensorLogger.csv";

const size_t ApplicationHeapSize = 128 * 1024 * 1024;

const int FrameRate = 60;

//!< 6 軸センサーの入力状態を変換
void Convert(::nn::hid::SixAxisSensorState* pOutState,
             const ::nn::hid::SevenSixAxisSensorState& state) NN_NOEXCEPT;

template<typename T, std::size_t U>
std::size_t GetArrayLength(const T (&staticArray)[U])
{
    NN_UNUSED(staticArray);
    return U;
}

const nn::hid::NpadIdType NpadIds[] = { nn::hid::NpadId::No1 };
const int NpadIdCountMax = static_cast<int>(GetArrayLength(NpadIds));

const nn::util::Unorm8x4 ColorArray[]  = {Color::Red,
                                          Color::Orange,
                                          Color::Yellow,
                                          Color::Green};

const int StyleIndices[] = { nn::hid::NpadStyleJoyDual::Index };

const int StyleIndicesCountMax = static_cast<int>(GetArrayLength(StyleIndices));

/**
 * @brief       描画する座標系の識別子です。
 */
enum CoordinateId
{
    CoordinateId_AccelerometerXy = 0,
    CoordinateId_AccelerometerYz,
    CoordinateId_AccelerometerZx,
    CoordinateId_AngularVelocityXy,
    CoordinateId_AngularVelocityYz,
    CoordinateId_AngularVelocityZx,
    CoordinateId_AngleX,
    CoordinateId_AngleY,
    CoordinateId_AngleZ,
    CoordinateId_DirectionXy,
    CoordinateId_DirectionYz,
    CoordinateId_DirectionZx,
};

//!< 座標系の原点座標を取得します。
void GetCoordinateOrigin(nn::util::Float2* pOutOrigin, CoordinateId id)
{
    const float X = 350;
    const float Y = 150;
    const float DeltaX = 250;
    const float DeltaY = 230;

    pOutOrigin->x = X + static_cast<int>(id) / 3 * DeltaX;
    pOutOrigin->y = Y + (static_cast<int>(id) % 3) * DeltaY;
}

//!< 座標系の名前を取得します。
void GetCoordinateName(char* pOutAxis1,
                       char* pOutAxis2,
                       ::std::string* pOutTitle,
                       CoordinateId id)
{
    switch (id)
    {
    case CoordinateId_AccelerometerXy :
        *pOutTitle = "AccelerometerXy";
        *pOutAxis1 = 'x';
        *pOutAxis2 = 'y';
        break;
    case CoordinateId_AccelerometerYz :
        *pOutTitle = "AccelerometerYz";
        *pOutAxis1 = 'y';
        *pOutAxis2 = 'z';
        break;
    case CoordinateId_AccelerometerZx :
        *pOutTitle = "AccelerometerZx";
        *pOutAxis1 = 'z';
        *pOutAxis2 = 'x';
        break;
    case CoordinateId_AngularVelocityXy:
        *pOutTitle = "AngularVelocityXy";
        *pOutAxis1 = 'x';
        *pOutAxis2 = 'y';
        break;
    case CoordinateId_AngularVelocityYz:
        *pOutTitle = "AngularVelocityYz";
        *pOutAxis1 = 'y';
        *pOutAxis2 = 'z';
        break;
    case CoordinateId_AngularVelocityZx:
        *pOutTitle = "AngularVelocityZx";
        *pOutAxis1 = 'z';
        *pOutAxis2 = 'x';
        break;
    case CoordinateId_AngleX:
        *pOutTitle = "AngleX";
        *pOutAxis1 = ' ';
        *pOutAxis2 = ' ';
        break;
    case CoordinateId_AngleY:
        *pOutTitle = "AngleY";
        *pOutAxis1 = ' ';
        *pOutAxis2 = ' ';
        break;
    case CoordinateId_AngleZ:
        *pOutTitle = "AngleZ";
        *pOutAxis1 = ' ';
        *pOutAxis2 = ' ';
        break;
    case CoordinateId_DirectionXy:
        *pOutTitle = "DirectionXy";
        *pOutAxis1 = 'x';
        *pOutAxis2 = 'y';
        break;
    case CoordinateId_DirectionYz:
        *pOutTitle = "DirectionYz";
        *pOutAxis1 = 'y';
        *pOutAxis2 = 'z';
        break;
    case CoordinateId_DirectionZx:
        *pOutTitle = "DirectionZx";
        *pOutAxis1 = 'z';
        *pOutAxis2 = 'x';
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }
}

//!< 6軸センサーの状態を描画します。
void WriteSixAxisSensorState(nn::gfx::util::DebugFontTextWriter* pTextWriter,
                             const INpadStyleSixAxisSensor* const pSixAxisSensor,
                             const float& offsetX,
                             const float& offsetY,
                             const nn::util::Unorm8x4& Color,
                             const int& playerNumber) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    const auto state = pSixAxisSensor->GetSixAxisSensorState(0);
    const nn::util::Unorm8x4& textColor = state.attributes.Test<nn::hid::SixAxisSensorAttribute::IsConnected>()
                                        ? Color : Color::Gray;

    pTextWriter->SetTextColor(textColor);
    pTextWriter->SetScale(1, 1);
    pTextWriter->SetCursor(offsetX, offsetY);
    pTextWriter->Print("Player: %d", playerNumber);

    pTextWriter->SetTextColor(textColor);
    pTextWriter->SetScale(0.7f, 0.7f);

    float posY = offsetY + 15.0f;
    const float DeltaY = 15.0f;

    pTextWriter->SetCursor(offsetX + 5, posY);
    pTextWriter->Print("acceleration");
    posY += DeltaY;
    pTextWriter->SetCursor(offsetX + 10, posY);
    pTextWriter->Print("(%8.5f,%8.5f,%8.5f)", state.acceleration.x,
                                              state.acceleration.y,
                                              state.acceleration.z);

    posY += DeltaY;
    pTextWriter->SetCursor(offsetX + 5, posY);
    pTextWriter->Print("angularVelocity");
    posY += DeltaY;
    pTextWriter->SetCursor(offsetX + 10, posY);
    pTextWriter->Print("(%8.5f,%8.5f,%8.5f)", state.angularVelocity.x,
                                              state.angularVelocity.y,
                                              state.angularVelocity.z);

    posY += DeltaY;
    pTextWriter->SetCursor(offsetX + 5, posY);
    pTextWriter->Print("angle");
    posY += DeltaY;
    pTextWriter->SetCursor(offsetX + 10, posY);
    pTextWriter->Print("(%8.5f,%8.5f,%8.5f)", state.angle.x,
                                              state.angle.y,
                                              state.angle.z);

    posY += DeltaY;
    pTextWriter->SetCursor(offsetX + 5, posY);
    pTextWriter->Print("samplingNumber");
    posY += DeltaY;
    pTextWriter->SetCursor(offsetX + 10, posY);
    pTextWriter->Print("(%d)", state.samplingNumber);

    posY += DeltaY;
    pTextWriter->SetCursor(offsetX + 5, posY);
    pTextWriter->Print("AtRest : %s", pSixAxisSensor->IsAtRest(0) ? "Y" : "N");

    posY += DeltaY;
    pTextWriter->SetCursor(offsetX + 5, posY);
    if (pSixAxisSensor->IsMeasuringStarted(0))
    {
        pTextWriter->Print("Measuring...");
    }
    else
    {
        pTextWriter->Print("PER[%] : %3.2f", pSixAxisSensor->GetPacketErrorRate(0) * 100);
    }

}

//!< 本体 6 軸センサーの状態を描画します。
void WriteConsoleSixAxisSensorState(nn::gfx::util::DebugFontTextWriter* pTextWriter,
                                    const float& offsetX,
                                    const float& offsetY,
                                    const nn::util::Unorm8x4& Color,
                                    const ::nn::hid::system::SixAxisSensorAccurateUserCalibrationState& calState,
                                    nn::Result result) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    const nn::util::Unorm8x4& textColor = g_ConsoleSixAxisSensorSampler.IsSamplingStarted() ? Color
                                                                                            : Color::Gray;
    ::nn::hid::tmp::SixAxisSensorCountState state = {};
    g_ConsoleSixAxisSensorSampler.GetCountStates(&state, 1);

    pTextWriter->SetTextColor(textColor);
    pTextWriter->SetScale(1, 1);
    pTextWriter->SetCursor(offsetX, offsetY);
    pTextWriter->Print("ConsoleSixAxisSensor");

    pTextWriter->SetTextColor(textColor);
    pTextWriter->SetScale(0.7f, 0.7f);

    float posY = offsetY + 15.0f;
    const float DeltaY = 15.0f;

    posY += DeltaY;
    pTextWriter->SetCursor(offsetX + 5, posY);
    pTextWriter->Print("[ Calibration State ] %x: %f", calState.stage, calState.progress);

    posY += DeltaY;
    pTextWriter->SetCursor(offsetX + 5, posY);
    pTextWriter->Print("[ Calibration Result ] %s",
        result.IsSuccess() ? "ResultSuccess" :
        ::nn::hid::system::ResultSixAxisSensorNotSupported::Includes(result) ? "NotSupported" :
        ::nn::hid::system::ResultSixAxisSensorWriteFailure::Includes(result) ? "WriteFailure" :
        ::nn::hid::system::ResultSixAxisSensorAccurateInvalidTermination::Includes(result) ? "InvalidTermination" :
        "UNDEFINED"
    );



    {
        posY += DeltaY;
        pTextWriter->SetCursor(offsetX + 5, posY);
        pTextWriter->Print("[ API Based ] -------------");

        ::nn::hid::SixAxisSensorState state;
        ::nn::hid::SevenSixAxisSensorState sevenState;
        g_ConsoleSixAxisSensorSampler.GetStates(&sevenState, 1);
        Convert(&state, sevenState);

        posY += DeltaY;
        pTextWriter->SetCursor(offsetX + 5, posY);
        pTextWriter->Print("samplingNumber: %d", state.samplingNumber);

        posY += DeltaY;
        pTextWriter->SetCursor(offsetX + 5, posY);
        pTextWriter->Print("IsAtRest : %s", ::nn::hid::IsSevenSixAxisSensorAtRest() ? "Y" : "N");

        posY += DeltaY;
        pTextWriter->SetCursor(offsetX + 5, posY);
        pTextWriter->Print("SensorFusionError : %f", ::nn::hid::GetSensorFusionErrorForDebug());

        auto gyroBias = ::nn::hid::GetGyroBiasForDebug();
        posY += DeltaY;
        pTextWriter->SetCursor(offsetX + 5, posY);
        pTextWriter->Print("gyroBias : (%f, %f, %f)"
            , gyroBias.v[0]
            , gyroBias.v[1]
            , gyroBias.v[2]
        );

        posY += DeltaY;
        pTextWriter->SetCursor(offsetX + 5, posY);
        pTextWriter->Print("acceleration");
        posY += DeltaY;
        pTextWriter->SetCursor(offsetX + 10, posY);
        pTextWriter->Print("(%4.2f, %4.2f, %4.2f)", state.acceleration.x,
                                                    state.acceleration.y,
                                                    state.acceleration.z);

        posY += DeltaY;
        pTextWriter->SetCursor(offsetX + 5, posY);
        pTextWriter->Print("angularVelocity");
        posY += DeltaY;
        pTextWriter->SetCursor(offsetX + 10, posY);
        pTextWriter->Print("(%4.2f, %4.2f, %4.2f)", state.angularVelocity.x,
                                                    state.angularVelocity.y,
                                                    state.angularVelocity.z);

        posY += DeltaY;
        pTextWriter->SetCursor(offsetX + 5, posY);
        pTextWriter->Print("angle");
        posY += DeltaY;
        pTextWriter->SetCursor(offsetX + 10, posY);
        pTextWriter->Print("(%4.2f, %4.2f, %4.2f)", state.angle.x,
                                                    state.angle.y,
                                                    state.angle.z);

        posY += DeltaY;
        pTextWriter->SetCursor(offsetX + 5, posY);
        pTextWriter->Print("direction");
        posY += DeltaY;
        pTextWriter->SetCursor(offsetX + 10, posY);
        pTextWriter->Print("(%4.2f, %4.2f, %4.2f)", state.direction.x.x,
                                                    state.direction.x.y,
                                                    state.direction.x.z);
        pTextWriter->Print("(%4.2f, %4.2f, %4.2f)", state.direction.y.x,
                                                    state.direction.y.y,
                                                    state.direction.y.z);
        pTextWriter->Print("(%4.2f, %4.2f, %4.2f)", state.direction.z.x,
                                                    state.direction.z.y,
                                                    state.direction.z.z);

        ::nn::hid::debug::ConsoleSixAxisSensorCountState states[::nn::hid::debug::ConsoleSixAxisSensorCountStateCountMax] = {};

        // Note: IPC するので非常に重い。ロギング時は Disable 推奨
        int count = ::nn::hid::debug::GetConsoleSixAxisSensorCountStates(states, ::nn::hid::debug::ConsoleSixAxisSensorCountStateCountMax);
        NN_ASSERT(count >= 0);
        const auto pState = &states[0];

        posY += DeltaY;
        pTextWriter->SetCursor(offsetX + 5, posY);
        pTextWriter->Print("FACTORY COUNT");

        posY += DeltaY;
        pTextWriter->SetCursor(offsetX + 5, posY);
        pTextWriter->Print("count: %d, status: (%lld, %lld, %lld)"
            , count
            , pState->deltaTime
            , pState->samplingNumber
            , pState->tick
        );

        posY += DeltaY;
        pTextWriter->SetCursor(offsetX + 5, posY);
        pTextWriter->Print("acc: (%d, %d, %d)"
            , pState->acceleration.x
            , pState->acceleration.y
            , pState->acceleration.z
        );

        posY += DeltaY;
        pTextWriter->SetCursor(offsetX + 5, posY);
        pTextWriter->Print("gyro: (%d, %d, %d)"
            , pState->angularVelocity.x
            , pState->angularVelocity.y
            , pState->angularVelocity.z
        );
    }

}// NOLINT(impl/function_size)

//!< 6軸センサーの状態を座標系上に描画します。
void WriteSixAxisSensorStateFigure(nn::gfx::util::DebugFontTextWriter* pTextWriter,
                                   const nn::hid::SixAxisSensorState& state,
                                   const nn::util::Unorm8x4& Color,
                                   const int playerNumber) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    const nn::util::Unorm8x4& textColor = state.attributes.Test<nn::hid::SixAxisSensorAttribute::IsConnected>()
                                        ? Color : Color::Gray;

    pTextWriter->SetTextColor(textColor);
    pTextWriter->SetScale(1, 1);

    pTextWriter->SetTextColor(textColor);
    pTextWriter->SetScale(0.8f, 0.8f);

    const float Coefficient  = 50.0f;

    nn::util::Float2 coordinateOrigin;

    // Accelerometer
    GetCoordinateOrigin(&coordinateOrigin, CoordinateId_AccelerometerXy);
    pTextWriter->SetCursor(coordinateOrigin.x + state.acceleration.x * Coefficient,
                           coordinateOrigin.y - state.acceleration.y * Coefficient);
    pTextWriter->Print("%d", playerNumber);

    GetCoordinateOrigin(&coordinateOrigin, CoordinateId_AccelerometerYz);
    pTextWriter->SetCursor(coordinateOrigin.x + state.acceleration.y * Coefficient,
                           coordinateOrigin.y - state.acceleration.z * Coefficient);
    pTextWriter->Print("%d", playerNumber);

    GetCoordinateOrigin(&coordinateOrigin, CoordinateId_AccelerometerZx);
    pTextWriter->SetCursor(coordinateOrigin.x + state.acceleration.z * Coefficient,
                           coordinateOrigin.y - state.acceleration.x * Coefficient);
    pTextWriter->Print("%d", playerNumber);

    // AngularVelocity
    GetCoordinateOrigin(&coordinateOrigin, CoordinateId_AngularVelocityXy);
    pTextWriter->SetCursor(coordinateOrigin.x + state.angularVelocity.x * Coefficient,
                           coordinateOrigin.y - state.angularVelocity.y * Coefficient);
    pTextWriter->Print("%d", playerNumber);

    GetCoordinateOrigin(&coordinateOrigin, CoordinateId_AngularVelocityYz);
    pTextWriter->SetCursor(coordinateOrigin.x + state.angularVelocity.y * Coefficient,
                           coordinateOrigin.y - state.angularVelocity.z * Coefficient);
    pTextWriter->Print("%d", playerNumber);

    GetCoordinateOrigin(&coordinateOrigin, CoordinateId_AngularVelocityZx);
    pTextWriter->SetCursor(coordinateOrigin.x + state.angularVelocity.z * Coefficient,
                           coordinateOrigin.y - state.angularVelocity.x * Coefficient);
    pTextWriter->Print("%d", playerNumber);

    // Angle
    GetCoordinateOrigin(&coordinateOrigin, CoordinateId_AngleX);
    pTextWriter->SetCursor(coordinateOrigin.x + cos(2 * nn::util::FloatPi * state.angle.x) * Coefficient,
                           coordinateOrigin.y - sin(2 * nn::util::FloatPi * state.angle.x) * Coefficient);
    pTextWriter->Print("%d", playerNumber);

    GetCoordinateOrigin(&coordinateOrigin, CoordinateId_AngleY);
    pTextWriter->SetCursor(coordinateOrigin.x + cos(2 * nn::util::FloatPi * state.angle.y) * Coefficient,
                           coordinateOrigin.y - sin(2 * nn::util::FloatPi * state.angle.y) * Coefficient);
    pTextWriter->Print("%d", playerNumber);

    GetCoordinateOrigin(&coordinateOrigin, CoordinateId_AngleZ);
    pTextWriter->SetCursor(coordinateOrigin.x + cos(2 * nn::util::FloatPi * state.angle.z) * Coefficient,
                           coordinateOrigin.y - sin(2 * nn::util::FloatPi * state.angle.z) * Coefficient);
    pTextWriter->Print("%d", playerNumber);

    // Direction
    GetCoordinateOrigin(&coordinateOrigin, CoordinateId_DirectionXy);
    pTextWriter->SetCursor(coordinateOrigin.x + state.direction.x.x * Coefficient,
                           coordinateOrigin.y - state.direction.x.y * Coefficient);
    pTextWriter->Print("%dx", playerNumber);
    pTextWriter->SetCursor(coordinateOrigin.x + state.direction.y.x * Coefficient,
                           coordinateOrigin.y - state.direction.y.y * Coefficient);
    pTextWriter->Print("%dy", playerNumber);
    pTextWriter->SetCursor(coordinateOrigin.x + state.direction.z.x * Coefficient,
                           coordinateOrigin.y - state.direction.z.y * Coefficient);
    pTextWriter->Print("%dz", playerNumber);

    GetCoordinateOrigin(&coordinateOrigin, CoordinateId_DirectionYz);
    pTextWriter->SetCursor(coordinateOrigin.x + state.direction.x.y * Coefficient,
                           coordinateOrigin.y - state.direction.x.z * Coefficient);
    pTextWriter->Print("%dx", playerNumber);
    pTextWriter->SetCursor(coordinateOrigin.x + state.direction.y.y * Coefficient,
                           coordinateOrigin.y - state.direction.y.z * Coefficient);
    pTextWriter->Print("%dy", playerNumber);
    pTextWriter->SetCursor(coordinateOrigin.x + state.direction.z.y * Coefficient,
                           coordinateOrigin.y - state.direction.z.z * Coefficient);
    pTextWriter->Print("%dz", playerNumber);

    GetCoordinateOrigin(&coordinateOrigin, CoordinateId_DirectionZx);
    pTextWriter->SetCursor(coordinateOrigin.x + state.direction.x.z * Coefficient,
                           coordinateOrigin.y - state.direction.x.x * Coefficient);
    pTextWriter->Print("%dx", playerNumber);
    pTextWriter->SetCursor(coordinateOrigin.x + state.direction.y.z * Coefficient,
                           coordinateOrigin.y - state.direction.y.x * Coefficient);
    pTextWriter->Print("%dy", playerNumber);
    pTextWriter->SetCursor(coordinateOrigin.x + state.direction.z.z * Coefficient,
                           coordinateOrigin.y - state.direction.z.x * Coefficient);
    pTextWriter->Print("%dz", playerNumber);
}

//!< 座標系を描画します。
void WriteCoordinateAxes(nn::gfx::util::DebugFontTextWriter* pTextWriter,
    CoordinateId id) NN_NOEXCEPT
{
    const nn::util::Unorm8x4& textColor = Color::White;

    pTextWriter->SetTextColor(textColor);
    pTextWriter->SetScale(0.7f, 0.7f);

    const int NumberOfCharacters = 10;
    const int Interval = 10;
    nn::util::Float2 origin;
    char axis1;
    char axis2;
    ::std::string title;
    GetCoordinateName(&axis1, &axis2, &title, id);
    GetCoordinateOrigin(&origin, id);

    for (int i = 0; i < NumberOfCharacters; i++)
    {
        pTextWriter->SetCursor(origin.x + Interval * i, origin.y);
        pTextWriter->Print("-");
        pTextWriter->SetCursor(origin.x - Interval * i, origin.y);
        pTextWriter->Print("-");

        pTextWriter->SetCursor(origin.x, origin.y + Interval * i);
        pTextWriter->Print("|");
        pTextWriter->SetCursor(origin.x, origin.y - Interval * i);
        pTextWriter->Print("|");
    }
    pTextWriter->SetCursor(origin.x + NumberOfCharacters * Interval, origin.y);
    pTextWriter->Print(">%c", axis1);
    pTextWriter->SetCursor(origin.x, origin.y - NumberOfCharacters * Interval);
    pTextWriter->Print("^%c", axis2);
    pTextWriter->SetCursor(origin.x - NumberOfCharacters * Interval,
        origin.y - NumberOfCharacters * Interval - 1.0f);
    pTextWriter->Print("%s", title.c_str());
}

//!< 座標系を描画します。
void WriteCoordinateAxes(nn::gfx::util::DebugFontTextWriter* pTextWriter) NN_NOEXCEPT
{
    WriteCoordinateAxes(pTextWriter, CoordinateId_AccelerometerXy);
    WriteCoordinateAxes(pTextWriter, CoordinateId_AccelerometerYz);
    WriteCoordinateAxes(pTextWriter, CoordinateId_AccelerometerZx);
    WriteCoordinateAxes(pTextWriter, CoordinateId_AngularVelocityXy);
    WriteCoordinateAxes(pTextWriter, CoordinateId_AngularVelocityYz);
    WriteCoordinateAxes(pTextWriter, CoordinateId_AngularVelocityZx);
    WriteCoordinateAxes(pTextWriter, CoordinateId_AngleX);
    WriteCoordinateAxes(pTextWriter, CoordinateId_AngleY);
    WriteCoordinateAxes(pTextWriter, CoordinateId_AngleZ);
    WriteCoordinateAxes(pTextWriter, CoordinateId_DirectionXy);
    WriteCoordinateAxes(pTextWriter, CoordinateId_DirectionYz);
    WriteCoordinateAxes(pTextWriter, CoordinateId_DirectionZx);

    WriteCoordinateAxes(pTextWriter, CoordinateId_AccelerometerXy);
    WriteCoordinateAxes(pTextWriter, CoordinateId_AccelerometerYz);
    WriteCoordinateAxes(pTextWriter, CoordinateId_AccelerometerZx);
    WriteCoordinateAxes(pTextWriter, CoordinateId_AngularVelocityXy);
    WriteCoordinateAxes(pTextWriter, CoordinateId_AngularVelocityYz);
    WriteCoordinateAxes(pTextWriter, CoordinateId_AngularVelocityZx);
    WriteCoordinateAxes(pTextWriter, CoordinateId_AngleX);
    WriteCoordinateAxes(pTextWriter, CoordinateId_AngleY);
    WriteCoordinateAxes(pTextWriter, CoordinateId_AngleZ);
    WriteCoordinateAxes(pTextWriter, CoordinateId_DirectionXy);
    WriteCoordinateAxes(pTextWriter, CoordinateId_DirectionYz);
    WriteCoordinateAxes(pTextWriter, CoordinateId_DirectionZx);
}

//!< 6軸センサーの入力状態を出力します。
void PrintSixAxisSensorState(const nn::hid::SixAxisSensorState& state)
{
    NN_LOG("SamplingNumber:%lld, DeltaTime[us]:%lld\n", state.samplingNumber,
                                                        state.deltaTime.GetMicroSeconds());
    NN_LOG("IsConnected:%s\n",
        state.attributes.Test<::nn::hid::SixAxisSensorAttribute::IsConnected>() ? "True" : "False");
    NN_LOG("Acc: %f, %f, %f\n",         state.acceleration.x,
                                        state.acceleration.y,
                                        state.acceleration.z);
    NN_LOG("Gyro: %f, %f, %f\n",        state.angularVelocity.x,
                                        state.angularVelocity.y,
                                        state.angularVelocity.z);
    NN_LOG("Angle: %f, %f, %f\n",       state.angle.x,
                                        state.angle.y,
                                        state.angle.z);
    NN_LOG("Direction.x: %f, %f, %f\n", state.direction.x.x,
                                        state.direction.x.y,
                                        state.direction.x.z);
    NN_LOG("Direction.y: %f, %f, %f\n", state.direction.y.x,
                                        state.direction.y.y,
                                        state.direction.y.z);
    NN_LOG("Direction.z: %f, %f, %f\n", state.direction.z.x,
                                        state.direction.z.y,
                                        state.direction.z.z);
    NN_LOG("\n");
}

//!< アプリの終了判定をします。
bool IsQuitRequired(const nn::hid::NpadButtonSet& buttons)
{
    return (buttons.Test<nn::hid::NpadButton::Plus>() ||
            buttons.Test<nn::hid::NpadButton::Minus>());
}

//!< NAND 上に書かれた本体 6 軸センサーのキャリブレーション結果をダンプします
void DumpCalibrationParametersOnNand() NN_NOEXCEPT
{
    ::nn::settings::system::ConsoleSixAxisSensorAccelerationBias
        consoleSixAxisSensorAccelerationBias;
    ::nn::settings::system::ConsoleSixAxisSensorAngularVelocityBias
        consoleSixAxisSensorAngularVelocityBias;
    ::nn::settings::system::ConsoleSixAxisSensorAccelerationGain
        consoleSixAxisSensorAccelerationGain;
    ::nn::settings::system::ConsoleSixAxisSensorAngularVelocityGain
        consoleSixAxisSensorAngularVelocityGain;
    ::nn::settings::system::ConsoleSixAxisSensorAngularVelocityTimeBias
        consoleSixAxisSensorAngularVelocityTimeBias;
    ::nn::settings::system::ConsoleSixAxisSensorAngularAcceleration
        consoleSixAxisSensorAngularAcceleration;

    ::nn::settings::system::GetConsoleSixAxisSensorAccelerationBias(
        &consoleSixAxisSensorAccelerationBias);
    ::nn::settings::system::GetConsoleSixAxisSensorAngularVelocityBias(
        &consoleSixAxisSensorAngularVelocityBias);
    ::nn::settings::system::GetConsoleSixAxisSensorAccelerationGain(
        &consoleSixAxisSensorAccelerationGain);
    ::nn::settings::system::GetConsoleSixAxisSensorAngularVelocityGain(
        &consoleSixAxisSensorAngularVelocityGain);
    ::nn::settings::system::GetConsoleSixAxisSensorAngularVelocityTimeBias(
        &consoleSixAxisSensorAngularVelocityTimeBias);
    ::nn::settings::system::GetConsoleSixAxisSensorAngularAcceleration(
        &consoleSixAxisSensorAngularAcceleration);

    NN_LOG(
        "ConsoleSixAxisSensor Calibration Parameter\n"
        "consoleSixAxisSensorAccelerationBias        : (%f, %f, %f)\n"
        "consoleSixAxisSensorAngularVelocityBias     : (%f, %f, %f)\n"
        "consoleSixAxisSensorAccelerationGain        : (%f, %f, %f, %f, %f, %f, %f, %f, %f)\n"
        "consoleSixAxisSensorAngularVelocityGain     : (%f, %f, %f, %f, %f, %f, %f, %f, %f)\n"
        "consoleSixAxisSensorAngularVelocityTimeBias : (%f, %f, %f)\n"
        "consoleSixAxisSensorAngularAcceleration     : (%f, %f, %f, %f, %f, %f, %f, %f, %f)\n"
        // consoleSixAxisSensorAccelerationBias
        , consoleSixAxisSensorAccelerationBias.x
        , consoleSixAxisSensorAccelerationBias.y
        , consoleSixAxisSensorAccelerationBias.z
        // consoleSixAxisSensorAngularVelocityBias
        , consoleSixAxisSensorAngularVelocityBias.x
        , consoleSixAxisSensorAngularVelocityBias.y
        , consoleSixAxisSensorAngularVelocityBias.z
        // consoleSixAxisSensorAccelerationGain
        , consoleSixAxisSensorAccelerationGain.xx
        , consoleSixAxisSensorAccelerationGain.xy
        , consoleSixAxisSensorAccelerationGain.xz
        , consoleSixAxisSensorAccelerationGain.yx
        , consoleSixAxisSensorAccelerationGain.yy
        , consoleSixAxisSensorAccelerationGain.yz
        , consoleSixAxisSensorAccelerationGain.zx
        , consoleSixAxisSensorAccelerationGain.zy
        , consoleSixAxisSensorAccelerationGain.zz
        // consoleSixAxisSensorAngularVelocityGain
        , consoleSixAxisSensorAngularVelocityGain.xx
        , consoleSixAxisSensorAngularVelocityGain.xy
        , consoleSixAxisSensorAngularVelocityGain.xz
        , consoleSixAxisSensorAngularVelocityGain.yx
        , consoleSixAxisSensorAngularVelocityGain.yy
        , consoleSixAxisSensorAngularVelocityGain.yz
        , consoleSixAxisSensorAngularVelocityGain.zx
        , consoleSixAxisSensorAngularVelocityGain.zy
        , consoleSixAxisSensorAngularVelocityGain.zz
        // consoleSixAxisSensorAngularVelocityTimeBias
        , consoleSixAxisSensorAngularVelocityTimeBias.x
        , consoleSixAxisSensorAngularVelocityTimeBias.y
        , consoleSixAxisSensorAngularVelocityTimeBias.z
        // consoleSixAxisSensorAngularAcceleration
        , consoleSixAxisSensorAngularAcceleration.xx
        , consoleSixAxisSensorAngularAcceleration.xy
        , consoleSixAxisSensorAngularAcceleration.xz
        , consoleSixAxisSensorAngularAcceleration.yx
        , consoleSixAxisSensorAngularAcceleration.yy
        , consoleSixAxisSensorAngularAcceleration.yz
        , consoleSixAxisSensorAngularAcceleration.zx
        , consoleSixAxisSensorAngularAcceleration.zy
        , consoleSixAxisSensorAngularAcceleration.zz
    );
}

//!< メインのループ処理を実行します。
void RunLoop(nn::gfx::util::DebugFontTextWriter* pTextWriter,
             GraphicsSystem* pGraphicsSystem,
             FontSystem* pFontSystem,
             std::vector<INpadStyleSixAxisSensor*>* pNpadStyleSixAxisSensors)
{
    while (NN_STATIC_CONDITION(true))
    {
        switch (GetWindowMessage(pGraphicsSystem->GetNativeWindowHandle()))
        {
        case WindowMessage_Close:
            g_Quits = true;
            break;
        default:
            break;
        }

        WriteCoordinateAxes(pTextWriter);

        for (std::vector<INpadStyleSixAxisSensor*>::iterator it = pNpadStyleSixAxisSensors->begin();
            it != pNpadStyleSixAxisSensors->end();
            ++it)
        {
            if (!(*it)->IsConnected())
            {
                continue;
            }

            // 状態更新
            (*it)->Update();

            nn::hid::NpadButtonSet triggerButtons = (*it)->GetTriggerButtons();

            g_CommandHandler.Update(triggerButtons);

            if (IsQuitRequired(triggerButtons))
            {
                g_Quits = true;
            }

            int count = (*it)->GetSixAxisSensorHandleCount();

            for (int i = 0; i < count; i++)
            {
                SixAxisSensorPointer* pPointer = (*it)->GetSixAxisSensorPointer(i);
                nn::hid::SixAxisSensorState state = (*it)->GetSixAxisSensorState(i);
                nn::hid::NpadIdType id = (*it)->GetNpadIdType();


                // 各操作形態で反映されるコマンド
                if (triggerButtons.Test<nn::hid::NpadButton::A>() ||
                    triggerButtons.Test<nn::hid::NpadButton::Right>())
                {
                    PrintSixAxisSensorState(state);
                    pPointer->Reset();
                    (*it)->SetGyroscopeZeroDriftMode(i, nn::hid::GyroscopeZeroDriftMode_Tight);
                    ::nn::hid::system::StartSixAxisSensorAccurateUserCalibration(
                        ::nn::hid::system::UniqueSixAxisSensorHandle());
                }

                if (triggerButtons.Test<nn::hid::NpadButton::B>() ||
                    triggerButtons.Test<nn::hid::NpadButton::Down>())
                {
                    (*it)->SetGyroscopeZeroDriftMode(i, nn::hid::GyroscopeZeroDriftMode_Standard);
                    ::nn::hid::system::CancelSixAxisSensorAccurateUserCalibration(
                        ::nn::hid::system::UniqueSixAxisSensorHandle());
                }

                if (triggerButtons.Test<nn::hid::NpadButton::X>() ||
                    triggerButtons.Test<nn::hid::NpadButton::Up>())
                {
                    (*it)->StartMeasuringPacketErrorRate(i);
                }
                if (triggerButtons.Test<nn::hid::NpadButton::Y>() ||
                    triggerButtons.Test<nn::hid::NpadButton::Left>())
                {
                    (*it)->StopMeasuringPacketErrorRate(i);
                }

                if (triggerButtons.Test<nn::hid::NpadButton::StickR>() ||
                    triggerButtons.Test<nn::hid::NpadButton::StickL>())
                {
                    DumpCalibrationParametersOnNand();
                }

                int idx = static_cast<uint32_t>(id);
                if (idx >= NpadIdCountMax)
                {
                    idx = 0;
                }

                pTextWriter->SetScale(3.f, 3.f);
                pTextWriter->SetTextColor(ColorArray[0]);
                pTextWriter->SetCursor(0.f, 0.f);
                pTextWriter->Print("Test: %d", g_CommandHandler.GetTestId());

                if (i == 0)
                {
                    WriteSixAxisSensorState(pTextWriter,
                                            (*it),
                                            25, 15 + 175 * static_cast<float>(idx),
                                            ColorArray[idx],
                                            idx + 1);
                }

                WriteSixAxisSensorStateFigure(pTextWriter,
                    state,
                    ColorArray[idx],
                    idx + 1);

                pTextWriter->SetScale(3.f, 3.f);
                ::nn::util::Vector3f cursor = pPointer->GetCursor();
                pTextWriter->SetTextColor(ColorArray[idx]);
                pTextWriter->SetCursor(cursor.GetX(), cursor.GetY());
                pTextWriter->Print("(-+-)");
            }
        }

        {
            // API based
            ::nn::hid::SevenSixAxisSensorState sevenStates[::nn::hid::SevenSixAxisSensorStateCountMax] = {};
            g_ConsoleSixAxisSensorSampler.GetStates(sevenStates,
                                                    ::nn::hid::SevenSixAxisSensorStateCountMax);
            ::nn::hid::SixAxisSensorState state;
            Convert(&state, sevenStates[0]);
            WriteSixAxisSensorStateFigure(pTextWriter,
                                          state,
                                          ColorArray[2],
                                          2 + 1);
        }
        ::nn::hid::system::SixAxisSensorAccurateUserCalibrationState calState;
        auto result = ::nn::hid::system::GetSixAxisSensorAccurateUserCalibrationState(
            &calState,
            ::nn::hid::system::UniqueSixAxisSensorHandle());

        WriteConsoleSixAxisSensorState(pTextWriter,
                                       25,
                                       15 + 175,
                                       ColorArray[1],
                                       calState,
                                       result);

        pGraphicsSystem->BeginDraw();
        pFontSystem->Draw();
        pGraphicsSystem->EndDraw();

        pGraphicsSystem->Synchronize(
            nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / FrameRate));

        if (g_Quits)
        {
            break;
        }
    }
} // NOLINT(impl/function_size)

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
const size_t GraphicsMemorySize = 8 * 1024 * 1024;

void* NvAllocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, size);
}

void NvFree(void* addr, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(addr);
}

void* NvReallocate(void* addr, size_t newSize, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}
#endif


void SamplerFunction(void *arg)
{
    NN_UNUSED(arg);
    NN_LOG("Sampling Started\n");

    int64_t recordedSamplingNumber = 0;

    ::nn::os::MultiWaitType multiWait;
    ::nn::os::TimerEventType samplingEvent;
    ::nn::os::MultiWaitHolderType samplingEventHolder;

    ::nn::os::InitializeMultiWait(&multiWait);

    // サンプリングイベントを登録
    ::nn::os::InitializeTimerEvent(&samplingEvent,
                                   ::nn::os::EventClearMode_ManualClear);
    ::nn::os::InitializeMultiWaitHolder(&samplingEventHolder,
                                        &samplingEvent);
    ::nn::os::LinkMultiWaitHolder(&multiWait,
                                  &samplingEventHolder);

    const int64_t IntervalMilliSecondsForSampling = 10;

    ::nn::os::StartPeriodicTimerEvent(&samplingEvent,
                                      ::nn::TimeSpan::FromMilliSeconds(IntervalMilliSecondsForSampling),
                                      ::nn::TimeSpan::FromMilliSeconds(IntervalMilliSecondsForSampling));

    bool runs = true;

    while(NN_STATIC_CONDITION(runs))
    {
        auto pHolder = nn::os::WaitAny(&multiWait);

        // サンプリングイベントがシグナルされた
        if(pHolder == &samplingEventHolder)
        {
            ::nn::os::ClearTimerEvent(&samplingEvent);

            ::nn::hid::tmp::SixAxisSensorCountState states[::nn::hid::ConsoleSixAxisSensorStateCountMax] = {};
            g_ConsoleSixAxisSensorSampler.GetCountStates(states,
                                                         ::nn::hid::ConsoleSixAxisSensorStateCountMax);

            for (int i = ::nn::hid::ConsoleSixAxisSensorStateCountMax - 1; i >= 0; i--)
            {
                if (recordedSamplingNumber < states[i].samplingNumber)
                {
                    if (recordedSamplingNumber + 1 != states[i].samplingNumber)
                    {
                        NN_LOG("[WARN]: %d packets are lost\n"
                            , states[i].samplingNumber - recordedSamplingNumber - 1);
                    }
                    g_Logger.AppendLine(states[i], g_CommandHandler.GetTestId());
                    recordedSamplingNumber = states[i].samplingNumber;
                }
            }
        }

        if (g_Quits)
        {
            break;
        }
    }
}

void LoggerFunction(void *arg)
{
    NN_UNUSED(arg);
    NN_LOG("Logging Started\n");

    ::nn::os::MultiWaitType multiWait;
    ::nn::os::TimerEventType loggingEvent;
    ::nn::os::MultiWaitHolderType loggingEventHolder;

    ::nn::os::InitializeMultiWait(&multiWait);

    // 書き込みイベントを登録
    ::nn::os::InitializeTimerEvent(&loggingEvent,
                                   ::nn::os::EventClearMode_ManualClear);
    ::nn::os::InitializeMultiWaitHolder(&loggingEventHolder,
                                        &loggingEvent);
    ::nn::os::LinkMultiWaitHolder(&multiWait,
                                  &loggingEventHolder);

    const int64_t IntervalMilliSecondsForLogging = 50;

    ::nn::os::StartPeriodicTimerEvent(&loggingEvent,
                                      ::nn::TimeSpan::FromMilliSeconds(IntervalMilliSecondsForLogging),
                                      ::nn::TimeSpan::FromMilliSeconds(IntervalMilliSecondsForLogging));

    bool runs = true;

    while(NN_STATIC_CONDITION(runs))
    {
        auto pHolder = nn::os::WaitAny(&multiWait);

        // 書き込みイベントがシグナルされた
        if(pHolder == &loggingEventHolder)
        {
            ::nn::os::ClearTimerEvent(&loggingEvent);

            g_Logger.Write();
        }

        if (g_Quits)
        {
            break;
        }
    }
}

//!< 姿勢回転の変換
void Convert(::nn::hid::DirectionState* pOutDirectionState,
             const ::nn::util::Vector4fType& quaternion) NN_NOEXCEPT
{
    // TORIAEZU
    nn::util::Matrix4x3f mat;
    ::nn::util::MatrixFromQuaternion(&mat, quaternion);

    pOutDirectionState->x.x = mat.GetAxisX().GetX();
    pOutDirectionState->x.y = mat.GetAxisX().GetY();
    pOutDirectionState->x.z = mat.GetAxisX().GetZ();
    pOutDirectionState->y.x = mat.GetAxisY().GetX();
    pOutDirectionState->y.y = mat.GetAxisY().GetY();
    pOutDirectionState->y.z = mat.GetAxisY().GetZ();
    pOutDirectionState->z.x = mat.GetAxisZ().GetX();
    pOutDirectionState->z.y = mat.GetAxisZ().GetY();
    pOutDirectionState->z.z = mat.GetAxisZ().GetZ();
}

//!< 6 軸センサーの入力状態を変換
void Convert(::nn::hid::SixAxisSensorState* pOutState,
             const ::nn::hid::SevenSixAxisSensorState& state) NN_NOEXCEPT
{
    pOutState->acceleration    = state.acceleration;
    pOutState->angularVelocity = state.angularVelocity;
    pOutState->samplingNumber  = state.samplingNumber;
    pOutState->deltaTime       = state.timeStamp;
    pOutState->attributes.Set(); // TORIAEZU
    Convert(&pOutState->direction, state.rotation);
}

} // namespace

extern "C" void nnMain()
{
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    nv::SetGraphicsAllocator(NvAllocate, NvFree,NvReallocate,NULL);
    nv::SetGraphicsDevtoolsAllocator(NvAllocate, NvFree,NvReallocate,NULL);
    nv::InitializeGraphics(std::malloc(GraphicsMemorySize), GraphicsMemorySize);
#endif

    std::vector<INpadStyleSixAxisSensor*> npadStyleSixAxisSensors; //!< 管理対象の操作形態を格納するコンテナーです。
    NN_LOG("ConsoleSixAxisSensor Sample Start.\n");

    for(int i = 0; i < NpadIdCountMax; i++)
    {
        // Add respective NpadStyles to the managed list
        npadStyleSixAxisSensors.push_back(reinterpret_cast<INpadStyleSixAxisSensor*>(new JoyDualSixAxisSensor(NpadIds[i])));
    }

    nn::hid::InitializeNpad();

    g_CommandHandler.SetConsoleSixAxisSensorSampler(&g_ConsoleSixAxisSensorSampler);
    g_CommandHandler.SetSixAxisSensorLogger(&g_Logger);

    g_Logger.Initialize(DumpFileName);
    g_ConsoleSixAxisSensorSampler.Initialize();

    // ワークバッファの設定
    nn::mem::StandardAllocator appAllocator;
    nn::Bit8* pAppMemory = new nn::Bit8[ApplicationHeapSize];
    appAllocator.Initialize(pAppMemory, ApplicationHeapSize);

    //使用する操作形態を設定
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleJoyDual::Mask);

    nn::hid::SetSupportedNpadIdType(NpadIds, NpadIdCountMax);

    NN_LOG("If you push any button, button state log will appear on the console.\n");
    NN_LOG("Push (+) and (-) Button to shutdown this application.\n");

    // 管理対象の操作形態に関する初期化
    for(std::vector<INpadStyleSixAxisSensor*>::iterator it = npadStyleSixAxisSensors.begin();
        it != npadStyleSixAxisSensors.end();
        ++it)
    {
        (*it)->Initialize();
    }

    ApplicationHeap applicationHeap;
    applicationHeap.Initialize(ApplicationHeapSize);

    GraphicsSystem* pGraphicsSystem = new GraphicsSystem();
    pGraphicsSystem->SetApplicationHeap(&applicationHeap);
    pGraphicsSystem->Initialize();

    EnableWindowMessage(pGraphicsSystem->GetNativeWindowHandle());

    FontSystem* pFontSystem = new FontSystem();
    pFontSystem->SetApplicationHeap(&applicationHeap);
    pFontSystem->SetGraphicsSystem(pGraphicsSystem);
    pFontSystem->Initialize();

    nn::gfx::util::DebugFontTextWriter& textWriter =
        pFontSystem->GetDebugFontTextWriter();

    // スレッドを生成
    auto resultLogger = nn::os::CreateThread(&g_LoggerThread,
                                             LoggerFunction,
                                             NULL,
                                             g_LoggerThreadStack,
                                             LoggerThreadStackSize,
                                             nn::os::HighestThreadPriority);
    NN_ASSERT(resultLogger.IsSuccess(), "Cannot create LoggerThread.");

    auto resultSampler = nn::os::CreateThread(&g_SamplerThread,
                                              SamplerFunction,
                                              NULL,
                                              g_SamplerThreadStack,
                                              SamplerThreadStackSize,
                                              nn::os::HighestThreadPriority);
    NN_ASSERT(resultSampler.IsSuccess(), "Cannot create SamplerThread.");

    // スレッドを開始
    nn::os::StartThread(&g_LoggerThread);
    nn::os::StartThread(&g_SamplerThread);

    // メインループ
    RunLoop(&textWriter,
            pGraphicsSystem,
            pFontSystem,
            &npadStyleSixAxisSensors);

    g_ConsoleSixAxisSensorSampler.StopSampling();

    nn::os::WaitThread(&g_LoggerThread);
    nn::os::WaitThread(&g_SamplerThread);

    nn::os::DestroyThread(&g_LoggerThread);
    nn::os::DestroyThread(&g_SamplerThread);

    pFontSystem->Finalize();
    delete pFontSystem;

    pGraphicsSystem->Finalize();
    delete pGraphicsSystem;

    applicationHeap.Finalize();

    g_Logger.Finalize();
}
