﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_Tick.h>
#include <nn/hid/hid_NpadCommonTypes.h>
#include <nn/hid/hid_NpadSixAxisSensor.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/hid/debug/hid_SixAxisSensor.h>

#include "SixAxisSensorPointer.h"

// #define DEBUG

/**
 * @brief       Packet Error Rate 測定に用いる構造体の定義です。
 */
struct MeasuringState
{
    bool isStarted;
    int64_t startedSamplingNumber;
    float packetErrorRate;
};

/**
 * @brief       操作形態ごとの処理を記述したインタフェースです。
 */
class INpadStyleSixAxisSensor
{
    NN_DISALLOW_COPY(INpadStyleSixAxisSensor);
    NN_DISALLOW_MOVE(INpadStyleSixAxisSensor);

protected:
    nn::hid::SixAxisSensorHandle m_SixAxisSensorHandles[nn::hid::NpadSixAxisSensorHandleCountMax];
    nn::hid::SixAxisSensorState m_SixAxisSensorStates[nn::hid::NpadSixAxisSensorHandleCountMax][nn::hid::SixAxisSensorStateCountMax];
    SixAxisSensorPointer m_Pointers[nn::hid::NpadSixAxisSensorHandleCountMax];
    int m_SixAxisSensorHandleCount;

    nn::hid::NpadStyleSet m_Style;
    nn::hid::NpadIdType m_Id;
    nn::hid::NpadButtonSet m_Buttons[2];

    MeasuringState m_MeasuringStates[nn::hid::NpadSixAxisSensorHandleCountMax];
    nn::os::Tick m_StartedTicks[nn::hid::NpadSixAxisSensorHandleCountMax];

public:

    INpadStyleSixAxisSensor(nn::hid::NpadIdType id, nn::hid::NpadStyleSet style) NN_NOEXCEPT
        : m_Pointers()
        , m_SixAxisSensorHandleCount(0)
        , m_Style(style)
        , m_Id(id)
    {
        for (int i = 0; i < nn::hid::NpadSixAxisSensorHandleCountMax; i++)
        {
            m_SixAxisSensorHandles[i] = nn::hid::SixAxisSensorHandle();
            for (auto& state : m_SixAxisSensorStates[i])
            {
                state = nn::hid::SixAxisSensorState();
            }
            m_MeasuringStates[i] = MeasuringState();
            m_MeasuringStates[i].isStarted = false;
        }

        for (auto& button : m_Buttons)
        {
            button = nn::hid::NpadButtonSet();
        }
    };

    virtual ~INpadStyleSixAxisSensor() NN_NOEXCEPT { /* 何もしない */ }

    virtual void Update() NN_NOEXCEPT = 0;

    virtual bool IsConnected() const NN_NOEXCEPT = 0;

    void Initialize() NN_NOEXCEPT
    {
        m_SixAxisSensorHandleCount = nn::hid::GetSixAxisSensorHandles(m_SixAxisSensorHandles,
                                                                      nn::hid::NpadSixAxisSensorHandleCountMax,
                                                                      m_Id,
                                                                      m_Style);
        NN_ASSERT_GREATER_EQUAL(nn::hid::NpadSixAxisSensorHandleCountMax, m_SixAxisSensorHandleCount);

        for (int i = 0; i < m_SixAxisSensorHandleCount; i++)
        {
            nn::hid::StartSixAxisSensor(m_SixAxisSensorHandles[i]);
#ifdef DEBUG
            if (m_Id == ::nn::hid::NpadId::No1)
            {
                // 限度品設定
                nn::hid::debug::SetShiftAccelerometerCalibrationValue(m_SixAxisSensorHandles[i], 0.13f, 1.06f);
                nn::hid::debug::SetShiftGyroscopeCalibrationValue(m_SixAxisSensorHandles[i], 20.f / 360.f, 1.06f);
                // デフォルト設定
                // nn::hid::debug::SetShiftAccelerometerCalibrationValue(m_SixAxisSensorHandles[i], 0.f, 1.f);
                // nn::hid::debug::SetShiftGyroscopeCalibrationValue(m_SixAxisSensorHandles[i], 0.f, 1.f);
            }
#endif
        }
    }

    void UpdateSixAxisSensor() NN_NOEXCEPT
    {
        for (int i = 0; i < m_SixAxisSensorHandleCount; i++)
        {
            nn::hid::GetSixAxisSensorStates(m_SixAxisSensorStates[i], nn::hid::SixAxisSensorStateCountMax, m_SixAxisSensorHandles[i]);
            m_Pointers[i].Update(m_SixAxisSensorStates[i][0].direction);
        }
    }

    nn::hid::NpadButtonSet GetTriggerButtons() const NN_NOEXCEPT
    {
        return (m_Buttons[0] ^ m_Buttons[1]) & m_Buttons[0];
    }

    nn::hid::NpadIdType GetNpadIdType() const NN_NOEXCEPT
    {
        return m_Id;
    }

    nn::hid::NpadStyleSet GetNpadStyleSet() const NN_NOEXCEPT
    {
        return m_Style;
    }

    int GetSixAxisSensorHandleCount() const NN_NOEXCEPT
    {
        return m_SixAxisSensorHandleCount;
    }

    bool IsAtRest(int count) const NN_NOEXCEPT
    {
        return nn::hid::IsSixAxisSensorAtRest(m_SixAxisSensorHandles[count]);
    }

    void SetGyroscopeZeroDriftMode(int count,
                                   const nn::hid::GyroscopeZeroDriftMode& mode) const NN_NOEXCEPT
    {
        return nn::hid::SetGyroscopeZeroDriftMode(m_SixAxisSensorHandles[count],
                                                  mode);
    }

    nn::hid::SixAxisSensorState GetSixAxisSensorState(int count) const NN_NOEXCEPT
    {
        return m_SixAxisSensorStates[count][0];
    }

    SixAxisSensorPointer* GetSixAxisSensorPointer(int count) NN_NOEXCEPT
    {
        return &m_Pointers[count];
    }

    float GetPacketErrorRate(int count) const NN_NOEXCEPT
    {
        return m_MeasuringStates[count].packetErrorRate;
    }

    bool IsMeasuringStarted(int count) const NN_NOEXCEPT
    {
        return m_MeasuringStates[count].isStarted;
    }

    void StartMeasuringPacketErrorRate(int count) NN_NOEXCEPT
    {
        auto& measuringState = m_MeasuringStates[count];
        measuringState.isStarted = true;

        m_StartedTicks[count] = nn::os::GetSystemTick();
        nn::hid::SixAxisSensorState state = {};
        nn::hid::GetSixAxisSensorState(&state, m_SixAxisSensorHandles[count]);
        measuringState.startedSamplingNumber = state.samplingNumber;
        NN_LOG("Measuring started\n");
    }

    void StopMeasuringPacketErrorRate(int count) NN_NOEXCEPT
    {
        auto& measuringState = m_MeasuringStates[count];
        if (!measuringState.isStarted)
        {
            NN_LOG("Measuring hasn't started yet\n");
            return;
        }

        const int64_t DeltaTimeMilliSeconds = (nn::os::GetSystemTick() - m_StartedTicks[count]).ToTimeSpan().GetMilliSeconds();
        const float ReciprocalExpectedReceivedPackets = 5.0f / DeltaTimeMilliSeconds;

        nn::hid::SixAxisSensorState state = {};
        nn::hid::GetSixAxisSensorState(&state, m_SixAxisSensorHandles[count]);

        const int64_t ReceivedPackets = state.samplingNumber - measuringState.startedSamplingNumber;
        measuringState.packetErrorRate = 1.0f - ReceivedPackets * ReciprocalExpectedReceivedPackets;

        // Clamp
        if (measuringState.packetErrorRate < 0.0f)
        {
            measuringState.packetErrorRate = 0.0f;
        }
        else if(measuringState.packetErrorRate > 1.0f)
        {
            measuringState.packetErrorRate = 1.0f;
        }

        measuringState.isStarted = false;

        NN_LOG("\t[%d] Measured = %d[ms], ReceivedPackets = %lld, PacketErrorRate = %3.2f\n"
            , count
            , DeltaTimeMilliSeconds
            , ReceivedPackets
            , measuringState.packetErrorRate);
        NN_LOG("Measuring done.\n");
    }
};
