﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_TimeSpan.h>
#include <nn/hid.h>
#include <nn/hid/debug/hid_TouchScreen.h>
#include <nn/os.h>
#include <nnt/nntest.h>

#include "../Common/testHid_TouchScreen.h"

template<typename T>
class TouchStateCountSuite : public ::testing::Test
{
protected:
    static ::nn::hid::TouchScreenState<T::Value>
        s_States[::nn::hid::TouchScreenStateCountMax];

    //!< TouchScreen の入力状態が入力無し状態であることを期待します。
    static void ExpectDefaultTouchScreenState(
        const ::nn::hid::TouchScreenState<T::Value>& state) NN_NOEXCEPT;
};

// TouchScreenState::touches の数が以下の場合についてそれぞれテストを実行する。
typedef ::testing::Types<
    ::nnt::hid::TouchStateCountType<1>,
    ::nnt::hid::TouchStateCountType<2>,
    ::nnt::hid::TouchStateCountType<3>,
    ::nnt::hid::TouchStateCountType<4>,
    ::nnt::hid::TouchStateCountType<5>,
    ::nnt::hid::TouchStateCountType<6>,
    ::nnt::hid::TouchStateCountType<7>,
    ::nnt::hid::TouchStateCountType<8>,
    ::nnt::hid::TouchStateCountType<9>,
    ::nnt::hid::TouchStateCountType<10>,
    ::nnt::hid::TouchStateCountType<11>,
    ::nnt::hid::TouchStateCountType<12>,
    ::nnt::hid::TouchStateCountType<13>,
    ::nnt::hid::TouchStateCountType<14>,
    ::nnt::hid::TouchStateCountType<15>,
    ::nnt::hid::TouchStateCountType<16>
    > TouchStateCountTypes;

TYPED_TEST_CASE(TouchStateCountSuite, TouchStateCountTypes);

//!< 任意のタッチ数において入力状態が正しく読み出されるか
TYPED_TEST(TouchStateCountSuite, ReadingStateTest1)
{
    ::nn::hid::InitializeTouchScreen();

    ::nn::hid::GetTouchScreenState(&TestFixture::s_States[0]);

    {
        SCOPED_TRACE("");
        TestFixture::ExpectDefaultTouchScreenState(TestFixture::s_States[0]);
    }

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::TouchScreenStateCountMax / 2 *
            ::nnt::hid::GetTouchScreenSamplingInterval().GetMilliSeconds()));

    int count = ::nn::hid::GetTouchScreenStates(
        TestFixture::s_States,
        ::nn::hid::TouchScreenStateCountMax);

    EXPECT_GE(::nn::hid::TouchScreenStateCountMax, count);

    for (int i = 0; i < count - 1; ++i)
    {
        EXPECT_EQ(1,
                  TestFixture::s_States[i].samplingNumber -
                  TestFixture::s_States[i + 1].samplingNumber);
    }

    for (int i = 0; i < count; ++i)
    {
        SCOPED_TRACE("");
        TestFixture::ExpectDefaultTouchScreenState(TestFixture::s_States[i]);
    }

    ::nn::hid::debug::FinalizeTouchScreen();
}

template<typename T>
::nn::hid::TouchScreenState<T::Value>
    TouchStateCountSuite<T>::s_States[::nn::hid::TouchScreenStateCountMax];

template<typename T>
void TouchStateCountSuite<T>::ExpectDefaultTouchScreenState(
    const ::nn::hid::TouchScreenState<T::Value>& state) NN_NOEXCEPT
{
    EXPECT_LE(0, state.samplingNumber);
    EXPECT_EQ(0, state.count);
}
