﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_TimeSpan.h>
#include <nn/hid.h>
#include <nn/hid/debug/hid_TouchScreen.h>
#include <nn/os.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nnt/nntest.h>

#include "../Common/testHid_TouchScreen.h"

template<typename T>
class TouchScreenInitializationSuite : public ::testing::Test
{
protected:
    static ::nn::hid::TouchScreenState<T::Value> s_State;
};

// TouchScreenState::touches の数が以下の場合についてそれぞれテストを実行する。
typedef ::testing::Types<
    ::nnt::hid::TouchStateCountType<::nn::hid::TouchStateCountMax>
    > TouchStateCountTypes;

TYPED_TEST_CASE(TouchScreenInitializationSuite, TouchStateCountTypes);

//!< TouchScreen の初期化状態は GetTouchScreenState() の呼び出しに反映されるか
TYPED_TEST(TouchScreenInitializationSuite, NoInitializationTest1)
{
    // TouchScreen が未初期化の状態で呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::GetTouchScreenState(&TestFixture::s_State), "");

    ::nn::hid::InitializeTouchScreen();

    ::nn::hid::InitializeTouchScreen();

    ::nn::hid::debug::FinalizeTouchScreen();

    // TouchScreen が終了しきっていないので成功する。
    ::nn::hid::GetTouchScreenState(&TestFixture::s_State);

    ::nn::hid::debug::FinalizeTouchScreen();

    // TouchScreen の終了処理後に呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::GetTouchScreenState(&TestFixture::s_State), "");
}

//!< TouchScreen の初期化状態は GetTouchScreenStates() の呼び出しに反映されるか
TYPED_TEST(TouchScreenInitializationSuite, NoInitializationTest2)
{
    // TouchScreen が未初期化の状態で呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::GetTouchScreenStates(&TestFixture::s_State, 1), "");

    ::nn::hid::InitializeTouchScreen();

    ::nn::hid::InitializeTouchScreen();

    ::nn::hid::debug::FinalizeTouchScreen();

    // TouchScreen が終了しきっていないので成功する。
    ::nn::hid::GetTouchScreenState(&TestFixture::s_State);

    ::nn::hid::debug::FinalizeTouchScreen();

    // TouchScreen の終了処理後に呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::GetTouchScreenStates(&TestFixture::s_State, 1), "");
}

//!< TouchScreen の初期化状態は SetTouchScreenAutoPilotState() の呼び出しに反映されるか
TYPED_TEST(TouchScreenInitializationSuite, NoInitializationTest3)
{
    ::nn::hid::debug::TouchScreenAutoPilotState<TypeParam::Value
        > autoPilotState = {};

    // TouchScreen が未初期化の状態で呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::debug::SetTouchScreenAutoPilotState(autoPilotState), "");

    ::nn::hid::InitializeTouchScreen();

    ::nn::hid::InitializeTouchScreen();

    ::nn::hid::debug::FinalizeTouchScreen();

    // TouchScreen が終了しきっていないので成功する。
    ::nn::hid::debug::SetTouchScreenAutoPilotState(autoPilotState);

    ::nnt::hid::FinalizeTouchScreenForAutoPilot();

    // TouchScreen の終了処理後に呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::debug::SetTouchScreenAutoPilotState(autoPilotState), "");
}

//!< TouchScreen の初期化状態は UnsetTouchScreenAutoPilotState() の呼び出しに反映されるか
TYPED_TEST(TouchScreenInitializationSuite, NoInitializationTest4)
{
    // TouchScreen が未初期化の状態で呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::debug::UnsetTouchScreenAutoPilotState(), "");

    ::nn::hid::InitializeTouchScreen();

    ::nn::hid::InitializeTouchScreen();

    ::nn::hid::debug::FinalizeTouchScreen();

    // TouchScreen が終了しきっていないので成功する。
    ::nn::hid::debug::UnsetTouchScreenAutoPilotState();

    ::nn::hid::debug::FinalizeTouchScreen();

    // TouchScreen の終了処理後に呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::debug::UnsetTouchScreenAutoPilotState(), "");
}

//!< TouchScreen の初期化処理は最低一つの入力状態の取得を保証するか
TYPED_TEST(TouchScreenInitializationSuite, StateReadingTest1)
{
    ::nn::hid::InitializeTouchScreen();

    // 初期化直後の入力状態の取得に成功する。
    ::nn::hid::GetTouchScreenState(&TestFixture::s_State);
    EXPECT_LE(0, TestFixture::s_State.samplingNumber);
    EXPECT_EQ(0, TestFixture::s_State.count);

    ::nn::hid::debug::FinalizeTouchScreen();
}

//!< TouchScreen の初期化処理はサンプリングを正しく有効化するか
TYPED_TEST(TouchScreenInitializationSuite, StateReadingTest2)
{
    ::nn::hid::InitializeTouchScreen();

    ::nn::hid::InitializeTouchScreen();

    ::nn::hid::GetTouchScreenState(&TestFixture::s_State);

    int64_t samplingNumber = TestFixture::s_State.samplingNumber;

    ::nn::hid::debug::FinalizeTouchScreen();

    // TouchScreen が終了しきっていないのでサンプリングは継続する。
    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::TouchScreenStateCountMax *
            ::nnt::hid::GetTouchScreenSamplingInterval().GetMilliSeconds()));

    ::nn::hid::GetTouchScreenState(&TestFixture::s_State);

    EXPECT_LT(samplingNumber + ::nn::hid::TouchScreenStateCountMax / 4,
              TestFixture::s_State.samplingNumber);

    ::nn::hid::debug::FinalizeTouchScreen();
#if defined(NN_BUILD_CONFIG_OS_WIN)
    samplingNumber = TestFixture::s_State.samplingNumber;

    // TouchScreen が終了しているのでサンプリングは停止する。
    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::TouchScreenStateCountMax *
            ::nnt::hid::GetTouchScreenSamplingInterval().GetMilliSeconds()));

    ::nn::hid::InitializeTouchScreen();

    ::nn::hid::GetTouchScreenState(&TestFixture::s_State);

    EXPECT_GT(samplingNumber + ::nn::hid::TouchScreenStateCountMax / 4,
              TestFixture::s_State.samplingNumber);

    ::nn::hid::debug::FinalizeTouchScreen();
#endif
}

template<typename T>
::nn::hid::TouchScreenState<T::Value>
    TouchScreenInitializationSuite<T>::s_State;
