﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <memory>
#include <nn/nn_TimeSpan.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/hid.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/debug/hid_TouchScreen.h>
#include <nn/os.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nnt/nntest.h>

#include "../Common/testHid_TouchScreen.h"

namespace {

//!< TouchScreen の自動操作状態をファジングします。
void FuzzTouchScreenAutoPilotState(
    ::nnt::hid::TouchScreenAutoPilotState* pState, int seed) NN_NOEXCEPT;

//!< タッチの値を変域内に収めます。
::nn::hid::TouchState ClampTouchState(const ::nn::hid::TouchState& state
                                      ) NN_NOEXCEPT;

//!< タッチの値を上限方向へ超過させます。
void MakeTouchStateOverflow(::nn::hid::TouchState* pState) NN_NOEXCEPT;

//!< タッチの値を下限方向へ超過させます。
void MakeTouchStateUnderflow(::nn::hid::TouchState* pState) NN_NOEXCEPT;

//!< タッチの値が上限にあることを期待します。
void ExpectMaximumTouchState(const ::nn::hid::TouchState& state) NN_NOEXCEPT;

//!< タッチの値が下限にあることを期待します。
void ExpectMinimumTouchState(const ::nn::hid::TouchState& state) NN_NOEXCEPT;

} // namespace

template<typename T>
class TouchScreenAutoPilotSuite : public ::testing::Test
{
protected:
    static ::nn::hid::TouchScreenState<T::Value>
        s_States1[::nn::hid::TouchScreenStateCountMax];

    static ::nn::hid::TouchScreenState<T::Value>
        s_States2[::nn::hid::TouchScreenStateCountMax];

    //!< TouchScreen の入力状態が入力無し状態であることを期待します。
    static void ExpectDefaultTouchScreenState(
        const ::nn::hid::TouchScreenState<T::Value>& state) NN_NOEXCEPT;

    //!< TouchScreen の入力状態がファジングされた状態であることを期待します。
    static void ExpectFuzzedTouchScreenState(
        const ::nn::hid::TouchScreenState<T::Value>& state) NN_NOEXCEPT;

    //!< 同じドライバの状態を起源とする TouchScreen の入力状態だと期待します。
    static void ExpectSameOriginTouchScreenStates(
        const ::nn::hid::TouchScreenState<T::Value>& lhs,
        const ::nn::hid::TouchScreenState<T::Value>& rhs) NN_NOEXCEPT;

    //!< TouchScreen の入力状態が隣り合っていることを期待します。
    static void ExpectAdjacentTouchScreenStates(
        const ::nn::hid::TouchScreenState<T::Value>& lhs,
        const ::nn::hid::TouchScreenState<T::Value>& rhs) NN_NOEXCEPT;
};

template<typename T>
class TouchAutoPilotStateCountSuite : public ::testing::Test
{
};

// TouchScreenState::touches の数が以下の場合についてそれぞれテストを実行する。
typedef ::testing::Types<
    ::nnt::hid::TouchStateCountType<1>,
    ::nnt::hid::TouchStateCountType<7>,
    ::nnt::hid::TouchStateCountType<::nn::hid::TouchStateCountMax>
    > TouchStateCountTypes;

// TouchScreenAutoPilotState::touches の数が以下の場合についてそれぞれテストを
// 実行する。
typedef ::testing::Types<
    ::nnt::hid::TouchStateCountType<1>,
    ::nnt::hid::TouchStateCountType<2>,
    ::nnt::hid::TouchStateCountType<3>,
    ::nnt::hid::TouchStateCountType<4>,
    ::nnt::hid::TouchStateCountType<5>,
    ::nnt::hid::TouchStateCountType<6>,
    ::nnt::hid::TouchStateCountType<7>,
    ::nnt::hid::TouchStateCountType<8>,
    ::nnt::hid::TouchStateCountType<9>,
    ::nnt::hid::TouchStateCountType<10>,
    ::nnt::hid::TouchStateCountType<11>,
    ::nnt::hid::TouchStateCountType<12>,
    ::nnt::hid::TouchStateCountType<13>,
    ::nnt::hid::TouchStateCountType<14>,
    ::nnt::hid::TouchStateCountType<15>,
    ::nnt::hid::TouchStateCountType<16>
    > TouchAutoPilotStateCountTypes;

TYPED_TEST_CASE(TouchScreenAutoPilotSuite, TouchStateCountTypes);

TYPED_TEST_CASE(TouchAutoPilotStateCountSuite, TouchAutoPilotStateCountTypes);

//!< TouchScreen の初期化処理は自動操作状態を正しく管理するか
TYPED_TEST(TouchScreenAutoPilotSuite, InitializationTest1)
{
    ::std::unique_ptr<
        ::nnt::hid::TouchScreenAutoPilotState
        > pAutoPilotState(new ::nnt::hid::TouchScreenAutoPilotState());
    FuzzTouchScreenAutoPilotState(pAutoPilotState.get(), 1);

    ::nn::hid::InitializeTouchScreen();

    ::nn::hid::debug::SetTouchScreenAutoPilotState(*pAutoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::TouchScreenStateCountMax / 2 *
            ::nnt::hid::GetTouchScreenSamplingInterval().GetMilliSeconds()));

    // 自動操作が反映された入力状態が取得される。
    ::nn::hid::GetTouchScreenState(&TestFixture::s_States1[0]);
    {
        SCOPED_TRACE("");
        TestFixture::ExpectFuzzedTouchScreenState(TestFixture::s_States1[0]);
    }

    ::nn::hid::InitializeTouchScreen();

    ::nn::hid::debug::FinalizeTouchScreen();

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::TouchScreenStateCountMax / 2 *
            ::nnt::hid::GetTouchScreenSamplingInterval().GetMilliSeconds()));

    // まだユーザが存在するので自動操作設定は解除されない。
    ::nn::hid::GetTouchScreenState(&TestFixture::s_States1[0]);
    {
        SCOPED_TRACE("");
        TestFixture::ExpectFuzzedTouchScreenState(TestFixture::s_States1[0]);
    }

    ::nnt::hid::FinalizeTouchScreenForAutoPilot();

    ::nn::hid::InitializeTouchScreen();

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::TouchScreenStateCountMax / 2 *
            ::nnt::hid::GetTouchScreenSamplingInterval().GetMilliSeconds()));

    // TouchScreen が完全に開放されたことで自動操作設定が解除される。
    ::nn::hid::GetTouchScreenState(&TestFixture::s_States1[0]);
    {
        SCOPED_TRACE("");
        TestFixture::ExpectDefaultTouchScreenState(TestFixture::s_States1[0]);
    }

    ::nnt::hid::FinalizeTouchScreenForAutoPilot();
}

//!< TouchScreen の入力状態は不正な自動操作状態が設定されても仕様を保証するか
TYPED_TEST(TouchScreenAutoPilotSuite, InvalidArgumentTest1)
{
    ::nn::hid::InitializeTouchScreen();

    ::std::unique_ptr<
        ::nnt::hid::TouchScreenAutoPilotState
        > pAutoPilotState(new ::nnt::hid::TouchScreenAutoPilotState());
    for (::nn::hid::TouchState& touch : pAutoPilotState->touches)
    {
        touch.fingerId =
            static_cast<int32_t>(&touch - pAutoPilotState->touches);
    }

    // タッチの数が下限未満
    pAutoPilotState->count = -1;
    ::nn::hid::debug::SetTouchScreenAutoPilotState(*pAutoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::TouchScreenStateCountMax / 2 *
            ::nnt::hid::GetTouchScreenSamplingInterval().GetMilliSeconds()));

    ::nn::hid::GetTouchScreenState(&TestFixture::s_States1[0]);
    EXPECT_EQ(0, TestFixture::s_States1[0].count);

    // タッチの数が上限超過
    pAutoPilotState->count = ::nn::hid::TouchStateCountMax + 1;
    ::nn::hid::debug::SetTouchScreenAutoPilotState(*pAutoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::TouchScreenStateCountMax / 2 *
            ::nnt::hid::GetTouchScreenSamplingInterval().GetMilliSeconds()));

    ::nn::hid::GetTouchScreenState(&TestFixture::s_States1[0]);
    EXPECT_EQ(TypeParam::Value, TestFixture::s_States1[0].count);
    for (const ::nn::hid::TouchState& touch : TestFixture::s_States1[0].touches)
    {
        EXPECT_EQ(
            static_cast<int32_t>(&touch - TestFixture::s_States1[0].touches),
            touch.fingerId);
    }

    // 変域を下限方向に超過
    pAutoPilotState->count = 1;
    MakeTouchStateUnderflow(&pAutoPilotState->touches[0]);
    ::nn::hid::debug::SetTouchScreenAutoPilotState(*pAutoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::TouchScreenStateCountMax / 2 *
            ::nnt::hid::GetTouchScreenSamplingInterval().GetMilliSeconds()));

    ::nn::hid::GetTouchScreenState(&TestFixture::s_States1[0]);
    EXPECT_EQ(1, TestFixture::s_States1[0].count);
    {
         SCOPED_TRACE("");
        ExpectMinimumTouchState(TestFixture::s_States1[0].touches[0]);
    }

    // 変域を上限方向に超過
    pAutoPilotState->count = 1;
    MakeTouchStateOverflow(&pAutoPilotState->touches[0]);
    ::nn::hid::debug::SetTouchScreenAutoPilotState(*pAutoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::TouchScreenStateCountMax / 2 *
            ::nnt::hid::GetTouchScreenSamplingInterval().GetMilliSeconds()));

    ::nn::hid::GetTouchScreenState(&TestFixture::s_States1[0]);
    EXPECT_EQ(1, TestFixture::s_States1[0].count);
    {
         SCOPED_TRACE("");
        ExpectMaximumTouchState(TestFixture::s_States1[0].touches[0]);
    }

    // タッチの識別子が重複
    pAutoPilotState->count = 2;
    pAutoPilotState->touches[1].fingerId = pAutoPilotState->touches[0].fingerId;
    ::nn::hid::debug::SetTouchScreenAutoPilotState(*pAutoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::TouchScreenStateCountMax / 2 *
            ::nnt::hid::GetTouchScreenSamplingInterval().GetMilliSeconds()));

    ::nn::hid::GetTouchScreenState(&TestFixture::s_States1[0]);
    EXPECT_EQ(1, TestFixture::s_States1[0].count);

    ::nnt::hid::FinalizeTouchScreenForAutoPilot();
}

//!< TouchScreen の入力状態は自動操作状態の設定を正しく反映するか
TYPED_TEST(TouchScreenAutoPilotSuite, StateReadingTest1)
{
    ::nn::hid::InitializeTouchScreen();

    ::nn::hid::GetTouchScreenState(&TestFixture::s_States1[0]);
    {
        SCOPED_TRACE("");
        TestFixture::ExpectDefaultTouchScreenState(TestFixture::s_States1[0]);
    }

    ::std::unique_ptr<
        ::nnt::hid::TouchScreenAutoPilotState
        > pAutoPilotState(new ::nnt::hid::TouchScreenAutoPilotState());

    for (int i = 0; i <= ::nn::hid::TouchScreenStateCountMax / 3; ++i)
    {
        FuzzTouchScreenAutoPilotState(pAutoPilotState.get(), i);
        ::nn::hid::debug::SetTouchScreenAutoPilotState(*pAutoPilotState);
        ::nn::os::SleepThread(
            ::nn::TimeSpan::FromMilliSeconds(
                2 * ::nnt::hid::GetTouchScreenSamplingInterval()
                        .GetMilliSeconds()));
    }

    int count = ::nn::hid::GetTouchScreenStates(
        TestFixture::s_States1,
        ::nn::hid::TouchScreenStateCountMax);
    EXPECT_LE(1, count);

    bool isActive = true;

    // TouchScreen の状態に応じて TouchScreen の入力状態は変化する。
    for (int i = 0; i < count; ++i)
    {
        const ::nn::hid::TouchScreenState<
            TypeParam::Value>& lhs = TestFixture::s_States1[i];

        EXPECT_LE(0, lhs.samplingNumber);

        if (lhs.count == 0)
        {
            isActive = false;
        }
        else
        {
            SCOPED_TRACE("");
            EXPECT_TRUE(isActive);
            TestFixture::ExpectFuzzedTouchScreenState(lhs);
        }

        if (i + 1 < count)
        {
            SCOPED_TRACE("");
            const ::nn::hid::TouchScreenState<
                TypeParam::Value>& rhs = TestFixture::s_States1[i + 1];
            TestFixture::ExpectAdjacentTouchScreenStates(lhs, rhs);
            for (int j = 0; j < lhs.count; ++j)
            {
                EXPECT_FALSE(
                    lhs.touches[j].attributes
                       .template Test<::nn::hid::TouchAttribute::End>());
                if (j < rhs.count)
                {
                    EXPECT_FALSE(
                        lhs.touches[j].attributes
                           .template Test<::nn::hid::TouchAttribute::Start>());
                }
                else
                {
                    EXPECT_TRUE(
                        lhs.touches[j].attributes
                           .template Test<::nn::hid::TouchAttribute::Start>());
                }
            }
        }
    }

    ::nnt::hid::FinalizeTouchScreenForAutoPilot();
}

//!< TouchScreen の入力状態は自動操作状態の解除を正しく反映するか
TYPED_TEST(TouchScreenAutoPilotSuite, StateReadingTest2)
{
    ::nn::hid::InitializeTouchScreen();

    ::std::unique_ptr<
        ::nnt::hid::TouchScreenAutoPilotState
        > pAutoPilotState(new ::nnt::hid::TouchScreenAutoPilotState());
    FuzzTouchScreenAutoPilotState(pAutoPilotState.get(), 10);

    ::nn::hid::debug::SetTouchScreenAutoPilotState(*pAutoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::TouchScreenStateCountMax *
            ::nnt::hid::GetTouchScreenSamplingInterval().GetMilliSeconds()));

    // 自動操作状態を解除
    ::nn::hid::debug::UnsetTouchScreenAutoPilotState();

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::TouchScreenStateCountMax / 2 *
            ::nnt::hid::GetTouchScreenSamplingInterval().GetMilliSeconds()));

    int count = ::nn::hid::GetTouchScreenStates(
        TestFixture::s_States1,
        ::nn::hid::TouchScreenStateCountMax);
    EXPECT_EQ(::nn::hid::TouchScreenStateCountMax, count);

    bool hasChanged = false;

    ::std::unique_ptr<
        ::nn::hid::TouchScreenState<TypeParam::Value>
        > pState(new ::nn::hid::TouchScreenState<TypeParam::Value>());

    // 入力状態列状で自動操作状態は連続的に切り替わる。
    for (int i = 0; i < count; ++i)
    {
       const ::nn::hid::TouchScreenState<
            TypeParam::Value>& lhs = TestFixture::s_States1[i];

        if (i + 1 < count)
        {
            SCOPED_TRACE("");
            TestFixture::ExpectAdjacentTouchScreenStates(
                lhs, TestFixture::s_States1[i + 1]);
        }

        if (lhs.touches[0].x != pState->touches[0].x)
        {
            EXPECT_FALSE(hasChanged);
            hasChanged = true;
            pState->count = ::std::min<int32_t>(pAutoPilotState->count,
                                                TypeParam::Value);
            for (int j = 0; j < TypeParam::Value; ++j)
            {
                pState->touches[j] =
                    ClampTouchState(pAutoPilotState->touches[j]);
            }

            // 自動操作状態を解除ではトリガはかからない
            for (int j = 0; j < lhs.count; ++j)
            {
                EXPECT_FALSE(
                    lhs.touches[j].attributes
                       .template Test<::nn::hid::TouchAttribute::End>());
            }
        }

        {
            SCOPED_TRACE("");
            TestFixture::ExpectSameOriginTouchScreenStates(*pState, lhs);
        }
    }

    EXPECT_TRUE(hasChanged);

    ::nnt::hid::FinalizeTouchScreenForAutoPilot();
}

//!< LIFO の境界を跨いで自動操作状態の設定を正しく反映するか
TYPED_TEST(TouchScreenAutoPilotSuite, StateReadingTest3)
{
    ::nn::hid::InitializeTouchScreen();

    ::std::unique_ptr<
        ::nnt::hid::TouchScreenAutoPilotState
        > pAutoPilotState(new ::nnt::hid::TouchScreenAutoPilotState());

    int lastCount = 0;

    // LIFO の境界を 2 回跨ぐまでサンプリングを継続
    for (int i = 0; i < ::nn::hid::TouchScreenStateCountMax * 3; ++i)
    {
        FuzzTouchScreenAutoPilotState(pAutoPilotState.get(), i);
        ::nn::hid::debug::SetTouchScreenAutoPilotState(*pAutoPilotState);
        ::nn::os::SleepThread(
            ::nn::TimeSpan::FromMilliSeconds(
                ::nnt::hid::GetTouchScreenSamplingInterval()
                    .GetMilliSeconds()));

        // 今回のサンプリング結果を格納するバッファ
        ::nn::hid::TouchScreenState<TypeParam::Value>* ptr1 =
              (i % 2 == 1) ? &TestFixture::s_States1[0]
                           : &TestFixture::s_States2[0];

        // 前回のサンプリング結果が格納されているバッファ
        ::nn::hid::TouchScreenState<TypeParam::Value>* ptr2 =
              (i % 2 == 1) ? &TestFixture::s_States2[0]
                           : &TestFixture::s_States1[0];

        // 取得される入力状態の数は単調増加する。
        int count = ::nn::hid::GetTouchScreenStates(
            ptr1,
            ::nn::hid::TouchScreenStateCountMax);
        EXPECT_LE(lastCount, count);

        bool isActive = true;

        for (int j = 0; j < count; ++j)
        {
            if (ptr1[j].count == 0)
            {
                isActive = false;
                EXPECT_LE(0, ptr1[i].samplingNumber);
            }
            else
            {
                SCOPED_TRACE("");
                EXPECT_TRUE(isActive);
                TestFixture::ExpectFuzzedTouchScreenState(ptr1[j]);
            }

            if (j + 1 < count)
            {
                SCOPED_TRACE("");
                TestFixture::ExpectAdjacentTouchScreenStates(ptr1[j],
                                                             ptr1[j + 1]);
            }

            // 前回のサンプリング結果と重複する箇所には同じ値が入る。
            if (lastCount > 0 &&
                ptr1[j].samplingNumber == ptr2[0].samplingNumber)
            {
                for (int k = 0; k < lastCount - j; ++k)
                {
                    SCOPED_TRACE("");
                    EXPECT_EQ(ptr2[k].samplingNumber,
                              ptr1[j + k].samplingNumber);
                    TestFixture::ExpectSameOriginTouchScreenStates(ptr2[k],
                                                                   ptr1[j + k]);
                }

                break;
            }
        }

        lastCount = count;
    }

    EXPECT_EQ(::nn::hid::TouchScreenStateCountMax, lastCount);

    ::nnt::hid::FinalizeTouchScreenForAutoPilot();
}

//!< TouchScreen の入力状態は自動操作状態のリリーストリガを正しく反映するか
TYPED_TEST(TouchScreenAutoPilotSuite, StateReadingTest4)
{
    ::nn::hid::InitializeTouchScreen();

    ::std::unique_ptr<
        ::nnt::hid::TouchScreenAutoPilotState
        > pAutoPilotState(new ::nnt::hid::TouchScreenAutoPilotState());
    FuzzTouchScreenAutoPilotState(pAutoPilotState.get(), 10);

    ::nn::hid::debug::SetTouchScreenAutoPilotState(*pAutoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::TouchScreenStateCountMax *
            ::nnt::hid::GetTouchScreenSamplingInterval().GetMilliSeconds()));

    // 自動操作状態を解放に
    ::nn::hid::debug::SetTouchScreenAutoPilotState(
        ::nnt::hid::TouchScreenAutoPilotState());

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::TouchScreenStateCountMax / 2 *
            ::nnt::hid::GetTouchScreenSamplingInterval().GetMilliSeconds()));

    int count = ::nn::hid::GetTouchScreenStates(
        TestFixture::s_States1,
        ::nn::hid::TouchScreenStateCountMax);
    EXPECT_EQ(::nn::hid::TouchScreenStateCountMax, count);

    bool hasChanged = false;

    ::std::unique_ptr<
        ::nn::hid::TouchScreenState<TypeParam::Value>
        > pState(new ::nn::hid::TouchScreenState<TypeParam::Value>());

    // 入力状態列状で自動操作状態は連続的に切り替わる。
    for (int i = 0; i < count; ++i)
    {
       const ::nn::hid::TouchScreenState<
            TypeParam::Value>& lhs = TestFixture::s_States1[i];

        if (i + 1 < count)
        {
            SCOPED_TRACE("");
            TestFixture::ExpectAdjacentTouchScreenStates(
                lhs, TestFixture::s_States1[i + 1]);
        }

        if (lhs.touches[0].x != pState->touches[0].x)
        {
            EXPECT_FALSE(hasChanged);
            hasChanged = true;
            pState->count = ::std::min<int32_t>(pAutoPilotState->count,
                                                TypeParam::Value);
            for (int j = 0; j < TypeParam::Value; ++j)
            {
                pState->touches[j] =
                    ClampTouchState(pAutoPilotState->touches[j]);
            }

            for (int j = 0; j < lhs.count; ++j)
            {
                EXPECT_TRUE(
                    lhs.touches[j].attributes
                       .template Test<::nn::hid::TouchAttribute::End>());
            }
        }

        {
            SCOPED_TRACE("");
            TestFixture::ExpectSameOriginTouchScreenStates(*pState, lhs);
        }
    }

    EXPECT_TRUE(hasChanged);

    ::nnt::hid::FinalizeTouchScreenForAutoPilot();
}

//!< 任意のタッチ数において自動操作状態の設定を正しく反映するか
TYPED_TEST(TouchAutoPilotStateCountSuite, ReadingStateTest1)
{
    ::nn::hid::InitializeTouchScreen();

    ::std::unique_ptr<
        ::nn::hid::debug::TouchScreenAutoPilotState<TypeParam::Value>
        > pAutoPilotState(
            new ::nn::hid::debug::TouchScreenAutoPilotState<TypeParam::Value>()
            );

    pAutoPilotState->count = static_cast<int32_t>(TypeParam::Value);
    for (::nn::hid::TouchState& touch : pAutoPilotState->touches)
    {
        auto seed = static_cast<int32_t>(&touch - pAutoPilotState->touches);
        touch.fingerId = seed;
        touch.x = seed + 1 + 64;
        touch.y = seed + 2 + 64;
        touch.diameterX = seed + 3 + 64;
        touch.diameterY = seed + 4 + 64;
        touch.rotationAngle = seed + 5;
    }

    ::nn::hid::debug::SetTouchScreenAutoPilotState(*pAutoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::TouchScreenStateCountMax / 2 *
            ::nnt::hid::GetTouchScreenSamplingInterval().GetMilliSeconds()));

    ::std::unique_ptr<
        ::nn::hid::TouchScreenState<::nn::hid::TouchStateCountMax>
        > pState(
            new ::nn::hid::TouchScreenState<::nn::hid::TouchStateCountMax>());

    ::nn::hid::GetTouchScreenState(pState.get());

    for (int32_t i = 0; i < pState->count; ++i)
    {
        const ::nn::hid::TouchState& touch = pState->touches[i];
        EXPECT_EQ(i, touch.fingerId);
        EXPECT_EQ(i + 1 + 64, touch.x);
        EXPECT_EQ(i + 2 + 64, touch.y);
        EXPECT_EQ(i + 3 + 64, touch.diameterX);
        EXPECT_EQ(i + 4 + 64, touch.diameterY);
        EXPECT_EQ(i + 5, touch.rotationAngle);
    }

    ::nnt::hid::FinalizeTouchScreenForAutoPilot();
}

namespace nnt { namespace hid {

template<int N>
const int TouchStateCountType<N>::Value;

}} // namespace nnt::hid

namespace {

//!< タッチの回転度数の最小値
const int32_t TouchRotationAngleMin = -270;

//!< タッチの回転度数の最大値
const int32_t TouchRotationAngleMax = 270;

//!< TouchScreen の座標域
const ::nn::hid::detail::Rectangle TouchScreenRectangle = {
#if   defined(NN_BUILD_CONFIG_SPEC_NX)
    15,     // x
    15,     // y
    1250,   // width
    690     // height
#else
    0,      // x
    0,      // y
    1280,   // width
    720     // height
#endif
};

void FuzzTouchScreenAutoPilotState(
    ::nnt::hid::TouchScreenAutoPilotState* pState, int seed) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);

    pState->count = (seed % ::nn::hid::TouchStateCountMax) + 1;

    for (int i = 0; i < ::nn::hid::TouchStateCountMax; ++i)
    {
        ::nn::hid::TouchState& touch = pState->touches[i];
        const int localSheed = (i + 1) * seed;
        touch.deltaTime = ::nn::TimeSpanType::FromMilliSeconds(localSheed * 3);
        touch.fingerId = (i + 1) * 5;
        touch.x = localSheed * 7 + 64;
        touch.y = localSheed * 3 + 64;
        touch.diameterX = localSheed * 5 + 64;
        touch.diameterY = localSheed * 7 + 64;
        touch.rotationAngle = localSheed * 3;
    }
}

::nn::hid::TouchState ClampTouchState(const ::nn::hid::TouchState& value
                                      ) NN_NOEXCEPT
{
    ::nn::hid::TouchState state = value;
    state.x = ::std::max(
        TouchScreenRectangle.x,
        ::std::min(state.x,
                   TouchScreenRectangle.x + TouchScreenRectangle.width - 1));
    state.y = ::std::max(
        TouchScreenRectangle.y,
        ::std::min(state.y,
                   TouchScreenRectangle.y + TouchScreenRectangle.height - 1));
    state.diameterX = ::std::max(0, ::std::min(state.diameterX,
                                               TouchScreenRectangle.width));
    state.diameterY = ::std::max(0, ::std::min(state.diameterY,
                                               TouchScreenRectangle.height));
    state.rotationAngle = ::std::max(TouchRotationAngleMin,
                                     ::std::min(state.rotationAngle,
                                                TouchRotationAngleMax));
    return state;
}

void MakeTouchStateOverflow(::nn::hid::TouchState* pState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);
    pState->x = TouchScreenRectangle.x + TouchScreenRectangle.width;
    pState->y = TouchScreenRectangle.y + TouchScreenRectangle.height;
    pState->diameterX = TouchScreenRectangle.width + 1;
    pState->diameterY = TouchScreenRectangle.height + 1;
    pState->rotationAngle = TouchRotationAngleMax + 1;
}

void MakeTouchStateUnderflow(::nn::hid::TouchState* pState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);
    pState->x = TouchScreenRectangle.x - 1;
    pState->y = TouchScreenRectangle.y - 1;
    pState->diameterX = -1;
    pState->diameterY = -1;
    pState->rotationAngle = TouchRotationAngleMin - 1;
}

void ExpectMaximumTouchState(const ::nn::hid::TouchState& state) NN_NOEXCEPT
{
    EXPECT_EQ(TouchScreenRectangle.x + TouchScreenRectangle.width - 1,
              state.x);
    EXPECT_EQ(TouchScreenRectangle.y + TouchScreenRectangle.height - 1,
              state.y);
    EXPECT_EQ(TouchScreenRectangle.width, state.diameterX);
    EXPECT_EQ(TouchScreenRectangle.height, state.diameterY);
    EXPECT_EQ(TouchRotationAngleMax, state.rotationAngle);
}

void ExpectMinimumTouchState(const ::nn::hid::TouchState& state) NN_NOEXCEPT
{
    EXPECT_EQ(TouchScreenRectangle.x, state.x);
    EXPECT_EQ(TouchScreenRectangle.y, state.y);
    EXPECT_EQ(0, state.diameterX);
    EXPECT_EQ(0, state.diameterY);
    EXPECT_EQ(TouchRotationAngleMin, state.rotationAngle);
}

} // namespace

template<typename T>
::nn::hid::TouchScreenState<T::Value>
    TouchScreenAutoPilotSuite<
        T>::s_States1[::nn::hid::TouchScreenStateCountMax];

template<typename T>
::nn::hid::TouchScreenState<T::Value>
    TouchScreenAutoPilotSuite<
        T>::s_States2[::nn::hid::TouchScreenStateCountMax];

template<typename T>
void TouchScreenAutoPilotSuite<T>::ExpectDefaultTouchScreenState(
    const ::nn::hid::TouchScreenState<T::Value>& state) NN_NOEXCEPT
{
    EXPECT_LE(0, state.samplingNumber);
    EXPECT_EQ(0, state.count);
}

template<typename T>
void TouchScreenAutoPilotSuite<T>::ExpectFuzzedTouchScreenState(
    const ::nn::hid::TouchScreenState<T::Value>& state) NN_NOEXCEPT
{
    const int touchStateCountMax = static_cast<int>(T::Value);
    EXPECT_LE(0, state.samplingNumber);
    EXPECT_GE(touchStateCountMax, state.count);
    const int count = ::std::min(state.count, touchStateCountMax);
    for (int i = 0; i < count; ++i)
    {
        const ::nn::hid::TouchState& touch = state.touches[i];
        const int localSheed = static_cast<int>(
            touch.deltaTime.GetMilliSeconds() / 3);
        EXPECT_EQ(0, localSheed % (i + 1));
        if (i == 0 && touchStateCountMax > state.count)
        {
            EXPECT_EQ(state.count,
                      (localSheed % ::nn::hid::TouchStateCountMax) + 1);
        }
        EXPECT_EQ(localSheed * 3, touch.deltaTime.GetMilliSeconds());
        EXPECT_EQ((i + 1) * 5, touch.fingerId);
        EXPECT_EQ(::std::min(
                      localSheed * 7 + 64,
                      TouchScreenRectangle.x + TouchScreenRectangle.width - 1),
                  touch.x);
        EXPECT_EQ(::std::min(
                      localSheed * 3 + 64,
                      TouchScreenRectangle.y + TouchScreenRectangle.height - 1),
                  touch.y);
        EXPECT_EQ(::std::min(localSheed * 5 + 64, TouchScreenRectangle.width),
                  touch.diameterX);
        EXPECT_EQ(::std::min(localSheed * 7 + 64, TouchScreenRectangle.height),
                  touch.diameterY);
        EXPECT_EQ(::std::min(localSheed * 3, TouchRotationAngleMax),
                  touch.rotationAngle);
    }
}

template<typename T>
void TouchScreenAutoPilotSuite<T>::ExpectSameOriginTouchScreenStates(
    const ::nn::hid::TouchScreenState<T::Value>& lhs,
    const ::nn::hid::TouchScreenState<T::Value>& rhs) NN_NOEXCEPT
{
    const int touchStateCountMax = static_cast<int>(T::Value);
    const int count = ::std::min(::std::min(lhs.count , rhs.count),
                                 touchStateCountMax);
    EXPECT_GE(touchStateCountMax, lhs.count);
    EXPECT_GE(touchStateCountMax, rhs.count);
    EXPECT_EQ(lhs.count, rhs.count);
    for (int i = 0; i < count; ++i)
    {
        const ::nn::hid::TouchState& lhsTouch = lhs.touches[i];
        const ::nn::hid::TouchState& rhsTouch = rhs.touches[i];
        EXPECT_EQ(lhsTouch.deltaTime.GetMilliSeconds(),
                  rhsTouch.deltaTime.GetMilliSeconds());
        EXPECT_EQ(lhsTouch.fingerId, rhsTouch.fingerId);
        EXPECT_EQ(lhsTouch.x, rhsTouch.x);
        EXPECT_EQ(lhsTouch.y, rhsTouch.y);
        EXPECT_EQ(lhsTouch.diameterX, rhsTouch.diameterX);
        EXPECT_EQ(lhsTouch.diameterY, rhsTouch.diameterY);
        EXPECT_EQ(lhsTouch.rotationAngle, rhsTouch.rotationAngle);
    }
}

template<typename T>
void TouchScreenAutoPilotSuite<T>::ExpectAdjacentTouchScreenStates(
    const ::nn::hid::TouchScreenState<T::Value>& lhs,
    const ::nn::hid::TouchScreenState<T::Value>& rhs) NN_NOEXCEPT
{
    const int touchStateCountMax = static_cast<int>(T::Value);
    const int count = ::std::min(::std::min(lhs.count, rhs.count),
                                 touchStateCountMax);
    EXPECT_GE(touchStateCountMax, lhs.count);
    EXPECT_GE(touchStateCountMax, rhs.count);
    EXPECT_EQ(1, lhs.samplingNumber - rhs.samplingNumber);
    for (int i = 0; i < count; ++i)
    {
        const ::nn::hid::TouchState& lhsTouch = lhs.touches[i];
        const ::nn::hid::TouchState& rhsTouch = rhs.touches[i];
        EXPECT_GE(lhsTouch.deltaTime.GetMilliSeconds(),
                  rhsTouch.deltaTime.GetMilliSeconds());
        EXPECT_GE(lhsTouch.fingerId, rhsTouch.fingerId);
        EXPECT_GE(lhsTouch.x, rhsTouch.x);
        EXPECT_GE(lhsTouch.y, rhsTouch.y);
        EXPECT_GE(lhsTouch.diameterX, rhsTouch.diameterX);
        EXPECT_GE(lhsTouch.diameterY, rhsTouch.diameterY);
        EXPECT_GE(lhsTouch.rotationAngle, rhsTouch.rotationAngle);
    }
}
