﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Base.h>
#include <nnt/nntest.h>

#include <nn/os/os_Thread.h>
#include <nn/nn_Log.h>
#include <cstdio>
#include <nn/os.h>
#include <nn/os/os_MultipleWait.h>
#include <../../Sources/Libraries/hid/detail/hid_SynchronizedTimer.h>

using namespace nn::hid::detail;

static const int TimerCount = 3;
static const int NativeEventCount = 2;
static const int HolderCount = TimerCount + NativeEventCount;

static SynchronizedTimer s_Timers[TimerCount];
static nn::os::EventType s_Event[NativeEventCount];

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
static int s_TimerPeriodsMs[] = {5, 3, 2};
static int s_EventSleepBeforeMs[] = {20, 30};
/* Expected results:
* 1st timer: 1sec / 5ms -> ~200
* 2nd timer: 1sec / 3ms -> ~333
* 3rd timer: 1sec * (20ms / 50ms) / 2ms -> ~200; error margin: one per period of 50ms -> 1s / 50ms -> 20
* 1st&2nd events: 1s / 50ms -> ~20
*/
static int s_ExpectedWakeupsMs[] = {200, 333, 200, 20, 20};
static int s_ExpectedWakeupsMarginMs[] = {1, 1, 20, 1, 1};
#elif defined(NN_BUILD_CONFIG_OS_WIN32)
// Windows does not seem to be very precise with high frequency timers, so lowering frequencies
static int s_TimerPeriodsMs[] = { 50, 30, 20 };
static int s_EventSleepBeforeMs[] = { 200, 300 };
static int s_ExpectedWakeupsMs[] = { 20, 33, 20, 2, 2 };
static int s_ExpectedWakeupsMarginMs[] = { 1, 1, 2, 0, 1 };
#endif

static std::atomic<bool> s_Quit { false };
static void ThreadFunction(void*)
{
    while ( ! s_Quit.load())
    {
        s_Timers[2].Enable(nn::TimeSpan::FromMilliSeconds(s_TimerPeriodsMs[2]));
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(s_EventSleepBeforeMs[0]));
        nn::os::SignalEvent(&s_Event[0]);

        s_Timers[2].Disable();
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(s_EventSleepBeforeMs[1]));
        nn::os::SignalEvent(&s_Event[1]);
    }
}

TEST(SynchronizedTimer, Test)
{
    SynchronizedTimerGroup group;

    for (int k = 0; k < NativeEventCount; k++)
    {
        nn::os::InitializeEvent(&s_Event[k], false, nn::os::EventClearMode_ManualClear);
    }

    s_Timers[0].Enable(nn::TimeSpan::FromMilliSeconds(s_TimerPeriodsMs[0]));
    s_Timers[1].Enable(nn::TimeSpan::FromMilliSeconds(s_TimerPeriodsMs[1]));
    // Third one will be enabled after linking for testing

    s_Timers[0].Link(&group);
    s_Timers[1].Link(&group);
    s_Timers[2].Link(&group);

    s_Timers[1].Unlink();
    s_Timers[1].Link(&group);

    nn::os::MultiWaitType wait;
    nn::os::InitializeMultiWait(&wait);

    nn::os::MultiWaitHolderType eventHolders[NativeEventCount];
    for (int k = 0; k < NativeEventCount; k++)
    {
        nn::os::InitializeMultiWaitHolder(&eventHolders[k], &s_Event[k]);
        nn::os::LinkMultiWaitHolder(&wait, &eventHolders[k]);
    }

    nn::os::ThreadType thread;
    static NN_ALIGNAS(4096) uint8_t s_Stack[16384];
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::CreateThread(&thread, ThreadFunction, nullptr, s_Stack, sizeof(s_Stack), nn::os::DefaultThreadPriority));
    nn::os::StartThread(&thread);

    void *holders[HolderCount];
    holders[0] = &s_Timers[0];
    holders[1] = &s_Timers[1];
    holders[2] = &s_Timers[2];
    holders[3] = &eventHolders[0];
    holders[4] = &eventHolders[1];

    static int wakeupCount[HolderCount];

    nn::TimeSpan end = nn::os::GetSystemTick().ToTimeSpan() + nn::TimeSpan::FromSeconds(1);
    while (nn::os::GetSystemTick().ToTimeSpan() < end)
    {
        static nn::os::Tick last[HolderCount];
        void* holder = group.GetExpiredTimer();

        if (holder == nullptr)
        {
            nn::TimeSpan timeout = group.GetTimeout();
            holder = nn::os::TimedWaitAny(&wait, timeout);
        }

        if (holder == nullptr)
        {
            holder = group.GetExpiredTimer();
        }

        if (holder == nullptr || holder == group.GetPeriodUpdatedHolder())
        {
            // holder may be nullptr in case of reconfiguration if the timeout expires
            // before the PeriodUpdated event is signaled.
            // Need to run WaitAny() again as there was a change in the timer periods
            continue;
        }

        int holderIndex;
        for (holderIndex = 0; holderIndex < HolderCount; holderIndex++)
        {
            if (holder == holders[holderIndex])
            {
                break;
            }
        }

        auto now = nn::os::GetSystemTick();
        // NN_LOG("[%lld] Hello from %d; delta=%lld us\n", (long long) now.ToTimeSpan().GetMicroSeconds(), (int) holderIndex, (long long)(now - last[holderIndex]).ToTimeSpan().GetMicroSeconds());

        if (holderIndex < TimerCount)
        {
            s_Timers[holderIndex].Clear();
        }
        else if (holderIndex >= TimerCount && holderIndex < TimerCount + NativeEventCount)
        {
            nn::os::ClearEvent(&s_Event[holderIndex - TimerCount]);
        }
        else
        {
            // Should not happen
            NN_ABORT();
        }

        last[holderIndex] = now;
        wakeupCount[holderIndex]++;
    }

    NN_LOG("Wakeup count: %d %d %d %d %d\n", wakeupCount[0], wakeupCount[1], wakeupCount[2], wakeupCount[3], wakeupCount[4]);

    // Note: timings on NX should be fine as we should be alone with FIFO scheduling
    // On Windows, the test can easily fail if the CPU is busy.
    for (int timerIndex = 0; timerIndex < HolderCount; timerIndex++)
    {
        ASSERT_LE(s_ExpectedWakeupsMs[timerIndex] - s_ExpectedWakeupsMarginMs[timerIndex], wakeupCount[timerIndex]);
        ASSERT_GE(s_ExpectedWakeupsMs[timerIndex] + s_ExpectedWakeupsMarginMs[timerIndex], wakeupCount[timerIndex]);
    }

    nn::os::UnlinkAllMultiWaitHolder(&wait);
    nn::os::FinalizeMultiWait(&wait);

    for (int k = 0; k < NativeEventCount; k++)
    {
        nn::os::FinalizeMultiWaitHolder(&eventHolders[k]);
    }

    s_Quit.store(true);
    nn::os::WaitThread(&thread);
    nn::os::DestroyThread(&thread);

    for (int eventIndex = 0; eventIndex < NativeEventCount; eventIndex++)
    {
        nn::os::FinalizeEvent(&s_Event[eventIndex]);
    }
}

TEST(SynchronizedTimer, NoSpuriousWakeup)
{
    SynchronizedTimer timer;
    SynchronizedTimerGroup group;

    timer.Link(&group);

    // Must be enabled after link to trigger corner case
    timer.Enable(nn::TimeSpan::FromMilliSeconds(50));

    nn::TimeSpan end = nn::os::GetSystemTick().ToTimeSpan() + nn::TimeSpan::FromMilliSeconds(100);

    nn::os::MultiWaitType wait;
    nn::os::InitializeMultiWait(&wait);
    nn::os::LinkMultiWaitHolder(&wait, group.GetPeriodUpdatedHolder());

    int wakeupCount = 0;
    while (nn::os::GetSystemTick().ToTimeSpan() < end)
    {
        SynchronizedTimer* expired = group.GetExpiredTimer();

        if (expired == nullptr)
        {
            nn::TimeSpan timeout = group.GetTimeout();
            nn::os::MultiWaitHolderType* holder = nn::os::TimedWaitAny(&wait, timeout);
            if (holder == group.GetPeriodUpdatedHolder())
            {
                continue;
            }
        }

        expired = group.GetExpiredTimer();
        if (expired == nullptr)
        {
            continue;
        }

        ASSERT_EQ(&timer, expired);
        timer.Clear();
        wakeupCount++;
    }
    ASSERT_EQ(wakeupCount, 2);

    nn::os::UnlinkAllMultiWaitHolder(&wait);
    nn::os::FinalizeMultiWait(&wait);
}
