﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_TimeSpan.h>
#include <nn/hid/debug/hid_SleepButton.h>
#include <nn/hid/system/hid_SleepButton.h>
#include <nn/os.h>
#include <nnt/nntest.h>

#include "../Common/testHid_SleepButton.h"

namespace
{

::nn::hid::system::SleepButtonState
    g_States[::nn::hid::system::SleepButtonStateCountMax];

//!< 指定の優先度のスレッドから入力状態の読み出しが可能かテストします。
void ThreadingTest(int priority);

} // namespace

//!< 入力状態の読み出し時にバッファは正しく利用されるか
TEST(SleepButtonReadingStateSuite, ReadingStateTest1)
{
    for (::nn::hid::system::SleepButtonState& state : g_States)
    {
        state = ::nn::hid::system::SleepButtonState();
        state.samplingNumber = 0xDEADBEEF;
        state.buttons.Set(1, true);
    }

    ::nn::hid::system::InitializeSleepButton();

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::system::SleepButtonStateCountMax / 2 *
            ::nnt::hid::GetSleepButtonSamplingInterval().GetMilliSeconds()));

    int count = ::nn::hid::system::GetSleepButtonStates(
        g_States, ::nn::hid::system::SleepButtonStateCountMax);

    EXPECT_GE(::nn::hid::system::SleepButtonStateCountMax, count);

    for (int i = count; i < ::nn::hid::system::SleepButtonStateCountMax; ++i)
    {
        // 入力状態の書き込まれなかった領域は値を維持する。
        EXPECT_EQ(0xDEADBEEF, g_States[i].samplingNumber);
        ::nn::hid::system::SleepButtonSet buttons;
        buttons.Reset();
        buttons.Set(1, true);
        EXPECT_EQ(buttons, g_States[i].buttons);
    }

    ::nn::hid::debug::FinalizeSleepButton();
}

//!< 高優先度のスレッドから入力状態の読み出しが可能か
TEST(SleepButtonReadingStateSuite, ThreadingTest1)
{
    ThreadingTest(::nn::os::HighestThreadPriority);
}

//!< 低優先度のスレッドから入力状態の読み出しが可能か
TEST(SleepButtonReadingStateSuite, ThreadingTest2)
{
    ThreadingTest(::nn::os::LowestThreadPriority);
}

namespace
{

void ThreadingTest(int priority)
{
    ::nn::os::ThreadType* pThread = ::nn::os::GetCurrentThread();

    // 指定の優先度に設定する。
    const int original = ::nn::os::ChangeThreadPriority(pThread, priority);

    ::nn::hid::system::InitializeSleepButton();

    int lastCount = 1;
    int64_t lastSamplingNumber = 0;

    for (int i = 0; i < 60; ++i)
    {
        int count = ::nn::hid::system::GetSleepButtonStates(
            g_States, ::nn::hid::system::SleepButtonStateCountMax);
        EXPECT_LE(lastCount, count);

        EXPECT_LE(lastSamplingNumber, g_States[0].samplingNumber);

        for (int j = 0; j < count - 1; ++j)
        {
            const ::nn::hid::system::SleepButtonState& lhs = g_States[j];
            const ::nn::hid::system::SleepButtonState& rhs = g_States[j + 1];
            EXPECT_EQ(1, lhs.samplingNumber - rhs.samplingNumber);
        }

        lastCount = count;
        lastSamplingNumber = g_States[0].samplingNumber;

        ::nn::os::SleepThread(::nn::TimeSpan::FromMilliSeconds(16));
    }

    EXPECT_EQ(::nn::hid::system::SleepButtonStateCountMax, lastCount);

    ::nn::hid::debug::FinalizeSleepButton();

    // 優先度を元に戻す。
    ::nn::os::ChangeThreadPriority(pThread, original);
}

} // namespace
