﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/hid/debug/hid_SleepButton.h>
#include <nn/hid/system/hid_SleepButton.h>
#include <nn/os.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nnt/nntest.h>

#include "../Common/testHid_SleepButton.h"

namespace
{

::nn::hid::system::SleepButtonState
    g_States1[::nn::hid::system::SleepButtonStateCountMax];
::nn::hid::system::SleepButtonState
    g_States2[::nn::hid::system::SleepButtonStateCountMax];

//!< イベントのシグナル状況を確認するか否かを表す値です。
const bool EnablesSignalAsserion =
#if defined(NN_BUILD_CONFIG_OS_WIN)
    true;
#else
    false;
#endif

//!< スリープボタンの自動操作状態をファジングします。
void FuzzSleepButtonAutoPilotState(
    ::nn::hid::debug::SleepButtonAutoPilotState* pState, int seed) NN_NOEXCEPT;

//!< スリープボタンの入力状態が入力無し状態であることを期待します。
void ExpectDefaultSleepButtonState(
    const ::nn::hid::system::SleepButtonState& state) NN_NOEXCEPT;

//!< スリープボタンの入力状態がファジングされた入力状態であることを期待します。
void ExpectFuzzedSleepButtonState(
    const ::nn::hid::system::SleepButtonState& state) NN_NOEXCEPT;

//!< スリープボタンの入力状態が同じ自動操作の状態から決定されたことを期待します。
void ExpectSameOriginSleepButtonStates(
    const ::nn::hid::system::SleepButtonState& lhs,
    const ::nn::hid::system::SleepButtonState& rhs) NN_NOEXCEPT;

//!< スリープボタンの入力状態が隣り合っていることを期待します。
void ExpectAdjacentSleepButtonStates(
    const ::nn::hid::system::SleepButtonState& lhs,
    const ::nn::hid::system::SleepButtonState& rhs) NN_NOEXCEPT;

} // namespace

//!< SleepButton の初期化処理は自動操作状態を正しく管理するか
TEST(SleepButtonAutoPilotSuite, InitializationTest1)
{
    ::nn::os::SystemEventType event = {};

    ::nn::hid::system::BindSleepButtonEvent(&event,
                                            ::nn::os::EventClearMode_AutoClear);

    if (NN_STATIC_CONDITION(EnablesSignalAsserion))
    {
        EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&event));
    }

    ::nn::hid::debug::SleepButtonAutoPilotState autoPilotState;
    FuzzSleepButtonAutoPilotState(&autoPilotState, 1);

    ::nn::hid::system::InitializeSleepButton();

    if (NN_STATIC_CONDITION(EnablesSignalAsserion))
    {
        EXPECT_FALSE(::nn::os::TryWaitSystemEvent(&event));
    }

    ::nn::hid::debug::SetSleepButtonAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::system::SleepButtonStateCountMax / 2 *
            ::nnt::hid::GetSleepButtonSamplingInterval().GetMilliSeconds()));

    if (NN_STATIC_CONDITION(EnablesSignalAsserion))
    {
        EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&event));
    }

    // 自動操作が反映された入力状態が取得される。
    ::nn::hid::system::GetSleepButtonStates(&g_States1[0], 1);
    {
        SCOPED_TRACE("");
        ExpectFuzzedSleepButtonState(g_States1[0]);
    }

    ::nn::hid::system::InitializeSleepButton();

    ::nn::hid::debug::FinalizeSleepButton();

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::system::SleepButtonStateCountMax / 2 *
            ::nnt::hid::GetSleepButtonSamplingInterval().GetMilliSeconds()));

    if (NN_STATIC_CONDITION(EnablesSignalAsserion))
    {
        EXPECT_FALSE(::nn::os::TryWaitSystemEvent(&event));
    }

    // まだユーザが存在するので自動操作設定は解除されない。
    ::nn::hid::system::GetSleepButtonStates(&g_States1[0], 1);
    {
        SCOPED_TRACE("");
        ExpectFuzzedSleepButtonState(g_States1[0]);
    }

    ::nnt::hid::FinalizeSleepButtonForAutoPilot();

    ::nn::hid::system::InitializeSleepButton();

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::system::SleepButtonStateCountMax / 2 *
            ::nnt::hid::GetSleepButtonSamplingInterval().GetMilliSeconds()));

    // スリープボタンが完全に開放されたことで自動操作設定が解除される。
    ::nn::hid::system::GetSleepButtonStates(&g_States1[0], 1);
    {
        SCOPED_TRACE("");
        ExpectDefaultSleepButtonState(g_States1[0]);
    }

    ::nnt::hid::FinalizeSleepButtonForAutoPilot();

    ::nn::os::DestroySystemEvent(&event);
}

//!< スリープボタンの入力状態は不正な自動操作状態が設定されても仕様を保証するか
TEST(SleepButtonAutoPilotSuite, InvalidArgumentTest1)
{
    ::nn::hid::debug::SleepButtonAutoPilotState autoPilotState = {};

    // 無効なビットフラグも設定
    autoPilotState.buttons.Set( 0, true);
    autoPilotState.buttons.Set( 8, true);
    autoPilotState.buttons.Set(16, true);
    autoPilotState.buttons.Set(24, true);

    ::nn::hid::system::InitializeSleepButton();

    ::nn::hid::debug::SetSleepButtonAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::system::SleepButtonStateCountMax / 2 *
            ::nnt::hid::GetSleepButtonSamplingInterval().GetMilliSeconds()));

    ::nn::hid::system::SleepButtonState& state = g_States1[0];
    ::nn::hid::system::GetSleepButtonStates(&state, 1);

    ::nnt::hid::FinalizeSleepButtonForAutoPilot();

    // 有効なビットフラグのみ結果に反映
    EXPECT_TRUE (state.buttons.Test( 0));
    EXPECT_FALSE(state.buttons.Test( 8));
    EXPECT_FALSE(state.buttons.Test(16));
    EXPECT_FALSE(state.buttons.Test(24));
}

//!< スリープボタンの入力状態は自動操作状態の設定を正しく反映するか
TEST(SleepButtonAutoPilotSuite, StateReadingTest1)
{
    ::nn::os::SystemEventType event = {};

    ::nn::hid::system::BindSleepButtonEvent(&event,
                                            ::nn::os::EventClearMode_AutoClear);

    if (NN_STATIC_CONDITION(EnablesSignalAsserion))
    {
        EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&event));
    }

    ::nn::hid::system::InitializeSleepButton();

    if (NN_STATIC_CONDITION(EnablesSignalAsserion))
    {
        EXPECT_FALSE(::nn::os::TryWaitSystemEvent(&event));
    }

    ::nn::hid::system::GetSleepButtonStates(&g_States1[0], 1);
    {
        SCOPED_TRACE("");
        ExpectDefaultSleepButtonState(g_States1[0]);
    }

    ::nn::hid::debug::SleepButtonAutoPilotState autoPilotState;

    for (int i = 0; i <= ::nn::hid::system::SleepButtonStateCountMax / 3; ++i)
    {
        FuzzSleepButtonAutoPilotState(&autoPilotState, i + 1);
        ::nn::hid::debug::SetSleepButtonAutoPilotState(autoPilotState);
        ::nn::os::SleepThread(
            ::nn::TimeSpan::FromMilliSeconds(
                2 * ::nnt::hid::GetSleepButtonSamplingInterval()
                        .GetMilliSeconds()));
        if (NN_STATIC_CONDITION(EnablesSignalAsserion))
        {
            EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&event));
        }
    }

    int count = ::nn::hid::system::GetSleepButtonStates(
        g_States1, ::nn::hid::system::SleepButtonStateCountMax);
    EXPECT_LE(1, count);

    // 自動操作状態に応じてスリープボタンの入力状態は変化する。
    for (int i = 0; i < count; ++i)
    {
        const ::nn::hid::system::SleepButtonState& lhs = g_States1[i];

        EXPECT_LE(0, lhs.samplingNumber);

        if (i + 1 < count)
        {
            SCOPED_TRACE("");
            ExpectAdjacentSleepButtonStates(lhs, g_States1[i + 1]);
        }
    }

    ::nnt::hid::FinalizeSleepButtonForAutoPilot();

    ::nn::os::DestroySystemEvent(&event);
}

//!< スリープボタンの入力状態は自動操作状態の解除を正しく反映するか
TEST(SleepButtonAutoPilotSuite, StateReadingTest2)
{
    ::nn::os::SystemEventType event = {};

    ::nn::hid::system::BindSleepButtonEvent(&event,
                                            ::nn::os::EventClearMode_AutoClear);

    if (NN_STATIC_CONDITION(EnablesSignalAsserion))
    {
        EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&event));
    }

    ::nn::hid::system::InitializeSleepButton();

    ::nn::hid::debug::SleepButtonAutoPilotState autoPilotState;
    FuzzSleepButtonAutoPilotState(&autoPilotState, 1);

    if (NN_STATIC_CONDITION(EnablesSignalAsserion))
    {
        EXPECT_FALSE(::nn::os::TryWaitSystemEvent(&event));
    }

    ::nn::hid::debug::SetSleepButtonAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::system::SleepButtonStateCountMax *
            ::nnt::hid::GetSleepButtonSamplingInterval().GetMilliSeconds()));

    if (NN_STATIC_CONDITION(EnablesSignalAsserion))
    {
        EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&event));
    }

    // 自動操作状態を解除
    ::nn::hid::debug::UnsetSleepButtonAutoPilotState();

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::system::SleepButtonStateCountMax / 2 *
            ::nnt::hid::GetSleepButtonSamplingInterval().GetMilliSeconds()));

    if (NN_STATIC_CONDITION(EnablesSignalAsserion))
    {
        EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&event));
    }

    int count = ::nn::hid::system::GetSleepButtonStates(
        g_States1, ::nn::hid::system::SleepButtonStateCountMax);
    EXPECT_EQ(::nn::hid::system::SleepButtonStateCountMax, count);

    bool hasChanged = false;

    ::nn::hid::system::SleepButtonState state = {};

    // 入力状態列状で自動操作状態は連続的に切り替わる。
    for (int i = 0; i < count; ++i)
    {
        if (i + 1 < count)
        {
            const ::nn::hid::system::SleepButtonState& lhs = g_States1[i];
            const ::nn::hid::system::SleepButtonState& rhs = g_States1[i + 1];
            SCOPED_TRACE("");
            EXPECT_EQ(1, lhs.samplingNumber - rhs.samplingNumber);
        }

        if (g_States1[i].buttons != state.buttons)
        {
            EXPECT_FALSE(hasChanged);
            hasChanged = true;
            state.buttons = autoPilotState.buttons;
        }

        {
            SCOPED_TRACE("");
            ExpectSameOriginSleepButtonStates(state, g_States1[i]);
        }
    }

    EXPECT_TRUE(hasChanged);

    ::nnt::hid::FinalizeSleepButtonForAutoPilot();

    ::nn::os::DestroySystemEvent(&event);
}

//!< LIFO の境界を跨いで自動操作状態の設定を正しく反映するか
TEST(SleepButtonAutoPilotSuite, StateReadingTest3)
{
    ::nn::hid::system::InitializeSleepButton();

    ::nn::hid::debug::SleepButtonAutoPilotState autoPilotState;

    int lastCount = 0;

    // LIFO の境界を 2 回跨ぐまでサンプリングを継続
    for (int i = 0; i < ::nn::hid::system::SleepButtonStateCountMax * 3; ++i)
    {
        FuzzSleepButtonAutoPilotState(&autoPilotState, i);
        ::nn::hid::debug::SetSleepButtonAutoPilotState(autoPilotState);
        ::nn::os::SleepThread(
            ::nn::TimeSpan::FromMilliSeconds(
                ::nnt::hid::GetSleepButtonSamplingInterval()
                    .GetMilliSeconds()));

        // 今回のサンプリング結果を格納するバッファ
        ::nn::hid::system::SleepButtonState *ptr1 =
              (i % 2 == 1) ? &g_States1[0] : &g_States2[0];

        // 前回のサンプリング結果が格納されているバッファ
        ::nn::hid::system::SleepButtonState *ptr2 =
              (i % 2 == 1) ? &g_States2[0] : &g_States1[0];

        // 取得される入力状態の数は単調増加する。
        int count = ::nn::hid::system::GetSleepButtonStates(
            ptr1, ::nn::hid::system::SleepButtonStateCountMax);
        EXPECT_LE(lastCount, count);

        for (int j = 0; j < count; ++j)
        {
            if (j + 1 < count)
            {
                SCOPED_TRACE("");
                ExpectAdjacentSleepButtonStates(ptr1[j], ptr1[j + 1]);
            }

            // 前回のサンプリング結果と重複する箇所には同じ値が入る。
            if (lastCount > 0 &&
                ptr1[j].samplingNumber == ptr2[0].samplingNumber)
            {
                for (int k = 0; k < lastCount - j; ++k)
                {
                    SCOPED_TRACE("");
                    EXPECT_EQ(ptr2[k].samplingNumber,
                              ptr1[j + k].samplingNumber);
                    ExpectSameOriginSleepButtonStates(ptr2[k], ptr1[j + k]);
                }

                break;
            }
        }

        lastCount = count;
    }

    EXPECT_EQ(::nn::hid::system::SleepButtonStateCountMax, lastCount);

    ::nnt::hid::FinalizeSleepButtonForAutoPilot();
}

namespace
{

void FuzzSleepButtonAutoPilotState(
    ::nn::hid::debug::SleepButtonAutoPilotState* pState, int seed) NN_NOEXCEPT
{
    pState->buttons.Reset();
    pState->buttons.Set<::nn::hid::system::SleepButton::Active>(seed % 2 == 1);
}

void ExpectDefaultSleepButtonState(
    const ::nn::hid::system::SleepButtonState& state) NN_NOEXCEPT
{
    EXPECT_LE(0, state.samplingNumber);
    EXPECT_TRUE(state.buttons.IsAllOff());
}

void ExpectFuzzedSleepButtonState(
    const ::nn::hid::system::SleepButtonState& state) NN_NOEXCEPT
{
    EXPECT_LE(0, state.samplingNumber);
    ::nn::hid::system::SleepButtonSet buttons = {};
    buttons.Set<::nn::hid::system::SleepButton::Active>(true);
    EXPECT_EQ(buttons, state.buttons);
}

void ExpectSameOriginSleepButtonStates(
    const ::nn::hid::system::SleepButtonState& lhs,
    const ::nn::hid::system::SleepButtonState& rhs) NN_NOEXCEPT
{
    EXPECT_EQ(lhs.buttons, rhs.buttons);
}

void ExpectAdjacentSleepButtonStates(
    const ::nn::hid::system::SleepButtonState& lhs,
    const ::nn::hid::system::SleepButtonState& rhs) NN_NOEXCEPT
{
    EXPECT_EQ(1, lhs.samplingNumber - rhs.samplingNumber);
}

} // namespace
