﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits> // for numeric_limits

#include <nn/nn_Log.h>
#include <nn/nn_SdkAssert.h>

#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/xcd/xcd_Input.h>

#include <nnt/nntest.h>

// 各軸正方向回転させた後しばらく静止させたときのデータ系列
#include "Resources/SixAxisSensorProcessor/testHid_RotateBasic.h"

#include "../../../Programs/Iris/Sources/Libraries/hid/detail/hid_SixAxisSensorAppletSetting.h"
#include "../../../Programs/Iris/Sources/Libraries/hid/detail/hid_SixAxisSensorProcessor.h"

namespace {

const int SixAxisSensorInputCount = 6; // 6軸入力の数
const int ConvertFsrFrom8gTo2g = 4; // FSR 8G の単位系のデータを 2G の単位系に変換する係数

// データ系列から nn::xcd::SixAxisSensorState を取得します。
void GetSixAxisSensorState(nn::xcd::SixAxisSensorState* pOutSixAxisSensorState,
                           const int16_t* pData) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutSixAxisSensorState);

    int index = 0;

    pOutSixAxisSensorState->accelerometer.x = pData[index++];
    pOutSixAxisSensorState->accelerometer.y = pData[index++];
    pOutSixAxisSensorState->accelerometer.z = pData[index++];
    pOutSixAxisSensorState->gyroscope.x = pData[index++];
    pOutSixAxisSensorState->gyroscope.y = pData[index++];
    pOutSixAxisSensorState->gyroscope.z = pData[index++];
}

// SixAxisSensorProcessor を初期化します。
void Initialize(nn::hid::detail::SixAxisSensorProcessor* pProcessor,
                nn::hid::detail::SixAxisSensorSetting* pSetting)
{
    pSetting->accelerometerSetting.Reset(true);
    pSetting->gyroscopeSetting.Reset(true);
    pSetting->sixAxisSensorProcessorSetting.Reset(true);
    pSetting->sixAxisSensorProcessorSetting.EnableSampling(true);

    pProcessor->SetSixAxisSensorSetting(pSetting);

    // キャリブレーション値を設定
    nn::xcd::SensorCalibrationValue value = {{0, 0, 0},
                                             {16384, 16384, 16384},
                                             {0, 0, 0},
                                             {13371, 13371, 13371}};
    pProcessor->SetSensorCalibrationValue(value);
}

} // namespace

//!< SixAxisSensorProcessor の初期状態が仕様通りか
TEST(SixAxisSensorProcessor, InitialState)
{
    nn::hid::detail::SixAxisSensorSetting setting; // 6軸センサーの設定値
    nn::hid::detail::SixAxisSensorProcessor processor;

    setting.accelerometerSetting.Reset(true);
    setting.gyroscopeSetting.Reset(true);
    setting.sixAxisSensorProcessorSetting.Reset(true);

    processor.SetSixAxisSensorSetting(&setting);

    // 初期状態の確認
    EXPECT_TRUE(!setting.sixAxisSensorProcessorSetting.IsBaseDirectionEnabled());
    EXPECT_TRUE(setting.accelerometerSetting.IsSixAxisSensorFusionEnabled());

    nn::xcd::SensorCalibrationValue value = {};
    processor.SetSensorCalibrationValue(value);

    setting.sixAxisSensorProcessorSetting.EnableBaseDirection(true);
    EXPECT_TRUE(setting.sixAxisSensorProcessorSetting.IsBaseDirectionEnabled());

    // IsAtRest は別途

    setting.accelerometerSetting.EnableSixAxisSensorFusion(true);
    EXPECT_TRUE(setting.accelerometerSetting.IsSixAxisSensorFusionEnabled());
}

//!< SixAxisSensorProcessor がコントローラの静止状態を検出するか
TEST(SixAxisSensorProcessor, IsAtRest)
{
    nn::hid::detail::SixAxisSensorProcessor processor;
    nn::hid::detail::SixAxisSensorSetting setting; // 6軸センサーの設定値
    Initialize(&processor, &setting);

    EXPECT_TRUE(!processor.IsSixAxisSensorAtRest());

    for(int i = 0; i < 500; i++)
    {
        // データ生成
        nn::xcd::SixAxisSensorState state = {};
        state.accelerometer.x = 1;
        state.accelerometer.y = 1;
        state.accelerometer.z = 1;
        state.gyroscope.x = 0;
        state.gyroscope.y = 0;
        state.gyroscope.z = 0;
        state.sampleNumber = i;

        // データ注入
        processor.Sampling(state);
    }

    EXPECT_TRUE(processor.IsSixAxisSensorAtRest());
}

//!< SixAxisSensorProcessor がコントローラの回転を検出するか
TEST(SixAxisSensorProcessor, RotateBasic)
{
    const int TotalInputSequenceCount = sizeof(nnt::hid::TestRotateBasic) / sizeof(nnt::hid::TestRotateBasic[0]);
    const int SixAxisSensorStateCountMax = TotalInputSequenceCount / SixAxisSensorInputCount;
    const int16_t* pTestData = nnt::hid::TestRotateBasic; // 各軸正方向に回転させた時のコントローラ入力系列

    nn::hid::detail::SixAxisSensorProcessor processor;
    nn::hid::detail::SixAxisSensorSetting setting; // 6軸センサーの設定値
    Initialize(&processor, &setting);

    // 入力データを注入
    for(int i = 0; i < SixAxisSensorStateCountMax; i++)
    {
        nn::xcd::SixAxisSensorState state;
        int index = SixAxisSensorInputCount * i;
        GetSixAxisSensorState(&state, &pTestData[index]);
        state.sampleNumber = i;
        state.deltaTimeInSec = 0.005f;

        processor.Sampling(state);
    }

    nn::hid::SixAxisSensorState state;
    EXPECT_TRUE(processor.GetSixAxisSensorStates(&state, 1) == 1);

    NN_LOG("Rotate Angle : (%.2f, %.2f, %.2f\n)"
        , state.angle.x
        , state.angle.y
        , state.angle.z
    );

    EXPECT_LT(0.5f, state.angle.x);
    EXPECT_LT(0.5f, state.angle.y);
    EXPECT_LT(0.5f, state.angle.z);
}

//!< SixAxisSensorProcessor のクランプ処理ができているか
TEST(SixAxisSensorProcessor, ClampValueMax)
{
    const nn::xcd::SensorState AccelerometerMax = {
        std::numeric_limits<int16_t>::max() * ConvertFsrFrom8gTo2g,
        std::numeric_limits<int16_t>::max() * ConvertFsrFrom8gTo2g,
        std::numeric_limits<int16_t>::max() * ConvertFsrFrom8gTo2g
    };

    const nn::xcd::SensorState GyroscopeMax = {
        std::numeric_limits<int16_t>::max(),
        std::numeric_limits<int16_t>::max(),
        std::numeric_limits<int16_t>::max()
    };

    // 加工済みセンサー値を取得
    nn::xcd::SixAxisSensorState state = {};

    state.accelerometer = ::nn::util::Float3{{{
        static_cast<float>(AccelerometerMax.x),
        static_cast<float>(AccelerometerMax.y),
        static_cast<float>(AccelerometerMax.z)
    }}};

    state.gyroscope = ::nn::util::Float3{{{
        static_cast<float>(GyroscopeMax.x),
        static_cast<float>(GyroscopeMax.y),
        static_cast<float>(GyroscopeMax.z)
    }}};
    state.sampleNumber = 1;

    nn::hid::detail::SixAxisSensorProcessor processor;
    nn::hid::detail::SixAxisSensorSetting setting; // 6軸センサーの設定値
    Initialize(&processor, &setting);

    processor.Sampling(state);

    ::nn::hid::SixAxisSensorState hidState = {};
    processor.GetSixAxisSensorStates(&hidState, 1);

    // 6軸センサーの値が規定値にクランプされているか
    for(const auto& accelerometer : hidState.acceleration.v)
    {
        EXPECT_EQ(accelerometer, nn::hid::AccelerometerMax);
    }
    for(const auto& angularVelocity : hidState.angularVelocity.v)
    {
        EXPECT_EQ(angularVelocity, nn::hid::AngularVelocityMax);
    }
}

//!< SixAxisSensorProcessor のクランプ処理ができているか
TEST(SixAxisSensorProcessor, ClampValueMin)
{
    // PadState の生成
    const nn::xcd::SensorState AccelerometerMin = {
        std::numeric_limits<int16_t>::min() * ConvertFsrFrom8gTo2g,
        std::numeric_limits<int16_t>::min() * ConvertFsrFrom8gTo2g,
        std::numeric_limits<int16_t>::min() * ConvertFsrFrom8gTo2g
    };

    const nn::xcd::SensorState GyroscopeMin = {
        std::numeric_limits<int16_t>::min(),
        std::numeric_limits<int16_t>::min(),
        std::numeric_limits<int16_t>::min()
    };

    // 加工済みセンサー値を取得
    nn::xcd::SixAxisSensorState state = {};

    state.accelerometer = ::nn::util::Float3{{{
        static_cast<float>(AccelerometerMin.x),
        static_cast<float>(AccelerometerMin.y),
        static_cast<float>(AccelerometerMin.z)
    }}};

    state.gyroscope = ::nn::util::Float3{{{
        static_cast<float>(GyroscopeMin.x),
        static_cast<float>(GyroscopeMin.y),
        static_cast<float>(GyroscopeMin.z)
    }}};
    state.sampleNumber = 1;

    nn::hid::detail::SixAxisSensorProcessor processor;
    nn::hid::detail::SixAxisSensorSetting setting; // 6軸センサーの設定値
    Initialize(&processor, &setting);

    processor.Sampling(state);

    ::nn::hid::SixAxisSensorState hidState = {};
    processor.GetSixAxisSensorStates(&hidState, 1);

    // 6軸センサーの値が規定値にクランプされているか
    for(const auto& accelerometer : hidState.acceleration.v)
    {
        EXPECT_EQ(accelerometer, -nn::hid::AccelerometerMax);
    }
    for(const auto& angularVelocity : hidState.angularVelocity.v)
    {
        EXPECT_EQ(angularVelocity, -nn::hid::AngularVelocityMax);
    }
}
