﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_NpadSixAxisSensor.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/hid/debug/hid_Npad.h>
#include <nn/hid/system/hid_Npad.h>

#include <nn/TargetConfigs/build_Base.h>
#include <nnt/nntest.h>

#include "../../../../../Programs/Iris/Sources/Libraries/hid/detail/hid_SixAxisSensor.h" // 公開ヘッダ更新後、削除予定

namespace
{
    nn::hid::NpadIdType g_NpadIds[] = { nn::hid::NpadId::No1,
                                        nn::hid::NpadId::No2,
                                        nn::hid::NpadId::No3,
                                        nn::hid::NpadId::No4,
                                        nn::hid::NpadId::No5,
                                        nn::hid::NpadId::No6,
                                        nn::hid::NpadId::No7,
                                        nn::hid::NpadId::No8 };

    const int NpadIdCountMax = sizeof(g_NpadIds) / sizeof(g_NpadIds[0]);

    nn::hid::SixAxisSensorState g_States[nn::hid::SixAxisSensorStateCountMax];
    nn::hid::SixAxisSensorHandle g_Handles[nn::hid::NpadSixAxisSensorHandleCountMax];

    void NoInitializationTest(const nn::hid::NpadStyleSet styleSet)
    {
        // 未初期化なので失敗する
        for(auto& id : g_NpadIds)
        {
            int handleCount = nn::hid::GetSixAxisSensorHandles(g_Handles,
                                                               nn::hid::NpadSixAxisSensorHandleCountMax,
                                                               id,
                                                               styleSet);

            for (int i = 0; i < handleCount; i++)
            {
                EXPECT_DEATH_IF_SUPPORTED(nn::hid::GetSixAxisSensorStates(g_States,
                                                                          nn::hid::SixAxisSensorStateCountMax,
                                                                          g_Handles[i]), "");
            }
        }

        // Npad の初期化
        nn::hid::InitializeNpad();
        nn::hid::SetSupportedNpadIdType(g_NpadIds, NpadIdCountMax);

        for (auto& id : g_NpadIds)
        {
            int handleCount = nn::hid::GetSixAxisSensorHandles(g_Handles,
                                                               nn::hid::NpadSixAxisSensorHandleCountMax,
                                                               id,
                                                               styleSet);
            // 初期化済なので成功する
            for (int i = 0; i < handleCount; i++)
            {
                int stateCount = nn::hid::GetSixAxisSensorStates(g_States,
                                                                 nn::hid::SixAxisSensorStateCountMax,
                                                                 g_Handles[i]);
                EXPECT_LE(0, stateCount);
            }

        }

        nn::hid::debug::FinalizeNpad();
    }

    void StateReadingTest(const nn::hid::NpadStyleSet styleSet, int expectedHandleCounts)
    {
        nn::hid::InitializeNpad();

        nn::hid::SetSupportedNpadIdType(g_NpadIds, NpadIdCountMax);

        for(auto& id : g_NpadIds)
        {
            int handleCount = nn::hid::GetSixAxisSensorHandles(g_Handles,
                                                           nn::hid::NpadSixAxisSensorHandleCountMax,
                                                           id,
                                                           styleSet);
            // 取得した handle の数が所望の値になっているか
            EXPECT_EQ(handleCount, expectedHandleCounts);

            for(int i = 0; i < handleCount; i++)
            {
                nn::hid::StartSixAxisSensor(g_Handles[i]);

                // デバイスの状態に依らず GetSixAxisSensorStates() の呼出しが可能か
                int stateCount = nn::hid::GetSixAxisSensorStates(g_States,
                                                                 nn::hid::SixAxisSensorStateCountMax,
                                                                 g_Handles[i]);
                EXPECT_LE(0, stateCount);
            }
        }
    }

    void SettingTest(const nn::hid::NpadStyleSet styleSet)
    {
        nn::hid::InitializeNpad();

        nn::hid::SetSupportedNpadIdType(g_NpadIds, NpadIdCountMax);

        for (auto& id : g_NpadIds)
        {
            auto handleCount = nn::hid::GetSixAxisSensorHandles(g_Handles,
                                                                nn::hid::NpadSixAxisSensorHandleCountMax,
                                                                id,
                                                                styleSet);

            for(int i = 0; i < handleCount; i++)
            {
                const auto handle = g_Handles[i];

                // [6軸センサーのセンサーフュージョン関連] -------------------------------------------------------
                // デフォルト ON に設定されているか
                ASSERT_TRUE(nn::hid::IsSixAxisSensorFusionEnabled(handle));

                // 正しく OFF に設定されているか
                nn::hid::EnableSixAxisSensorFusion(handle, false);
                ASSERT_TRUE(!nn::hid::IsSixAxisSensorFusionEnabled(handle));

                // 正しく ON に設定されているか(複数回連続で実行された際の対処のためデフォルト設定に書き戻します)
                nn::hid::EnableSixAxisSensorFusion(handle, true);
                ASSERT_TRUE(nn::hid::IsSixAxisSensorFusionEnabled(handle));

                float revisePower;
                float reviseRange;
                const float DefaultRevisePower = 0.030f;
                const float DefaultReviseRange = 0.400f;
                const float TestRevisePower = 0.020f;
                const float TestReviseRange = 0.100f;

                // デフォルト設定値は仕様通りか
                ::nn::hid::GetSixAxisSensorFusionParameters(&revisePower,
                                                            &reviseRange,
                                                            handle);
                EXPECT_EQ(revisePower, DefaultRevisePower);
                EXPECT_EQ(reviseRange, DefaultReviseRange);

                // 設定値を変更できるか
                ::nn::hid::SetSixAxisSensorFusionParameters(handle,
                                                            TestRevisePower,
                                                            TestReviseRange);
                ::nn::hid::GetSixAxisSensorFusionParameters(&revisePower,
                                                            &reviseRange,
                                                            handle);
                EXPECT_EQ(revisePower, TestRevisePower);
                EXPECT_EQ(reviseRange, TestReviseRange);

                // 事前条件は有効か
                EXPECT_DEATH_IF_SUPPORTED(::nn::hid::SetSixAxisSensorFusionParameters(handle,
                                          1.1f,
                                          TestReviseRange), "");
                EXPECT_DEATH_IF_SUPPORTED(::nn::hid::SetSixAxisSensorFusionParameters(handle,
                                          -0.1f,
                                          TestReviseRange), "");

                // 設定値リセットできるか
                ResetSixAxisSensorFusionParameters(handle);
                ::nn::hid::GetSixAxisSensorFusionParameters(&revisePower,
                                                            &reviseRange,
                                                            handle);
                EXPECT_EQ(revisePower, DefaultRevisePower);
                EXPECT_EQ(reviseRange, DefaultReviseRange);

                // [加速度センサーの補正パラメータ関連] -------------------------------------------------------
                float playRadius;
                float sensitivity;
                const float DefaultPlayRadius = 0.0f;
                const float DefaultSensitivity = 1.0f;
                const float TestPlayRadius = 0.020f;
                const float TestSensitivity = 0.100f;

                // デフォルト設定値は仕様通りか
                ::nn::hid::GetAccelerometerParameters(&playRadius,
                                                      &sensitivity,
                                                      handle);
                EXPECT_EQ(playRadius, DefaultPlayRadius);
                EXPECT_EQ(sensitivity, DefaultSensitivity);

                // 設定値を変更できるか
                ::nn::hid::SetAccelerometerParameters(handle,
                                                      TestPlayRadius,
                                                      TestSensitivity);
                ::nn::hid::GetAccelerometerParameters(&playRadius,
                                                      &sensitivity,
                                                      handle);
                EXPECT_EQ(playRadius, TestPlayRadius);
                EXPECT_EQ(sensitivity, TestSensitivity);

                // 事前条件は有効か
                EXPECT_DEATH_IF_SUPPORTED(::nn::hid::SetAccelerometerParameters(handle,
                                          -0.1f,
                                          TestSensitivity), "");
                EXPECT_DEATH_IF_SUPPORTED(::nn::hid::SetAccelerometerParameters(handle,
                                          TestPlayRadius,
                                          1.1f), "");

                // 設定値リセットできるか
                ::nn::hid::ResetAccelerometerParameters(handle);
                ::nn::hid::GetAccelerometerParameters(&playRadius,
                                                      &sensitivity,
                                                      handle);
                EXPECT_EQ(playRadius, DefaultPlayRadius);
                EXPECT_EQ(sensitivity, DefaultSensitivity);

                // [加速度センサーについて遊びの挙動方法関連] -------------------------------------------------------
                ::nn::hid::AccelerometerPlayMode playMode;
                const ::nn::hid::AccelerometerPlayMode DefaultPlayMode = ::nn::hid::AccelerometerPlayMode_Tight;
                const ::nn::hid::AccelerometerPlayMode TestPlayMode = ::nn::hid::AccelerometerPlayMode_Loose;

                // デフォルト設定値は仕様通りか
                ::nn::hid::GetAccelerometerPlayMode(&playMode, handle);
                EXPECT_EQ(playMode, DefaultPlayMode);

                // 設定値を変更できるか
                ::nn::hid::SetAccelerometerPlayMode(handle, TestPlayMode);
                ::nn::hid::GetAccelerometerPlayMode(&playMode, handle);
                EXPECT_EQ(playMode, TestPlayMode);

                // 設定値リセットできるか
                ::nn::hid::ResetAccelerometerPlayMode(handle);
                ::nn::hid::GetAccelerometerPlayMode(&playMode, handle);
                EXPECT_EQ(playMode, DefaultPlayMode);

                // [ゼロ点のドリフト補正モード関連] -------------------------------------------------------
                ::nn::hid::GyroscopeZeroDriftMode driftMode;
                const ::nn::hid::GyroscopeZeroDriftMode DefaultDriftMode = ::nn::hid::GyroscopeZeroDriftMode_Standard;
                const ::nn::hid::GyroscopeZeroDriftMode TestDriftMode = ::nn::hid::GyroscopeZeroDriftMode_Tight;

                // デフォルト設定値は仕様通りか
                driftMode = ::nn::hid::GetGyroscopeZeroDriftMode(handle);
                EXPECT_EQ(driftMode, DefaultDriftMode);

                // 設定値を変更できるか
                ::nn::hid::SetGyroscopeZeroDriftMode(handle, TestDriftMode);
                driftMode = ::nn::hid::GetGyroscopeZeroDriftMode(handle);
                EXPECT_EQ(driftMode, TestDriftMode);

                // 初期状態に戻すため設定値を書き戻し
                ::nn::hid::SetGyroscopeZeroDriftMode(handle, DefaultDriftMode);
                driftMode = ::nn::hid::GetGyroscopeZeroDriftMode(handle);
                EXPECT_EQ(driftMode, DefaultDriftMode);
            }
        }
    } // NOLINT(impl/function_size)
}

//!< Npad の初期化状態は GetSixAxisSensorHandle(), GetSixAxisSensorState() の呼び出しに反映されるか
TEST(NpadSixAxisSensorInitialization, NoInitializationTest1)
{
    NoInitializationTest(nn::hid::NpadStyleFullKey::Mask);
    NoInitializationTest(nn::hid::NpadStyleJoyDual::Mask);
    NoInitializationTest(nn::hid::NpadStyleJoyLeft::Mask);
    NoInitializationTest(nn::hid::NpadStyleJoyRight::Mask);
}

//!< SixAxisSensor の初期化処理は最低一つの入力状態の取得を保証するか
TEST(NpadSixAxisSensorInitialization, StateReadingTest1)
{
    StateReadingTest(nn::hid::NpadStyleFullKey::Mask, 1);
    StateReadingTest(nn::hid::NpadStyleJoyDual::Mask, 2);
    StateReadingTest(nn::hid::NpadStyleJoyLeft::Mask, 1);
    StateReadingTest(nn::hid::NpadStyleJoyRight::Mask, 1);
}

//!< SixAxisSensor の設定値変更が仕様通りか
TEST(NpadSixAxisSensorSetting, SettingTest1)
{
    SettingTest(nn::hid::NpadStyleFullKey::Mask);
    SettingTest(nn::hid::NpadStyleJoyDual::Mask);
    SettingTest(nn::hid::NpadStyleJoyLeft::Mask);
    SettingTest(nn::hid::NpadStyleJoyRight::Mask);
}
