﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/system/hid_Npad.h>
#include <nnt/nntest.h>

#include "testHid_NpadCommon.h"
#include "../Common/testHid_Npad.h"


namespace
{
void TestNpadStyleCombination(nn::hid::NpadStyleSet style, int step)
{
    step++;
    for (int i = 0; i < 2; i++)
    {
        if (i == 1)
        {
            style |= ::testhid::NpadStyles[step - 1];
        }

        if (step == ::testhid::NpadStyleCountMax)
        {
            nn::hid::SetSupportedNpadStyleSet(style);

            auto currentSupportedStyle = nn::hid::GetSupportedNpadStyleSet();

            EXPECT_EQ(currentSupportedStyle, style);

            // 設定した SupportedStyle は各 Npad 毎の現行の Style には影響を与えない
            // Stlye は未接続時は AllOff
            for (auto& npadId : ::testhid::NpadIds)
            {
                auto currentStyle = nn::hid::GetNpadStyleSet(npadId);
                auto deviceType = nn::hid::system::GetNpadDeviceType(npadId);
                if (deviceType.IsAllOff())
                {
                    EXPECT_TRUE(currentStyle.IsAllOff());
                }
            }
        }
        else
        {
            TestNpadStyleCombination(style, step);
        }
    }
}

}

//!< NpadId 未設定時は現在のスタイル読み出し / イベントのバインドに失敗する
TEST(NpadSupportedStyleTest, NpadNotInitialized)
{
    nn::os::SystemEventType systemEvents[::testhid::NpadIdCountMax];

    for (int i = 0; i < ::testhid::NpadIdMinimalSetCountMax; i++)
    {
        auto& npadId = ::testhid::NpadIdsMinimalSet[i];
        EXPECT_DEATH_IF_SUPPORTED(
            nn::hid::GetNpadStyleSet(npadId), "");
        EXPECT_DEATH_IF_SUPPORTED(
            nn::hid::BindNpadStyleSetUpdateEvent(npadId, &systemEvents[i], ::nn::os::EventClearMode_AutoClear), "");
    }
}

//!< スタイルの設定が正しくできるか
TEST(NpadSupportedStyleTest, StyleMatrixTest)
{
    ::testhid::EnableIdAll();

    auto style = nn::hid::NpadStyleSet();
    TestNpadStyleCombination(style, 0);

    ::testhid::DisableIdAll();
}

//!< 未知のスタイルの設定が正しく行えるか
TEST(NpadSupportedStyleTest, FutureStyle)
{
    ::testhid::EnableIdAll();

    auto style = nn::hid::NpadStyleSet();
    style.Reset();
    style[20].Flip();
    nn::hid::SetSupportedNpadStyleSet(style);
    EXPECT_EQ(style, nn::hid::GetSupportedNpadStyleSet());

    // 全ビットフラグたてた状態では Failする
    style.Set();
    EXPECT_DEATH_IF_SUPPORTED(
        nn::hid::SetSupportedNpadStyleSet(style), "");

    ::testhid::DisableIdAll();
}

//!< Style 更新時の イベントの Bind
TEST(NpadSupportedStyleTest, BindStyleUpdateEvent)
{
    nn::os::SystemEventType systemEvents;
    nn::os::SystemEventType systemEventsSpare;

    ::testhid::EnableIdAll();

    nn::hid::NpadIdType npadId = nn::hid::NpadId::No1;

    nn::hid::BindNpadStyleSetUpdateEvent(npadId, &systemEvents, ::nn::os::EventClearMode_AutoClear);

    // 別のイベントを Bind し直すことができる
    nn::hid::BindNpadStyleSetUpdateEvent(npadId, &systemEventsSpare, ::nn::os::EventClearMode_ManualClear);

    ::testhid::DisableIdAll();

    // Disable 後に Bind すると Abort
    EXPECT_DEATH_IF_SUPPORTED(
        nn::hid::BindNpadStyleSetUpdateEvent(npadId, &systemEvents, ::nn::os::EventClearMode_AutoClear), "");
}
