﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/debug/hid_Npad.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/os.h>
#include <nnt/nntest.h>

#include "../Common/testHid_Npad.h"

namespace
{

nn::hid::NpadIdType g_NpadIds[] = {
                                   nn::hid::NpadId::No1,
                                   nn::hid::NpadId::No2,
                                   nn::hid::NpadId::No3,
                                   nn::hid::NpadId::No4,
                                   nn::hid::NpadId::No5,
                                   nn::hid::NpadId::No6,
                                   nn::hid::NpadId::No7,
                                   nn::hid::NpadId::No8,
                                   nn::hid::NpadId::Handheld,
                                   nn::hid::system::NpadSystemId::Other,
                                  };

const int NpadIdCountMax = sizeof(g_NpadIds) / sizeof(g_NpadIds[0]);

struct Npad
{
    typedef ::nn::hid::NpadIdType Id;
    typedef ::nn::hid::NpadFullKeyState State;
    typedef ::nn::hid::NpadAttributesSet AttributeSet;
    typedef ::nn::hid::NpadButtonSet ButtonSet;
    static State s_States[::nn::hid::NpadStateCountMax];
};

//!< 入力状態の読み出し時にバッファが正しく利用されるかテストします。
template<typename T>
void ReadingStateTest(int npadIdIndex) NN_NOEXCEPT
{
    const typename T::Id& npadId = g_NpadIds[npadIdIndex];

    for (typename T::State& state : T::s_States)
    {
        state = typename T::State();
        state.samplingNumber = 0xDEADBEEF;
        state.attributes.Set(1, true);
        state.buttons.Set(2, true);
        state.analogStickR.x = 0x2222;
        state.analogStickR.y = 0x3333;
        state.analogStickL.x = 0x4444;
        state.analogStickL.y = 0x5555;
    }

    nn::hid::InitializeNpad();
    nn::hid::SetSupportedNpadIdType(&npadId, 1);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::NpadStateCountMax / 2 *
            ::nnt::hid::GetNpadSamplingInterval().GetMilliSeconds()));

    int count = ::nn::hid::GetNpadStates(T::s_States,
                                         ::nn::hid::NpadStateCountMax,
                                         npadId);

    EXPECT_GE(::nn::hid::NpadStateCountMax, count);

    for (int i = count; i < ::nn::hid::NpadStateCountMax; ++i)
    {
        // 入力状態の書き込まれなかった領域は値を維持する。
        EXPECT_EQ(0xDEADBEEF, T::s_States[i].samplingNumber);
        typename T::AttributeSet attributes = {};
        attributes.Set(1, true);
        typename T::ButtonSet buttons = {};
        buttons.Set(2, true);
        EXPECT_EQ(attributes, T::s_States[i].attributes);
        EXPECT_EQ(buttons, T::s_States[i].buttons);
        EXPECT_EQ(0x2222, T::s_States[i].analogStickR.x);
        EXPECT_EQ(0x3333, T::s_States[i].analogStickR.y);
        EXPECT_EQ(0x4444, T::s_States[i].analogStickL.x);
        EXPECT_EQ(0x5555, T::s_States[i].analogStickL.y);
    }

    nn::hid::debug::FinalizeNpad();
}

//!< 指定の優先度のスレッドから入力状態の読み出しが可能かテストします。
template<typename TState>
void ThreadingTest(int npadIdIndex, int priority) NN_NOEXCEPT
{
    ::nn::os::ThreadType* pThread = ::nn::os::GetCurrentThread();

    // 指定の優先度に設定する。
    const int original = ::nn::os::ChangeThreadPriority(pThread, priority);

    nn::hid::InitializeNpad();

    nn::hid::SetSupportedNpadIdType(g_NpadIds, NpadIdCountMax);

    int lastCount = 1;
    int64_t lastSamplingNumber = 0;
    bool isConnected = false;

    for (int i = 0; i < 60; ++i)
    {
        int count = ::nn::hid::GetNpadStates(TState::s_States,
                                             ::nn::hid::NpadStateCountMax,
                                             g_NpadIds[npadIdIndex]);
        EXPECT_LE(lastCount, count);

        EXPECT_LE(lastSamplingNumber, TState::s_States[0].samplingNumber);

        for (int j = 0; j < count - 1; ++j)
        {
            const typename TState::State& lhs = TState::s_States[j];
            const typename TState::State& rhs = TState::s_States[j + 1];
            EXPECT_EQ(1, lhs.samplingNumber - rhs.samplingNumber);
        }
        isConnected = (TState::s_States[0].attributes & nn::hid::NpadAttribute::IsConnected::Mask).IsAnyOn();
        lastCount = count;
        lastSamplingNumber = TState::s_States[0].samplingNumber;

        ::nn::os::SleepThread(::nn::TimeSpan::FromMilliSeconds(16));
    }
    if (isConnected)
    {
        EXPECT_EQ(::nn::hid::NpadStateCountMax, lastCount);
    }
    else
    {
        EXPECT_LE(1, lastCount);
    }

    nn::hid::debug::FinalizeNpad();

    // 優先度を元に戻す。
    ::nn::os::ChangeThreadPriority(pThread, original);
}

} // namespace

//!< 入力状態の読み出し時にバッファは正しく利用されるか
TEST(NpadReadingState, ReadingStateTest1)
{
    ReadingStateTest<Npad>(0);
}

//!< 高優先度のスレッドから入力状態の読み出しが可能か
TEST(NpadReadingState, ThreadingTest1)
{
    ThreadingTest<Npad>(0, ::nn::os::HighestThreadPriority);
}

//!< 低優先度のスレッドから入力状態の読み出しが可能か
TEST(NpadReadingState, ThreadingTest2)
{
    ThreadingTest<Npad>(0, ::nn::os::LowestThreadPriority);
}

namespace
{
    ::nn::hid::NpadFullKeyState Npad::s_States[::nn::hid::NpadStateCountMax];
}
