﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_TimeSpan.h>
#include <nn/hid/hid_Keyboard.h>
#include <nn/hid/debug/hid_Keyboard.h>
#include <nn/os.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nnt/nntest.h>

#include "../Common/testHid_Keyboard.h"

namespace
{
::nn::hid::KeyboardState g_State;
} // namespace

//!< Keyboard の初期化状態は GetKeyboardState() の呼び出しに反映されるか
TEST(KeyboardInitializationSuite, NoInitializationTest1)
{
    // Keyboard が未初期化の状態で呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(::nn::hid::GetKeyboardState(&g_State), "");

    ::nn::hid::InitializeKeyboard();

    ::nn::hid::InitializeKeyboard();

    ::nn::hid::debug::FinalizeKeyboard();

    // Keyboard が終了しきっていないので成功する。
    ::nn::hid::GetKeyboardState(&g_State);

    ::nn::hid::debug::FinalizeKeyboard();

    // Keyboard の終了処理後に呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(::nn::hid::GetKeyboardState(&g_State), "");
}

//!< Keyboard の初期化状態は GetKeyboardStates() の呼び出しに反映されるか
TEST(KeyboardInitializationSuite, NoInitializationTest2)
{
    // Keyboard が未初期化の状態で呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(::nn::hid::GetKeyboardStates(&g_State, 1), "");

    ::nn::hid::InitializeKeyboard();

    ::nn::hid::InitializeKeyboard();

    ::nn::hid::debug::FinalizeKeyboard();

    // Keyboard が終了しきっていないので成功する。
    ::nn::hid::GetKeyboardState(&g_State);

    ::nn::hid::debug::FinalizeKeyboard();

    // Keyboard の終了処理後に呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(::nn::hid::GetKeyboardStates(&g_State, 1), "");
}

//!< Keyboard の初期化状態は SetKeyboardAutoPilotState() の呼び出しに反映されるか
TEST(KeyboardInitializationSuite, NoInitializationTest3)
{
    ::nn::hid::debug::KeyboardAutoPilotState autoPilotState = {};

    // Keyboard が未初期化の状態で呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::debug::SetKeyboardAutoPilotState(autoPilotState), "");

    ::nn::hid::InitializeKeyboard();

    ::nn::hid::InitializeKeyboard();

    ::nn::hid::debug::FinalizeKeyboard();

    // Keyboard が終了しきっていないので成功する。
    ::nn::hid::debug::SetKeyboardAutoPilotState(autoPilotState);

    ::nnt::hid::FinalizeKeyboardForAutoPilot();

    // Keyboard の終了処理後に呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::debug::SetKeyboardAutoPilotState(autoPilotState), "");
}

//!< Keyboard の初期化状態は UnsetKeyboardAutoPilotState() の呼び出しに反映されるか
TEST(KeyboardInitializationSuite, NoInitializationTest4)
{
    // Keyboard が未初期化の状態で呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::debug::UnsetKeyboardAutoPilotState(), "");

    ::nn::hid::InitializeKeyboard();

    ::nn::hid::InitializeKeyboard();

    ::nn::hid::debug::FinalizeKeyboard();

    // Keyboard が終了しきっていないので成功する。
    ::nn::hid::debug::UnsetKeyboardAutoPilotState();

    ::nn::hid::debug::FinalizeKeyboard();

    // Keyboard の終了処理後に呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::debug::UnsetKeyboardAutoPilotState(), "");
}

//!< Keyboard の初期化処理は最低一つの入力状態の取得を保証するか
TEST(KeyboardInitializationSuite, StateReadingTest1)
{
    ::nn::hid::InitializeKeyboard();

    // 初期化直後の入力状態の取得に成功する。
    ::nn::hid::GetKeyboardState(&g_State);
    EXPECT_LE(0, g_State.samplingNumber);
    EXPECT_TRUE(g_State.keys.IsAllOff());
    EXPECT_FALSE(
        g_State.modifiers.Test<::nn::hid::KeyboardModifier::Control>());
    EXPECT_FALSE(
        g_State.modifiers.Test<::nn::hid::KeyboardModifier::Shift>());
    EXPECT_FALSE(
        g_State.modifiers.Test<::nn::hid::KeyboardModifier::LeftAlt>());
    EXPECT_FALSE(
        g_State.modifiers.Test<::nn::hid::KeyboardModifier::RightAlt>());
    EXPECT_FALSE(
        g_State.modifiers.Test<::nn::hid::KeyboardModifier::Gui>());

    ::nn::hid::debug::FinalizeKeyboard();
}

//!< Keyboard の初期化処理はサンプリングを正しく有効化するか
TEST(KeyboardInitializationSuite, StateReadingTest2)
{
    ::nn::hid::InitializeKeyboard();

    ::nn::hid::InitializeKeyboard();

    ::nn::hid::GetKeyboardState(&g_State);

    int64_t samplingNumber = g_State.samplingNumber;

    ::nn::hid::debug::FinalizeKeyboard();

    // Keyboard が終了しきっていないのでサンプリングは継続する。
    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::KeyboardStateCountMax *
            ::nnt::hid::GetKeyboardSamplingInterval().GetMilliSeconds()));

    ::nn::hid::GetKeyboardState(&g_State);

    EXPECT_LT(samplingNumber + ::nn::hid::KeyboardStateCountMax / 4,
              g_State.samplingNumber);

    ::nn::hid::debug::FinalizeKeyboard();
#if defined(NN_BUILD_CONFIG_OS_WIN)
    samplingNumber = g_State.samplingNumber;

    // Keyboard が終了しているのでサンプリングは停止する。
    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::KeyboardStateCountMax *
            ::nnt::hid::GetKeyboardSamplingInterval().GetMilliSeconds()));

    ::nn::hid::InitializeKeyboard();

    ::nn::hid::GetKeyboardState(&g_State);

    EXPECT_GT(samplingNumber + ::nn::hid::KeyboardStateCountMax / 4,
              g_State.samplingNumber);

    ::nn::hid::debug::FinalizeKeyboard();
#endif
}
