﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_TimeSpan.h>
#include <nn/hid/hid_Keyboard.h>
#include <nn/hid/debug/hid_Keyboard.h>
#include <nn/os.h>
#include <nnt/nntest.h>

#include "../Common/testHid_Keyboard.h"

namespace
{

::nn::hid::KeyboardState g_States1[::nn::hid::KeyboardStateCountMax];
::nn::hid::KeyboardState g_States2[::nn::hid::KeyboardStateCountMax];

//!< Keyboard の自動操作状態をファジングします。
void FuzzKeyboardAutoPilotState(
    ::nn::hid::debug::KeyboardAutoPilotState* pState, int seed) NN_NOEXCEPT;

//!< Keyboard の入力状態が入力無し状態であることを期待します。
void ExpectDefaultKeyboardState(const ::nn::hid::KeyboardState& state
                                ) NN_NOEXCEPT;

//!< Keyboard の入力状態がファジングされた入力状態であることを期待します。
void ExpectFuzzedKeyboardState(const ::nn::hid::KeyboardState& state
                               ) NN_NOEXCEPT;

//!< Keyboard の入力状態が同じドライバの状態から決定されたことを期待します。
void ExpectSameOriginKeyboardStates(const ::nn::hid::KeyboardState& lhs,
                                    const ::nn::hid::KeyboardState& rhs
                                    ) NN_NOEXCEPT;

//!< Keyboard の入力状態が隣り合っていることを期待します。
void ExpectAdjacentKeyboardStates(const ::nn::hid::KeyboardState& lhs,
                                  const ::nn::hid::KeyboardState& rhs
                                  ) NN_NOEXCEPT;

} // namespace

//!< Keyboard の初期化処理は初期化処理は自動操作状態を正しく管理するか
TEST(KeyboardAutoPilotSuite, InitializationTest1)
{
    ::nn::hid::debug::KeyboardAutoPilotState autoPilotState;
    FuzzKeyboardAutoPilotState(&autoPilotState, 1);

    ::nn::hid::InitializeKeyboard();

    ::nn::hid::debug::SetKeyboardAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::KeyboardStateCountMax / 2 *
            ::nnt::hid::GetKeyboardSamplingInterval().GetMilliSeconds()));

    // 自動操作が反映された入力状態が取得される。
    ::nn::hid::GetKeyboardState(&g_States1[0]);
    {
        SCOPED_TRACE("");
        ExpectFuzzedKeyboardState(g_States1[0]);
    }

    ::nn::hid::InitializeKeyboard();

    ::nn::hid::debug::FinalizeKeyboard();

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::KeyboardStateCountMax / 2 *
            ::nnt::hid::GetKeyboardSamplingInterval().GetMilliSeconds()));

    // まだユーザが存在するので自動操作設定は解除されない。
    ::nn::hid::GetKeyboardState(&g_States1[0]);
    {
        SCOPED_TRACE("");
        ExpectFuzzedKeyboardState(g_States1[0]);
    }

    ::nnt::hid::FinalizeKeyboardForAutoPilot();

    ::nn::hid::InitializeKeyboard();

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::KeyboardStateCountMax / 2 *
            ::nnt::hid::GetKeyboardSamplingInterval().GetMilliSeconds()));

    // Keyboard が完全に開放されたことで自動操作設定が解除される。
    ::nn::hid::GetKeyboardState(&g_States1[0]);
    {
        SCOPED_TRACE("");
        ExpectDefaultKeyboardState(g_States1[0]);
    }

    ::nnt::hid::FinalizeKeyboardForAutoPilot();
}

//!< Keyboard の入力状態は自動操作状態の設定を正しく反映するか
TEST(KeyboardAutoPilotSuite, StateReadingTest1)
{
    ::nn::hid::InitializeKeyboard();

    ::nn::hid::GetKeyboardState(&g_States1[0]);
    {
        SCOPED_TRACE("");
        ExpectDefaultKeyboardState(g_States1[0]);
    }

    ::nn::hid::debug::KeyboardAutoPilotState autoPilotState;

    for (int i = 0; i <= ::nn::hid::KeyboardStateCountMax / 3; ++i)
    {
        FuzzKeyboardAutoPilotState(&autoPilotState, i);
        ::nn::hid::debug::SetKeyboardAutoPilotState(autoPilotState);
        ::nn::os::SleepThread(
            ::nn::TimeSpan::FromMilliSeconds(
                2 * ::nnt::hid::GetKeyboardSamplingInterval()
                        .GetMilliSeconds()));
    }

    int count = ::nn::hid::GetKeyboardStates(
        g_States1, ::nn::hid::KeyboardStateCountMax);
    EXPECT_LE(1, count);

    bool isActive = true;

    // 自動操作状態に応じて Keyboard の入力状態は変化する。
    for (int i = 0; i < count; ++i)
    {
        const ::nn::hid::KeyboardState& lhs = g_States1[i];

        EXPECT_LE(0, lhs.samplingNumber);

        if (lhs.keys.IsAllOff())
        {
            isActive = false;
        }
        else
        {
            SCOPED_TRACE("");
            EXPECT_TRUE(isActive);
            ExpectFuzzedKeyboardState(lhs);
        }

        if (i + 1 < count)
        {
            SCOPED_TRACE("");
            ExpectAdjacentKeyboardStates(lhs, g_States1[i + 1]);
        }
    }

    ::nnt::hid::FinalizeKeyboardForAutoPilot();
}

//!< Keyboard の入力状態は自動操作状態の解除を正しく反映するか
TEST(KeyboardAutoPilotSuite, StateReadingTest2)
{
    ::nn::hid::InitializeKeyboard();

    ::nn::hid::debug::KeyboardAutoPilotState autoPilotState;
    FuzzKeyboardAutoPilotState(&autoPilotState, 4);

    ::nn::hid::debug::SetKeyboardAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::KeyboardStateCountMax *
            ::nnt::hid::GetKeyboardSamplingInterval().GetMilliSeconds()));

    // 自動操作状態を解除
    ::nn::hid::debug::UnsetKeyboardAutoPilotState();

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::KeyboardStateCountMax / 2 *
            ::nnt::hid::GetKeyboardSamplingInterval().GetMilliSeconds()));

    int count = ::nn::hid::GetKeyboardStates(g_States1,
                                             ::nn::hid::KeyboardStateCountMax);
    EXPECT_EQ(::nn::hid::KeyboardStateCountMax, count);

    bool hasChanged = false;

    ::nn::hid::KeyboardState state = g_States1[0];
    state.keys = ::nn::hid::KeyboardKeySet();

    // 入力状態列状で自動操作状態は連続的に切り替わる。
    for (int i = 0; i < count; ++i)
    {
        if (i + 1 < count)
        {
            SCOPED_TRACE("");
            ExpectAdjacentKeyboardStates(g_States1[i], g_States1[i + 1]);
        }

        if (g_States1[i].modifiers != state.modifiers)
        {
            EXPECT_FALSE(hasChanged);
            hasChanged = true;
            state.modifiers = autoPilotState.modifiers;
            state.keys = autoPilotState.keys;
        }

        {
            SCOPED_TRACE("");
            ExpectSameOriginKeyboardStates(state, g_States1[i]);
        }
    }

    EXPECT_TRUE(hasChanged);

    ::nnt::hid::FinalizeKeyboardForAutoPilot();
}

//!< LIFO の境界を跨いで自動操作状態の設定を正しく反映するか
TEST(KeyboardAutoPilotSuite, StateReadingTest3)
{
    ::nn::hid::InitializeKeyboard();

    ::nn::hid::debug::KeyboardAutoPilotState autoPilotState;

    int lastCount = 0;

    // LIFO の境界を 2 回跨ぐまでサンプリングを継続
    for (int i = 0; i < ::nn::hid::KeyboardStateCountMax * 3; ++i)
    {
        FuzzKeyboardAutoPilotState(&autoPilotState, i);
        ::nn::hid::debug::SetKeyboardAutoPilotState(autoPilotState);
        ::nn::os::SleepThread(
            ::nn::TimeSpan::FromMilliSeconds(
                ::nnt::hid::GetKeyboardSamplingInterval().GetMilliSeconds()));

        // 今回のサンプリング結果を格納するバッファ
        ::nn::hid::KeyboardState *ptr1 = (i % 2 == 1) ? &g_States1[0]
                                                      : &g_States2[0];

        // 前回のサンプリング結果が格納されているバッファ
        ::nn::hid::KeyboardState *ptr2 = (i % 2 == 1) ? &g_States2[0]
                                                      : &g_States1[0];

        // 取得される入力状態の数は単調増加する。
        int count = ::nn::hid::GetKeyboardStates(
            ptr1,
            ::nn::hid::KeyboardStateCountMax);
        EXPECT_LE(lastCount, count);

        bool isActive = true;

        for (int j = 0; j < count; ++j)
        {
            if (ptr1[j].keys.IsAllOff())
            {
                isActive = false;
                EXPECT_LE(0, ptr1[i].samplingNumber);
            }
            else
            {
                SCOPED_TRACE("");
                EXPECT_TRUE(isActive);
                ExpectFuzzedKeyboardState(ptr1[j]);
            }

            if (j + 1 < count)
            {
                SCOPED_TRACE("");
                ExpectAdjacentKeyboardStates(ptr1[j], ptr1[j + 1]);
            }

            // 前回のサンプリング結果と重複する箇所には同じ値が入る。
            if (lastCount > 0 &&
                ptr1[j].samplingNumber == ptr2[0].samplingNumber)
            {
                for (int k = 0; k < lastCount - j; ++k)
                {
                    SCOPED_TRACE("");
                    EXPECT_EQ(ptr2[k].samplingNumber,
                              ptr1[j + k].samplingNumber);
                    ExpectSameOriginKeyboardStates(ptr2[k], ptr1[j + k]);
                }

                break;
            }
        }

        lastCount = count;
    }

    EXPECT_EQ(::nn::hid::KeyboardStateCountMax, lastCount);

    ::nnt::hid::FinalizeKeyboardForAutoPilot();
}

//!< Keyboard の入力状態は切断エミュレーションの設定を正しく反映するか
TEST(KeyboardAutoPilotSuite, StateReadingTest4)
{
    ::nn::hid::debug::KeyboardAutoPilotState autoPilotState = {};

    ::nn::hid::InitializeKeyboard();

    ::nn::hid::debug::SetKeyboardAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::KeyboardStateCountMax / 2 *
            ::nnt::hid::GetKeyboardSamplingInterval().GetMilliSeconds()));

    ::nn::hid::GetKeyboardState(&g_States1[0]);

    EXPECT_FALSE(
        g_States1[0].attributes.Test<
            ::nn::hid::KeyboardAttribute::IsConnected>());

    ::nnt::hid::FinalizeKeyboardForAutoPilot();
}

namespace
{

void FuzzKeyboardAutoPilotState(
    ::nn::hid::debug::KeyboardAutoPilotState* pState, int seed) NN_NOEXCEPT
{
    const int offset = (seed % (pState->modifiers.GetCount() - 5)) + 5;
    pState->modifiers.Reset();
    pState->modifiers.Set(offset, true);
    pState->attributes.Reset();
    pState->attributes.Set<::nn::hid::KeyboardAttribute::IsConnected>();
    pState->keys.Reset();
    pState->keys.Set(offset + pState->modifiers.GetCount(), true);
}

void ExpectDefaultKeyboardState(
    const ::nn::hid::KeyboardState& state) NN_NOEXCEPT
{
    EXPECT_LE(0, state.samplingNumber);
    EXPECT_FALSE(
        state.modifiers.Test<::nn::hid::KeyboardModifier::Control>());
    EXPECT_FALSE(
        state.modifiers.Test<::nn::hid::KeyboardModifier::Shift>());
    EXPECT_FALSE(
        state.modifiers.Test<::nn::hid::KeyboardModifier::LeftAlt>());
    EXPECT_FALSE(
        state.modifiers.Test<::nn::hid::KeyboardModifier::RightAlt>());
    EXPECT_FALSE(
        state.modifiers.Test<::nn::hid::KeyboardModifier::Gui>());
    EXPECT_TRUE(state.keys.IsAllOff());
}

void ExpectFuzzedKeyboardState(
    const ::nn::hid::KeyboardState& state) NN_NOEXCEPT
{
    EXPECT_LE(0, state.samplingNumber);
    EXPECT_EQ(1, state.modifiers.CountPopulation());
    EXPECT_TRUE(
        state.attributes.Test<::nn::hid::KeyboardAttribute::IsConnected>());
    EXPECT_EQ(1, state.keys.CountPopulation());
    for (int i = 0; i < state.keys.GetCount(); ++i)
    {
        if (state.keys.Test(i))
        {
            EXPECT_TRUE(state.modifiers.Test(i % state.modifiers.GetCount()));
        }
    }
}

void ExpectSameOriginKeyboardStates(
    const ::nn::hid::KeyboardState& lhs,
    const ::nn::hid::KeyboardState& rhs) NN_NOEXCEPT
{
    EXPECT_EQ(lhs.modifiers, rhs.modifiers);
    EXPECT_EQ(lhs.keys, rhs.keys);
}

void ExpectAdjacentKeyboardStates(
    const ::nn::hid::KeyboardState& lhs,
    const ::nn::hid::KeyboardState& rhs) NN_NOEXCEPT
{
    EXPECT_EQ(1, lhs.samplingNumber - rhs.samplingNumber);
}

} // namespace
