﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_TimeSpan.h>
#include <nn/hid.h>
#include <nn/hid/debug/hid_DebugPad.h>
#include <nn/os.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nnt/nntest.h>

#include "../Common/testHid_DebugPad.h"

namespace
{
::nn::hid::DebugPadState g_State;
} // namespace

//!< DebugPad の初期化状態は GetDebugPadState() の呼び出しに反映されるか
TEST(DebugPadInitializationSuite, NoInitializationTest1)
{
    // DebugPad が未初期化の状態で呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(::nn::hid::GetDebugPadState(&g_State), "");

    ::nn::hid::InitializeDebugPad();

    ::nn::hid::InitializeDebugPad();

    ::nn::hid::debug::FinalizeDebugPad();

    // DebugPad が終了しきっていないので成功する。
    ::nn::hid::GetDebugPadState(&g_State);

    ::nn::hid::debug::FinalizeDebugPad();

    // DebugPad の終了処理後に呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(::nn::hid::GetDebugPadState(&g_State), "");
}

//!< DebugPad の初期化状態は GetDebugPadStates() の呼び出しに反映されるか
TEST(DebugPadInitializationSuite, NoInitializationTest2)
{
    // DebugPad が未初期化の状態で呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(::nn::hid::GetDebugPadStates(&g_State, 1), "");

    ::nn::hid::InitializeDebugPad();

    ::nn::hid::InitializeDebugPad();

    ::nn::hid::debug::FinalizeDebugPad();

    // DebugPad が終了しきっていないので成功する。
    ::nn::hid::GetDebugPadState(&g_State);

    ::nn::hid::debug::FinalizeDebugPad();

    // DebugPad の終了処理後に呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(::nn::hid::GetDebugPadStates(&g_State, 1), "");
}

//!< DebugPad の初期化状態は SetDebugPadAutoPilotState() の呼び出しに反映されるか
TEST(DebugPadInitializationSuite, NoInitializationTest3)
{
    ::nn::hid::debug::DebugPadAutoPilotState autoPilotState = {};

    // DebugPad が未初期化の状態で呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::debug::SetDebugPadAutoPilotState(autoPilotState), "");

    ::nn::hid::InitializeDebugPad();

    ::nn::hid::InitializeDebugPad();

    ::nn::hid::debug::FinalizeDebugPad();

    // DebugPad が終了しきっていないので成功する。
    ::nn::hid::debug::SetDebugPadAutoPilotState(autoPilotState);

    ::nnt::hid::FinalizeDebugPadForAutoPilot();

    // DebugPad の終了処理後に呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::debug::SetDebugPadAutoPilotState(autoPilotState), "");
}

//!< DebugPad の初期化状態は UnsetDebugPadAutoPilotState() の呼び出しに反映されるか
TEST(DebugPadInitializationSuite, NoInitializationTest4)
{
    // DebugPad が未初期化の状態で呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::debug::UnsetDebugPadAutoPilotState(), "");

    ::nn::hid::InitializeDebugPad();

    ::nn::hid::InitializeDebugPad();

    ::nn::hid::debug::FinalizeDebugPad();

    // DebugPad が終了しきっていないので成功する。
    ::nn::hid::debug::UnsetDebugPadAutoPilotState();

    ::nn::hid::debug::FinalizeDebugPad();

    // DebugPad の終了処理後に呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::debug::UnsetDebugPadAutoPilotState(), "");
}

//!< DebugPad の初期化処理は最低一つの入力状態の取得を保証するか
TEST(DebugPadInitializationSuite, StateReadingTest1)
{
    ::nn::hid::InitializeDebugPad();

    // 初期化直後の入力状態の取得に成功する。
    ::nn::hid::GetDebugPadState(&g_State);
    EXPECT_LE(0, g_State.samplingNumber);
    EXPECT_TRUE(g_State.buttons.IsAllOff());
    EXPECT_EQ(0, g_State.analogStickR.x);
    EXPECT_EQ(0, g_State.analogStickR.y);
    EXPECT_EQ(0, g_State.analogStickL.x);
    EXPECT_EQ(0, g_State.analogStickL.y);

    ::nn::hid::debug::FinalizeDebugPad();
}

//!< DebugPad の初期化処理はサンプリングを正しく有効化するか
TEST(DebugPadInitializationSuite, StateReadingTest2)
{
    ::nn::hid::InitializeDebugPad();

    ::nn::hid::InitializeDebugPad();

    ::nn::hid::GetDebugPadState(&g_State);

    int64_t samplingNumber = g_State.samplingNumber;

    ::nn::hid::debug::FinalizeDebugPad();

    // DebugPad が終了しきっていないのでサンプリングは継続する。
    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::DebugPadStateCountMax *
            ::nnt::hid::GetDebugPadSamplingInterval().GetMilliSeconds()));

    ::nn::hid::GetDebugPadState(&g_State);

    EXPECT_LT(samplingNumber + ::nn::hid::DebugPadStateCountMax / 4,
              g_State.samplingNumber);

    ::nn::hid::debug::FinalizeDebugPad();
#if defined(NN_BUILD_CONFIG_OS_WIN)
    samplingNumber = g_State.samplingNumber;

    // DebugPad が終了しているのでサンプリングは停止する。
    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::DebugPadStateCountMax *
            ::nnt::hid::GetDebugPadSamplingInterval().GetMilliSeconds()));

    ::nn::hid::InitializeDebugPad();

    ::nn::hid::GetDebugPadState(&g_State);

    EXPECT_GT(samplingNumber + ::nn::hid::DebugPadStateCountMax / 4,
              g_State.samplingNumber);

    ::nn::hid::debug::FinalizeDebugPad();
#endif
}
