﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_TimeSpan.h>
#include <nn/hid/debug/hid_CaptureButton.h>
#include <nn/hid/system/hid_CaptureButton.h>
#include <nn/os.h>
#include <nnt/nntest.h>

#include "../Common/testHid_CaptureButton.h"

namespace
{

::nn::hid::system::CaptureButtonState
    g_States[::nn::hid::system::CaptureButtonStateCountMax];

//!< 指定の優先度のスレッドから入力状態の読み出しが可能かテストします。
void ThreadingTest(int priority);

} // namespace

//!< 入力状態の読み出し時にバッファは正しく利用されるか
TEST(CaptureButtonReadingStateSuite, ReadingStateTest1)
{
    for (::nn::hid::system::CaptureButtonState& state : g_States)
    {
        state = ::nn::hid::system::CaptureButtonState();
        state.samplingNumber = 0xDEADBEEF;
        state.buttons.Set(1, true);
    }

    ::nn::hid::system::InitializeCaptureButton();

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::system::CaptureButtonStateCountMax / 2 *
            ::nnt::hid::GetCaptureButtonSamplingInterval().GetMilliSeconds()));

    int count = ::nn::hid::system::GetCaptureButtonStates(
        g_States, ::nn::hid::system::CaptureButtonStateCountMax);

    EXPECT_GE(::nn::hid::system::CaptureButtonStateCountMax, count);

    for (int i = count; i < ::nn::hid::system::CaptureButtonStateCountMax; ++i)
    {
        // 入力状態の書き込まれなかった領域は値を維持する。
        EXPECT_EQ(0xDEADBEEF, g_States[i].samplingNumber);
        ::nn::hid::system::CaptureButtonSet buttons;
        buttons.Reset();
        buttons.Set(1, true);
        EXPECT_EQ(buttons, g_States[i].buttons);
    }

    ::nn::hid::debug::FinalizeCaptureButton();
}

//!< 高優先度のスレッドから入力状態の読み出しが可能か
TEST(CaptureButtonReadingStateSuite, ThreadingTest1)
{
    ThreadingTest(::nn::os::HighestThreadPriority);
}

//!< 低優先度のスレッドから入力状態の読み出しが可能か
TEST(CaptureButtonReadingStateSuite, ThreadingTest2)
{
    ThreadingTest(::nn::os::LowestThreadPriority);
}

namespace
{

void ThreadingTest(int priority)
{
    ::nn::os::ThreadType* pThread = ::nn::os::GetCurrentThread();

    // 指定の優先度に設定する。
    const int original = ::nn::os::ChangeThreadPriority(pThread, priority);

    ::nn::hid::system::InitializeCaptureButton();

    int lastCount = 1;
    int64_t lastSamplingNumber = 0;

    for (int i = 0; i < 60; ++i)
    {
        int count = ::nn::hid::system::GetCaptureButtonStates(
            g_States, ::nn::hid::system::CaptureButtonStateCountMax);
        EXPECT_LE(lastCount, count);

        EXPECT_LE(lastSamplingNumber, g_States[0].samplingNumber);

        for (int j = 0; j < count - 1; ++j)
        {
            const ::nn::hid::system::CaptureButtonState& lhs = g_States[j];
            const ::nn::hid::system::CaptureButtonState& rhs = g_States[j + 1];
            EXPECT_EQ(1, lhs.samplingNumber - rhs.samplingNumber);
        }

        lastCount = count;
        lastSamplingNumber = g_States[0].samplingNumber;

        ::nn::os::SleepThread(::nn::TimeSpan::FromMilliSeconds(16));
    }

    EXPECT_EQ(::nn::hid::system::CaptureButtonStateCountMax, lastCount);

    ::nn::hid::debug::FinalizeCaptureButton();

    // 優先度を元に戻す。
    ::nn::os::ChangeThreadPriority(pThread, original);
}

} // namespace
