﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nnt/nntest.h>
#include <algorithm>

#include <nn/os/os_MessageQueue.h>

#include <hos/nvhdcp_up.h>

#include "../Shared/testHdcp_Config.h"
#include "../Shared/testHdcp_Helper.h"
#include "../Shared/testHdcp_Util-hardware.nx.h"
#include "../Shared/HdcpRawTestInitializer.h"

/**
 * @brief   Switch HDCP settings in dock-in/out.
 */
TEST(RawConnected, SwitchHdcpSetting)
{
    nnt::hdcp::HdcpRawTestInitializer initializer;
    initializer.ShowWarningBasedOnHdmiConnection(true);
    bool isHdmiConnected = initializer.IsHdmiConnected();
    ASSERT_EQ(true, isHdmiConnected);

    HDCP_CLIENT_HANDLE handle;
    HDCP_RET_ERROR error;
    int64_t start = 0LL;
    int64_t end = 0LL;

    start = nnt::hdcp::GetCurrentMilliSeconds();
    error = hdcp_open(&handle, nnt::hdcp::HdcpDeviceId);
    end = nnt::hdcp::GetCurrentMilliSeconds();

    // The setting of HDCP can be changed regardless of the state of dock-in/out.

    // Dock out.
    nnt::hdcp::TriggerPseudoDockOut(0);

    // Enable HDCP.
    error = hdcp_enable(handle, true);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);

    // Cannot finish authentication due to no display.
    error = HDCP_RET_READS_FAILED;
    for (int sec = 1; sec <= nnt::hdcp::AuthenticationTimeLimitSec && HDCP_RET_SUCCESS != error; ++sec) {
        NN_LOG("Authenticating HDCP... %d sec.\n", sec);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
        error = hdcp_status(handle);
    }
    end = nnt::hdcp::GetCurrentMilliSeconds();
    EXPECT_EQ(HDCP_RET_NOT_PLUGGED, error);

    // Dock in.
    nnt::hdcp::TriggerPseudoDockIn(0);

    // Can finish authentication.
    error = HDCP_RET_READS_FAILED;
    for (int sec = 1; sec <= nnt::hdcp::AuthenticationTimeLimitSec && HDCP_RET_SUCCESS != error; ++sec) {
        NN_LOG("Authenticating HDCP... %d sec.\n", sec);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
        error = hdcp_status(handle);
    }
    end = nnt::hdcp::GetCurrentMilliSeconds();
    EXPECT_EQ(HDCP_RET_SUCCESS, error);

    // Dock out.
    nnt::hdcp::TriggerPseudoDockOut(10);

    // Disable HDCP.
    error = hdcp_enable(handle, false);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);

    // Dock in.
    nnt::hdcp::TriggerPseudoDockIn(0);

    // Cannot finish authentication due to HDCP disabled.
    error = HDCP_RET_READS_FAILED;
    for (int sec = 1; sec <= nnt::hdcp::AuthenticationTimeLimitSec && HDCP_RET_SUCCESS != error; ++sec) {
        NN_LOG("Authenticating HDCP... %d sec.\n", sec);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
        error = hdcp_status(handle);
    }
    end = nnt::hdcp::GetCurrentMilliSeconds();
    EXPECT_EQ(HDCP_RET_DISABLED, error);

    error = hdcp_close(handle);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);
}

/**
 * @brief   Show the elapsed time of disabling HDCP.
 */
TEST(RawConnected, ShowElapsedTimeOfDisabling)
{
    nnt::hdcp::HdcpRawTestInitializer initializer;
    initializer.ShowWarningBasedOnHdmiConnection(true);
    bool isHdmiConnected = initializer.IsHdmiConnected();
    ASSERT_EQ(true, isHdmiConnected);

    HDCP_CLIENT_HANDLE handle;
    HDCP_RET_ERROR error;
    int64_t start = 0LL;
    int64_t end = 0LL;

    start = nnt::hdcp::GetCurrentMilliSeconds();
    error = hdcp_open(&handle, nnt::hdcp::HdcpDeviceId);
    end = nnt::hdcp::GetCurrentMilliSeconds();
    // Within 100 msec.
    EXPECT_GE(100, end - start);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);

    ///////////////////////////////////////////////////

    int startSleepTime = 100;
    int endSleepTime = 2000;
    int divides = 21;
    NN_LOG("Start -------------------------------------\n");
    for (int i1 = 0; i1 < divides; ++i1) {
        int64_t base = nnt::hdcp::GetCurrentMilliSeconds();
        int64_t sleepTime = (int64_t)((endSleepTime - startSleepTime) * i1 / (divides - 1.0) + startSleepTime);
        // Enable HDCP.
        start = nnt::hdcp::GetCurrentMilliSeconds();
        error = hdcp_enable(handle, true);
        end = nnt::hdcp::GetCurrentMilliSeconds();
        NN_LOG("Elapsed time of enabling :[%lld](msec)\n", end - start);
        EXPECT_EQ(HDCP_RET_SUCCESS, error);

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(sleepTime));

        // Disable HDCP.
        start = nnt::hdcp::GetCurrentMilliSeconds();
        error = hdcp_enable(handle, false);
        end = nnt::hdcp::GetCurrentMilliSeconds();
        NN_LOG("Elapsed time of disabling:[%lld](msec)\n", end - start);
        EXPECT_EQ(HDCP_RET_SUCCESS, error);

        end = nnt::hdcp::GetCurrentMilliSeconds();
        NN_LOG("Elapsed time of total:[%lld](msec) sleep time:[%lld](msec)\n", end - base, sleepTime);
    }
    NN_LOG("End ---------------------------------------\n");

    start = nnt::hdcp::GetCurrentMilliSeconds();
    error = hdcp_close(handle);
    end = nnt::hdcp::GetCurrentMilliSeconds();
    EXPECT_GE(100, end - start);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);
}
#if 0
// NOTE:
//  As this stress test is too stressful, it is removed now.
/**
 * @brief   Check if HDCP authentication is working well even if DPCD is stressful.
 */
TEST(RawConnected, UnderDpcdStressed)
{
    nnt::hdcp::HdcpRawTestInitializer initializer;
    initializer.ShowWarningBasedOnHdmiConnection(true);

    HDCP_CLIENT_HANDLE handle;
    HDCP_RET_ERROR error;
    int64_t start = 0LL;
    int64_t end = 0LL;

    error = hdcp_open(&handle, nnt::hdcp::HdcpDeviceId);
    ASSERT_EQ(HDCP_RET_SUCCESS, error);

    // Disable HDCP.
    error = hdcp_enable(handle, false);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);
    // Impose stress.
    initializer.StartThreadToImposeStressOnDpcd();
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));

    int trails = 10;
    for (int i1 = 0; i1 < trails; ++i1)
    {
        NN_LOG("trail:[%d/%d]-----------------\n", i1, trails);
        // Enable HDCP.
        error = hdcp_enable(handle, true);
        EXPECT_EQ(HDCP_RET_SUCCESS, error);
        error = hdcp_status(handle);
        EXPECT_NE(HDCP_RET_SUCCESS, error);

        // The status of HDCP goes to "ENABLED" within a few seconds usually.
        // But in this case, some cases may fail due to DPCD stress.
        for (int sec = 1; sec <= nnt::hdcp::AuthenticationTimeLimitSec && HDCP_RET_SUCCESS != error; ++sec) {
            NN_LOG("Authenticating HDCP... %d sec.\n", sec);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
            error = hdcp_status(handle);
        }
        if (HDCP_RET_AUTH_IN_PROCESS == error)
        {
            NN_LOG("Authentication has not been finished.\n");
        }
        else if (HDCP_RET_NOT_PLUGGED == error)
        {
            NN_LOG("HDCP_RET_NOT_PLUGGED is returned. Returning thie error code is possible.\n");
        }
        else if (HDCP_RET_READS_FAILED == error)
        {
            NN_LOG("--------------------------------------------------------------\n");
            NN_LOG("Why is HDCP_RET_READS_FAILED returned here despite the progress time is short?\n");
            NN_LOG("--------------------------------------------------------------\n");
            EXPECT_NE(HDCP_RET_READS_FAILED, error);
        }
        else
        {
            EXPECT_EQ(HDCP_RET_SUCCESS, error);
        }

        int waitingBaseTime = 50;
        for (int i1 = 0; i1<60; ++i1)
        {
            start = nnt::hdcp::GetCurrentMilliSeconds();
            error = hdcp_enable(handle, false);
            EXPECT_EQ(HDCP_RET_SUCCESS, error);
            end = nnt::hdcp::GetCurrentMilliSeconds();
            EXPECT_GE(300, end - start);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(waitingBaseTime));
            start = nnt::hdcp::GetCurrentMilliSeconds();
            error = hdcp_enable(handle, true);
            EXPECT_EQ(HDCP_RET_SUCCESS, error);
            end = nnt::hdcp::GetCurrentMilliSeconds();
            EXPECT_GE(300, end - start);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(waitingBaseTime));
        }

        // Disable HDCP.
        error = hdcp_enable(handle, false);
        EXPECT_EQ(HDCP_RET_SUCCESS, error);
    }
    // Stop imposing stress.
    initializer.StopThreadToImposeStressOnDpcd();

    // Enable HDCP.
    error = hdcp_enable(handle, true);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);
    error = hdcp_status(handle);
    EXPECT_EQ(HDCP_RET_AUTH_IN_PROCESS, error);

    // The status of HDCP goes to "ENABLED" a few seconds.
    start = nnt::hdcp::GetCurrentMilliSeconds();
    for (int sec = 1; sec <= nnt::hdcp::AuthenticationTimeLimitSec && HDCP_RET_SUCCESS != error; ++sec) {
        NN_LOG("Authenticating HDCP... %d sec.\n", sec);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
        error = hdcp_status(handle);
    }
    end = nnt::hdcp::GetCurrentMilliSeconds();
    EXPECT_GE(10 * 1000 * nnt::hdcp::AuthenticationTimeLimitSec, end - start);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);

    error = hdcp_close(handle);
    ASSERT_EQ(HDCP_RET_SUCCESS, error);
}
#endif
/**
 * @brief   Check if the processing time is expected.
 */
TEST(RawConnected, ProcessingTime)
{
    nnt::hdcp::HdcpRawTestInitializer initializer;
    initializer.ShowWarningBasedOnHdmiConnection(true);
    bool isHdmiConnected = initializer.IsHdmiConnected();
    ASSERT_EQ(true, isHdmiConnected);

    HDCP_CLIENT_HANDLE handle;
    HDCP_RET_ERROR error;
    int64_t start = 0LL;
    int64_t end = 0LL;

    start = nnt::hdcp::GetCurrentMilliSeconds();
    error = hdcp_open(&handle, nnt::hdcp::HdcpDeviceId);
    end = nnt::hdcp::GetCurrentMilliSeconds();
    // Within 100 msec.
    EXPECT_GE(100, end - start);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);

    // Disable HDCP and check if HDCP is disabled.
    start = nnt::hdcp::GetCurrentMilliSeconds();
    error = hdcp_enable(handle, false);
    end = nnt::hdcp::GetCurrentMilliSeconds();
    EXPECT_GE(100, end - start);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);
    start = nnt::hdcp::GetCurrentMilliSeconds();
    error = hdcp_status(handle);
    end = nnt::hdcp::GetCurrentMilliSeconds();
    EXPECT_GE(100, end - start);
    EXPECT_EQ(HDCP_RET_DISABLED, error);

    // Enable HDCP.
    start = nnt::hdcp::GetCurrentMilliSeconds();
    error = hdcp_enable(handle, true);
    end = nnt::hdcp::GetCurrentMilliSeconds();
    ///////////////////////////////////////////////////////////////////////////////////////
    // QUESTION (small thing):
    //    Why is elapsed time of enabling HDCP over 100 msec?
    //    Is it OK?
    ///////////////////////////////////////////////////////////////////////////////////////
    EXPECT_GE(200, end - start);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);
    start = nnt::hdcp::GetCurrentMilliSeconds();
    error = hdcp_status(handle);
    end = nnt::hdcp::GetCurrentMilliSeconds();
    EXPECT_GE(100, end - start);
    EXPECT_EQ(HDCP_RET_AUTH_IN_PROCESS, error);

    // The status of HDCP goes to "ENABLED" a few seconds.
    start = nnt::hdcp::GetCurrentMilliSeconds();
    for (int sec = 1; sec <= nnt::hdcp::AuthenticationTimeLimitSec && HDCP_RET_SUCCESS != error; ++sec) {
        NN_LOG("Authenticating HDCP... %d sec.\n", sec);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
        error = hdcp_status(handle);
    }
    end = nnt::hdcp::GetCurrentMilliSeconds();
    EXPECT_GE(10 * 1000 * nnt::hdcp::AuthenticationTimeLimitSec, end - start);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);

    // Disable HDCP.
    start = nnt::hdcp::GetCurrentMilliSeconds();
    error = hdcp_enable(handle, false);
    end = nnt::hdcp::GetCurrentMilliSeconds();
    // My understanding is that HDCP can be disabled quickly against "enabled".
    // So the short time limit is set here.
    EXPECT_GE(100, end - start);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);
    start = nnt::hdcp::GetCurrentMilliSeconds();
    error = hdcp_status(handle);
    end = nnt::hdcp::GetCurrentMilliSeconds();
    EXPECT_GE(100, end - start);
    EXPECT_EQ(HDCP_RET_DISABLED, error);

    start = nnt::hdcp::GetCurrentMilliSeconds();
    error = hdcp_close(handle);
    end = nnt::hdcp::GetCurrentMilliSeconds();
    EXPECT_GE(100, end - start);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);
}

/**
 * @brief   Check if authentication is successful after switching HDCP state repeatedly.
 */
TEST(RawConnected, AuthenticationAfterSwitching)
{
    nnt::hdcp::HdcpRawTestInitializer initializer;
    initializer.ShowWarningBasedOnHdmiConnection(true);
    bool isHdmiConnected = initializer.IsHdmiConnected();
    ASSERT_EQ(true, isHdmiConnected);

    const int AuthenticationTimeLimitSec = 10;

    HDCP_CLIENT_HANDLE handle;
    HDCP_RET_ERROR error;
    int64_t start = 0LL;
    int64_t end = 0LL;

    error = hdcp_open(&handle, nnt::hdcp::HdcpDeviceId);
    ASSERT_EQ(HDCP_RET_SUCCESS, error);

    int waitingTime[] = { 1, 10, 100, 200 };
    for (auto wait : waitingTime)
    {
        // Impose stress by switching HDCP status repeatedly.
        NN_LOG("waiting time on switching:[%d](sec)\n", wait);
        for (int i1 = 0; i1<5; ++i1)
        {
            error = hdcp_enable(handle, false);
            ASSERT_EQ(HDCP_RET_SUCCESS, error);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(wait));
            error = hdcp_enable(handle, true);
            ASSERT_EQ(HDCP_RET_SUCCESS, error);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(wait));
        }

        // The status of HDCP goes to "ENABLED" a few seconds.
        error = hdcp_status(handle);
        start = nnt::hdcp::GetCurrentMilliSeconds();
        for (int sec = 1; sec <= nnt::hdcp::AuthenticationTimeLimitSec && HDCP_RET_SUCCESS != error; ++sec) {
            NN_LOG("Authenticating HDCP... %d sec.\n", sec);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
            error = hdcp_status(handle);
        }
        end = nnt::hdcp::GetCurrentMilliSeconds();
        EXPECT_GE(10 * 1000 * nnt::hdcp::AuthenticationTimeLimitSec, end - start);
        EXPECT_EQ(HDCP_RET_SUCCESS, error);
    }

    error = hdcp_close(handle);
    ASSERT_EQ(HDCP_RET_SUCCESS, error);
}

/**
 * @brief   Check if repeating to switch HDCP state is successful.
 */
TEST(RawConnected, RepeatSwitch)
{
    nnt::hdcp::HdcpRawTestInitializer initializer;
    initializer.ShowWarningBasedOnHdmiConnection(true);
    bool isHdmiConnected = initializer.IsHdmiConnected();
    ASSERT_EQ(true, isHdmiConnected);

    HDCP_CLIENT_HANDLE handle;
    HDCP_RET_ERROR error;
    int64_t start = 0LL;
    int64_t end = 0LL;

    error = hdcp_open(&handle, nnt::hdcp::HdcpDeviceId);
    ASSERT_EQ(HDCP_RET_SUCCESS, error);

    for (int i1 = 0; i1<60; ++i1)
    {
        ///////////////////////////////////////////////////////////////////////////////////////
        // MEMO:
        // When waitingTime is short, the following log is not shown:
        // ---
        //   dp: Failed to write DPCD data. CMD 0x6803b, Status 0x10010000
        //   disconnect while waiting for repeater
        //   get repeater info failed
        ///////////////////////////////////////////////////////////////////////////////////////
        int waitingBaseTime = (i1 % 50) + 30;
        start = nnt::hdcp::GetCurrentMilliSeconds();
        error = hdcp_enable(handle, false);
        EXPECT_EQ(HDCP_RET_SUCCESS, error);
        end = nnt::hdcp::GetCurrentMilliSeconds();
        // Here, 650 msec needs to diable HDCP settings as worst case
        // because it is possible to disconnect before verifying vprime
        EXPECT_GE(650, end - start);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(waitingBaseTime));
        start = nnt::hdcp::GetCurrentMilliSeconds();
        error = hdcp_enable(handle, true);
        EXPECT_EQ(HDCP_RET_SUCCESS, error);
        end = nnt::hdcp::GetCurrentMilliSeconds();
        ///////////////////////////////////////////////////////////////////////////////////////
        // QUESTION (small thing):
        //    Why is elapsed time of enabling HDCP over 100 msec?
        //    Is it OK?
        ///////////////////////////////////////////////////////////////////////////////////////
        EXPECT_GE(200, end - start);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(waitingBaseTime));
    }
    error = hdcp_enable(handle, false);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);

    error = hdcp_close(handle);
    ASSERT_EQ(HDCP_RET_SUCCESS, error);
}

/**
 * @brief   Check if state transition is working well.
 */
TEST(RawConnected, StateTransition)
{
    nnt::hdcp::HdcpRawTestInitializer initializer;
    initializer.ShowWarningBasedOnHdmiConnection(true);
    bool isHdmiConnected = initializer.IsHdmiConnected();
    ASSERT_EQ(true, isHdmiConnected);

    HDCP_CLIENT_HANDLE handle;
    HDCP_RET_ERROR error;
    int64_t start = 0LL;
    int64_t end = 0LL;
    bool isSignaled;

    error = hdcp_open(&handle, nnt::hdcp::HdcpDeviceId);
    ASSERT_EQ(HDCP_RET_SUCCESS, error);

    std::uintptr_t communicationBuffer;
    std::uintptr_t blockBuffer;
    typedef struct
    {
        nn::os::MessageQueueType communicationQueue;
        nn::os::MessageQueueType blockQueue;
    } HdcpTestStateTransitionType;
    HdcpTestStateTransitionType testType;
    nn::os::InitializeMessageQueue(&testType.communicationQueue, &communicationBuffer, 1);
    nn::os::InitializeMessageQueue(&testType.blockQueue, &blockBuffer, 1);
    std::uintptr_t message = reinterpret_cast<std::uintptr_t>(nullptr);
    nn::os::SendMessageQueue(&testType.blockQueue, message);

    auto handler = [](HDCP_CLIENT_HANDLE handle, NvU32 state, void* context)
    {
        NN_SDK_LOG("[%d] %s() event is signaled. state:[%d] -------------\n", __LINE__, __FUNCTION__, state);
        NN_UNUSED(handle);
        auto testType = reinterpret_cast<HdcpTestStateTransitionType*>(context);
        std::uintptr_t message;
        nn::os::ReceiveMessageQueue(&message, &testType->blockQueue);
        // Timeout should not occur here.
        bool isSent = nn::os::TimedSendMessageQueue(&testType->communicationQueue, state, nn::TimeSpan::FromMilliSeconds(1000));
        EXPECT_EQ(true, isSent);
        nn::os::SendMessageQueue(&testType->blockQueue, message);
    };

    error = hdcp_event_state_transit(handle, handler, &testType);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);

    error = hdcp_init_asyncevents(handle);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);

    // Clear all events.
    error = hdcp_enable(handle, true);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);
    error = hdcp_enable(handle, false);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);
    while (nn::os::TimedReceiveMessageQueue(&message, &testType.communicationQueue, nn::TimeSpan::FromMilliSeconds(1000)))
        ;

    // STATE_OFF -> STATE_UNAUTHENTICATED -> STATE_LINK_VERIFY
    error = hdcp_enable(handle, true);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);

    nn::os::ReceiveMessageQueue(&message, &testType.communicationQueue);
    EXPECT_EQ(HDCP_STATE_UNAUTHENTICATED, message);
    error = hdcp_status(handle);
    EXPECT_EQ(HDCP_RET_AUTH_IN_PROCESS, error);
    nn::os::ReceiveMessageQueue(&message, &testType.communicationQueue);
    EXPECT_EQ(HDCP_STATE_LINK_VERIFY, message);
    error = hdcp_status(handle);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);

    // STATE_LINK_VERIFY -> STATE_OFF by HDCP disabled.
    error = hdcp_enable(handle, false);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);

    nn::os::ReceiveMessageQueue(&message, &testType.communicationQueue);
    EXPECT_EQ(HDCP_STATE_OFF, message);
    error = hdcp_status(handle);
    EXPECT_EQ(HDCP_RET_DISABLED, error);

    // STATE_OFF -> STATE_UNAUTHENTICATED -> STATE_LINK_VERIFY
    error = hdcp_enable(handle, true);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);

    nn::os::ReceiveMessageQueue(&message, &testType.communicationQueue);
    EXPECT_EQ(HDCP_STATE_UNAUTHENTICATED, message);
    error = hdcp_status(handle);
    EXPECT_EQ(HDCP_RET_AUTH_IN_PROCESS, error);
    nn::os::ReceiveMessageQueue(&message, &testType.communicationQueue);
    EXPECT_EQ(HDCP_STATE_LINK_VERIFY, message);
    error = hdcp_status(handle);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);

    // STATE_LINK_VERIFY -> STATE_OFF by dock-out.
    nnt::hdcp::TriggerPseudoDockOut(0);
    NN_SDK_LOG("[%d] %s() DOCK-OUT.\n", __LINE__, __FUNCTION__);

    nn::os::ReceiveMessageQueue(&message, &testType.communicationQueue);
    EXPECT_EQ(HDCP_STATE_OFF, message);
    error = hdcp_status(handle);
    EXPECT_EQ(HDCP_RET_NOT_PLUGGED, error);

    nnt::hdcp::TriggerPseudoDockIn(0);
    NN_SDK_LOG("[%d] %s() DOCK-IN.\n", __LINE__, __FUNCTION__);

    // STATE_OFF -> STATE_UNAUTHENTICATED -> STATE_OFF by HDCP disabled.
    error = hdcp_enable(handle, true);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);

    nn::os::ReceiveMessageQueue(&message, &testType.communicationQueue);
    EXPECT_EQ(HDCP_STATE_UNAUTHENTICATED, message);
    error = hdcp_status(handle);
    EXPECT_EQ(HDCP_RET_AUTH_IN_PROCESS, error);

    error = hdcp_enable(handle, false);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);

    nn::os::ReceiveMessageQueue(&message, &testType.communicationQueue);
    EXPECT_EQ(HDCP_STATE_OFF, message);
    error = hdcp_status(handle);
    EXPECT_EQ(HDCP_RET_DISABLED, error);

    // STATE_OFF -> STATE_UNAUTHENTICATED -> STATE_OFF by dock-out.
    error = hdcp_enable(handle, true);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);

    nn::os::ReceiveMessageQueue(&message, &testType.communicationQueue);
    EXPECT_EQ(HDCP_STATE_UNAUTHENTICATED, message);

    error = hdcp_status(handle);
    EXPECT_EQ(HDCP_RET_AUTH_IN_PROCESS, error);

    nnt::hdcp::TriggerPseudoDockOut(0);
    NN_SDK_LOG("[%d] %s() DOCK-OUT.\n", __LINE__, __FUNCTION__);

    nn::os::ReceiveMessageQueue(&message, &testType.communicationQueue);
    EXPECT_EQ(HDCP_STATE_OFF, message);
    error = hdcp_status(handle);
    EXPECT_EQ(HDCP_RET_NOT_PLUGGED, error);

    error = hdcp_enable(handle, false);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);

    // HDCP is disabled && docking-in  -> HDCP_RET_DISABLED
    // HDCP is disabled && docking-out -> HDCP_RET_NOT_PLUGGED
    // HDCP is enabled  && docking-out -> HDCP_RET_NOT_PLUGGED
    // So, HDCP_RET_NOT_PLUGGED should be returned here.
    error = hdcp_status(handle);
    EXPECT_EQ(HDCP_RET_NOT_PLUGGED, error);

    nnt::hdcp::TriggerPseudoDockIn(0);
    NN_SDK_LOG("[%d] %s() DOCK-IN.\n", __LINE__, __FUNCTION__);
    // After docking-in, no state change to HDCP_STATE_UNAUTHENTICATED occurs here
    // because HDCP setting is disabled.
    isSignaled = nn::os::TimedReceiveMessageQueue(&message,
        &testType.communicationQueue, nn::TimeSpan::FromMilliSeconds(1000 * 3));
    EXPECT_EQ(false, isSignaled);

    // There is the delay of the state change in DP driver side to actual dock-in.
    // So wait for making the state stable.
    error = hdcp_status(handle);
    int retryCount = 0;
    for (; retryCount<30 && HDCP_RET_NOT_PLUGGED == error; ++retryCount)
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        error = hdcp_status(handle);
    }
    // The polling for state change should be done within 500 msec.
    EXPECT_GE(5, retryCount);
    EXPECT_EQ(HDCP_RET_DISABLED, error);

    // There should be no event signal.
    isSignaled = nn::os::TimedReceiveMessageQueue(&message,
        &testType.communicationQueue, nn::TimeSpan::FromMilliSeconds(1000 * 3));
    EXPECT_EQ(false, isSignaled);

    // Right now, HDCP is disabled here.
    error = hdcp_status(handle);
    EXPECT_EQ(HDCP_RET_DISABLED, error);

    nnt::hdcp::TriggerPseudoDockOut(0);
    NN_SDK_LOG("[%d] %s() DOCK-OUT.\n", __LINE__, __FUNCTION__);
    message = -1;
    bool isReceived = nn::os::TimedReceiveMessageQueue(&message, &testType.communicationQueue, nn::TimeSpan::FromMilliSeconds(1000 * 5));
    EXPECT_EQ(false, isReceived);

    nnt::hdcp::TriggerPseudoDockIn(0);
    NN_SDK_LOG("[%d] %s() DOCK-IN.\n", __LINE__, __FUNCTION__);
    message = -1;
    isReceived = nn::os::TimedReceiveMessageQueue(&message, &testType.communicationQueue, nn::TimeSpan::FromMilliSeconds(1000 * 5));
    EXPECT_EQ(false, isReceived);
    EXPECT_NE(HDCP_STATE_UNAUTHENTICATED, message);

    // HDCP should be disabled yet.
    error = hdcp_status(handle);
    EXPECT_EQ(HDCP_RET_DISABLED, error);
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000 * 5));

    // HDCP starts to be enabled from here.
    error = hdcp_enable(handle, true);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);
    message = -1;
    isReceived = nn::os::TimedReceiveMessageQueue(&message, &testType.communicationQueue, nn::TimeSpan::FromMilliSeconds(1000 * 5));
    EXPECT_EQ(true, isReceived);
    EXPECT_EQ(HDCP_STATE_UNAUTHENTICATED, message);
    error = hdcp_status(handle);
    EXPECT_EQ(HDCP_RET_AUTH_IN_PROCESS, error);
    message = -1;
    isReceived = nn::os::TimedReceiveMessageQueue(&message, &testType.communicationQueue, nn::TimeSpan::FromMilliSeconds(1000 * 5));
    EXPECT_EQ(true, isReceived);
    EXPECT_EQ(HDCP_STATE_LINK_VERIFY, message);
    error = hdcp_status(handle);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);

    // Finalize.
    error = hdcp_enable(handle, false);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);
    message = -1;
    isReceived = nn::os::TimedReceiveMessageQueue(&message, &testType.communicationQueue, nn::TimeSpan::FromMilliSeconds(1000 * 5));
    EXPECT_EQ(true, isReceived);
    EXPECT_EQ(HDCP_STATE_OFF, message);

    error = hdcp_close(handle);
    ASSERT_EQ(HDCP_RET_SUCCESS, error);
} // NOLINT(impl/function_size)

/**
 * @brief   Check if the behavior of state transition initializer is OK.
 */
TEST(RawConnected, InitializeStateTransition)
{
    nnt::hdcp::HdcpRawTestInitializer initializer;
    initializer.ShowWarningBasedOnHdmiConnection(true);
    bool isHdmiConnected = initializer.IsHdmiConnected();
    ASSERT_EQ(true, isHdmiConnected);

    HDCP_CLIENT_HANDLE handle;
    HDCP_RET_ERROR error;
    int64_t start = 0LL;
    int64_t end = 0LL;

    error = hdcp_open(&handle, nnt::hdcp::HdcpDeviceId);
    ASSERT_EQ(HDCP_RET_SUCCESS, error);

    // Enabled -> Disabled.
    error = hdcp_enable(handle, true);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);
    error = hdcp_enable(handle, false);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);

    std::uintptr_t communicationBuffer;
    std::uintptr_t blockBuffer;
    typedef struct
    {
        nn::os::MessageQueueType communicationQueue;
        nn::os::MessageQueueType blockQueue;
    } HdcpTestStateTransitionType;
    HdcpTestStateTransitionType testType;
    nn::os::InitializeMessageQueue(&testType.communicationQueue, &communicationBuffer, 1);
    nn::os::InitializeMessageQueue(&testType.blockQueue, &blockBuffer, 1);
    std::uintptr_t message = reinterpret_cast<std::uintptr_t>(nullptr);
    nn::os::SendMessageQueue(&testType.blockQueue, message);

    auto handler = [](HDCP_CLIENT_HANDLE handle, NvU32 state, void* context)
    {
        NN_UNUSED(handle);
        auto testType = reinterpret_cast<HdcpTestStateTransitionType*>(context);
        std::uintptr_t message;
        nn::os::ReceiveMessageQueue(&message, &testType->blockQueue);
        message = state;
        // Timeout should not occur here.
        bool isSent = nn::os::TimedSendMessageQueue(&testType->communicationQueue, message, nn::TimeSpan::FromMilliSeconds(1000));
        EXPECT_EQ(true, isSent);
        nn::os::SendMessageQueue(&testType->blockQueue, message);
    };

    error = hdcp_event_state_transit(handle, handler, &testType);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);

    error = hdcp_init_asyncevents(handle);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);

    // All state transitions before initializing by hdcp_init_asyncevents should not be signaled.
    bool isReceived = nn::os::TimedReceiveMessageQueue(&message, &testType.communicationQueue, nn::TimeSpan::FromMilliSeconds(1000));
    EXPECT_EQ(false, isReceived);

    error = hdcp_close(handle);
    ASSERT_EQ(HDCP_RET_SUCCESS, error);
}

/**
 * @brief   Simple stress test.
 */
TEST(RawConnected, StressRandomly)
{
    nnt::hdcp::HdcpRawTestInitializer initializer;
    initializer.ShowWarningBasedOnHdmiConnection(true);
    bool isHdmiConnected = initializer.IsHdmiConnected();
    ASSERT_EQ(true, isHdmiConnected);

    HDCP_CLIENT_HANDLE handle;
    HDCP_RET_ERROR error;
    int64_t start = 0LL;
    int64_t end = 0LL;

    error = hdcp_open(&handle, nnt::hdcp::HdcpDeviceId);
    ASSERT_EQ(HDCP_RET_SUCCESS, error);

    for (int i1 = 0; i1<30; ++i1)
    {
        start = nnt::hdcp::GetCurrentMilliSeconds();
        hdcp_enable(handle, false);
        end = nnt::hdcp::GetCurrentMilliSeconds();
        EXPECT_GE(200, end - start);
        start = nnt::hdcp::GetCurrentMilliSeconds();
        hdcp_enable(handle, true);
        end = nnt::hdcp::GetCurrentMilliSeconds();
        EXPECT_GE(200, end - start);
        start = nnt::hdcp::GetCurrentMilliSeconds();
        hdcp_enable(handle, false);
        end = nnt::hdcp::GetCurrentMilliSeconds();
        EXPECT_GE(200, end - start);
        start = nnt::hdcp::GetCurrentMilliSeconds();
        hdcp_enable(handle, true);
        end = nnt::hdcp::GetCurrentMilliSeconds();
        EXPECT_GE(200, end - start);
        start = nnt::hdcp::GetCurrentMilliSeconds();
        hdcp_status(handle);
        end = nnt::hdcp::GetCurrentMilliSeconds();
        EXPECT_GE(200, end - start);
        start = nnt::hdcp::GetCurrentMilliSeconds();
        hdcp_enable(handle, true);
        end = nnt::hdcp::GetCurrentMilliSeconds();
        EXPECT_GE(200, end - start);
        start = nnt::hdcp::GetCurrentMilliSeconds();
        hdcp_status(handle);
        end = nnt::hdcp::GetCurrentMilliSeconds();
        EXPECT_GE(200, end - start);
        start = nnt::hdcp::GetCurrentMilliSeconds();
        hdcp_enable(handle, false);
        end = nnt::hdcp::GetCurrentMilliSeconds();
        EXPECT_GE(200, end - start);
        start = nnt::hdcp::GetCurrentMilliSeconds();
        hdcp_status(handle);
        end = nnt::hdcp::GetCurrentMilliSeconds();
        EXPECT_GE(200, end - start);
    }
    error = hdcp_enable(handle, false);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);

    error = hdcp_close(handle);
    ASSERT_EQ(HDCP_RET_SUCCESS, error);
}

/**
 * @brief   Check if the behavior of state transition after changing resolution is OK.
 */
TEST(RawConnected, ResolutionChange)
{
    nnt::hdcp::HdcpRawTestInitializer initializer;
    initializer.ShowWarningBasedOnHdmiConnection(true);

    HDCP_CLIENT_HANDLE handle;
    HDCP_RET_ERROR error;
    bool isSent = false;

    error = hdcp_open(&handle, nnt::hdcp::HdcpDeviceId);
    ASSERT_EQ(HDCP_RET_SUCCESS, error);

    std::uintptr_t communicationBuffer;
    std::uintptr_t blockBuffer;
    typedef struct
    {
        NvU32 state;
        HDCP_RET_ERROR status;
    } HdcpTestCommunicationData;
    typedef struct
    {
        nn::os::MessageQueueType communicationQueue;
        nn::os::MessageQueueType blockQueue;
        HdcpTestCommunicationData data;
    } HdcpTestStateTransitionType;
    HdcpTestStateTransitionType testType;
    nn::os::InitializeMessageQueue(&testType.communicationQueue, &communicationBuffer, 1);
    nn::os::InitializeMessageQueue(&testType.blockQueue, &blockBuffer, 1);
    std::uintptr_t message = reinterpret_cast<std::uintptr_t>(nullptr);
    nn::os::SendMessageQueue(&testType.blockQueue, message);

    auto handler = [](HDCP_CLIENT_HANDLE handle, NvU32 state, void* context)
    {
        HDCP_RET_ERROR status = hdcp_status(handle);
        //NN_SDK_LOG("[%d] %s() event is signaled. state:[%d] status:[%d]\n", __LINE__, __FUNCTION__, state, status);
        NN_UNUSED(handle);
        auto testType = reinterpret_cast<HdcpTestStateTransitionType*>(context);
        std::uintptr_t message;
        nn::os::ReceiveMessageQueue(&message, &testType->blockQueue);
        testType->data.state = state;
        testType->data.status = status;
        // Timeout should not occur here.
        bool isSent = nn::os::TimedSendMessageQueue(&testType->communicationQueue, reinterpret_cast<std::uintptr_t>(&testType->data), nn::TimeSpan::FromMilliSeconds(1000));
        EXPECT_EQ(true, isSent);
    };

    error = hdcp_event_state_transit(handle, handler, &testType);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);

    error = hdcp_init_asyncevents(handle);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);

    // Clear all events.
    error = hdcp_enable(handle, true);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);
    error = hdcp_enable(handle, false);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);
    isSent = nn::os::TimedSendMessageQueue(&testType.blockQueue, message, nn::TimeSpan::FromMilliSeconds(1000));
    EXPECT_EQ(true, isSent);
    while (nn::os::TimedReceiveMessageQueue(&message, &testType.communicationQueue, nn::TimeSpan::FromMilliSeconds(1000))) {
        nn::os::TimedSendMessageQueue(&testType.blockQueue, message, nn::TimeSpan::FromMilliSeconds(1000));
    }

    // STATE_OFF -> STATE_UNAUTHENTICATED -> STATE_LINK_VERIFY
    error = hdcp_enable(handle, true);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);

    nn::os::ReceiveMessageQueue(&message, &testType.communicationQueue);
    EXPECT_EQ(HDCP_STATE_UNAUTHENTICATED, testType.data.state);
    EXPECT_EQ(HDCP_RET_AUTH_IN_PROCESS, testType.data.status);
    isSent = nn::os::TimedSendMessageQueue(&testType.blockQueue, message, nn::TimeSpan::FromMilliSeconds(1000));
    EXPECT_EQ(true, isSent);

    nn::os::ReceiveMessageQueue(&message, &testType.communicationQueue);
    EXPECT_EQ(HDCP_STATE_LINK_VERIFY, testType.data.state);
    EXPECT_EQ(HDCP_RET_SUCCESS, testType.data.status);
    isSent = nn::os::TimedSendMessageQueue(&testType.blockQueue, message, nn::TimeSpan::FromMilliSeconds(1000));
    EXPECT_EQ(true, isSent);

    // STATE_LINK_VERIFY -> STATE_OFF -> STATE_UNAUTHENTICATED -> STATE_LINK_VERIFY
    initializer.ChangeDisplayResolution();
    nn::os::ReceiveMessageQueue(&message, &testType.communicationQueue);
    EXPECT_EQ(HDCP_STATE_OFF, testType.data.state);
    EXPECT_EQ(HDCP_RET_READS_FAILED, testType.data.status);
    isSent = nn::os::TimedSendMessageQueue(&testType.blockQueue, message, nn::TimeSpan::FromMilliSeconds(1000));
    EXPECT_EQ(true, isSent);

    nn::os::ReceiveMessageQueue(&message, &testType.communicationQueue);
    EXPECT_EQ(HDCP_STATE_UNAUTHENTICATED, testType.data.state);
    EXPECT_EQ(HDCP_RET_AUTH_IN_PROCESS, testType.data.status);
    isSent = nn::os::TimedSendMessageQueue(&testType.blockQueue, message, nn::TimeSpan::FromMilliSeconds(1000));
    EXPECT_EQ(true, isSent);

    nn::os::ReceiveMessageQueue(&message, &testType.communicationQueue);
    EXPECT_EQ(HDCP_STATE_LINK_VERIFY, testType.data.state);
    EXPECT_EQ(HDCP_RET_SUCCESS, testType.data.status);
    isSent = nn::os::TimedSendMessageQueue(&testType.blockQueue, message, nn::TimeSpan::FromMilliSeconds(1000));
    EXPECT_EQ(true, isSent);

    // STATE_LINK_VERIFY -> STATE_OFF -> STATE_UNAUTHENTICATED -> STATE_LINK_VERIFY
    initializer.ChangeDisplayResolution();
    nn::os::ReceiveMessageQueue(&message, &testType.communicationQueue);
    EXPECT_EQ(HDCP_STATE_OFF, testType.data.state);
    EXPECT_EQ(HDCP_RET_READS_FAILED, testType.data.status);
    isSent = nn::os::TimedSendMessageQueue(&testType.blockQueue, message, nn::TimeSpan::FromMilliSeconds(1000));
    EXPECT_EQ(true, isSent);

    nn::os::ReceiveMessageQueue(&message, &testType.communicationQueue);
    EXPECT_EQ(HDCP_STATE_UNAUTHENTICATED, testType.data.state);
    EXPECT_EQ(HDCP_RET_AUTH_IN_PROCESS, testType.data.status);
    isSent = nn::os::TimedSendMessageQueue(&testType.blockQueue, message, nn::TimeSpan::FromMilliSeconds(1000));
    EXPECT_EQ(true, isSent);

    nn::os::ReceiveMessageQueue(&message, &testType.communicationQueue);
    EXPECT_EQ(HDCP_STATE_LINK_VERIFY, testType.data.state);
    EXPECT_EQ(HDCP_RET_SUCCESS, testType.data.status);
    isSent = nn::os::TimedSendMessageQueue(&testType.blockQueue, message, nn::TimeSpan::FromMilliSeconds(1000));
    EXPECT_EQ(true, isSent);

    // STATE_LINK_VERIFY -> STATE_OFF -> STATE_UNAUTHENTICATED -> STATE_OFF -> STATE_UNAUTHENTICATED -> STATE_LINK_VERIFY
    initializer.ChangeDisplayResolution();
    nn::os::ReceiveMessageQueue(&message, &testType.communicationQueue);
    EXPECT_EQ(HDCP_STATE_OFF, testType.data.state);
    EXPECT_EQ(HDCP_RET_READS_FAILED, testType.data.status);
    isSent = nn::os::TimedSendMessageQueue(&testType.blockQueue, message, nn::TimeSpan::FromMilliSeconds(1000));
    EXPECT_EQ(true, isSent);

    nn::os::ReceiveMessageQueue(&message, &testType.communicationQueue);
    EXPECT_EQ(HDCP_STATE_UNAUTHENTICATED, testType.data.state);
    EXPECT_EQ(HDCP_RET_AUTH_IN_PROCESS, testType.data.status);
    isSent = nn::os::TimedSendMessageQueue(&testType.blockQueue, message, nn::TimeSpan::FromMilliSeconds(1000));
    EXPECT_EQ(true, isSent);

    initializer.ChangeDisplayResolution();
    nn::os::ReceiveMessageQueue(&message, &testType.communicationQueue);
    EXPECT_EQ(HDCP_STATE_OFF, testType.data.state);
    EXPECT_EQ(HDCP_RET_READS_FAILED, testType.data.status);
    isSent = nn::os::TimedSendMessageQueue(&testType.blockQueue, message, nn::TimeSpan::FromMilliSeconds(1000));
    EXPECT_EQ(true, isSent);

    nn::os::ReceiveMessageQueue(&message, &testType.communicationQueue);
    EXPECT_EQ(HDCP_STATE_UNAUTHENTICATED, testType.data.state);
    EXPECT_EQ(HDCP_RET_AUTH_IN_PROCESS, testType.data.status);
    isSent = nn::os::TimedSendMessageQueue(&testType.blockQueue, message, nn::TimeSpan::FromMilliSeconds(1000));
    EXPECT_EQ(true, isSent);

    nn::os::ReceiveMessageQueue(&message, &testType.communicationQueue);
    EXPECT_EQ(HDCP_STATE_LINK_VERIFY, testType.data.state);
    EXPECT_EQ(HDCP_RET_SUCCESS, testType.data.status);
    isSent = nn::os::TimedSendMessageQueue(&testType.blockQueue, message, nn::TimeSpan::FromMilliSeconds(1000));
    EXPECT_EQ(true, isSent);

    //
    error = hdcp_enable(handle, false);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);
    nn::os::ReceiveMessageQueue(&message, &testType.communicationQueue);
    EXPECT_EQ(HDCP_STATE_OFF, testType.data.state);
    EXPECT_EQ(HDCP_RET_DISABLED, testType.data.status);
    isSent = nn::os::TimedSendMessageQueue(&testType.blockQueue, message, nn::TimeSpan::FromMilliSeconds(1000));
    EXPECT_EQ(true, isSent);

    error = hdcp_close(handle);
    ASSERT_EQ(HDCP_RET_SUCCESS, error);
} // NOLINT(impl/function_size)
