﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nnt/nntest.h>
#include <algorithm>

#include "../Shared/testHdcp_Config.h"
#include "../Shared/testHdcp_Helper.h"
#include "../Shared/testHdcp_Util-hardware.nx.h"
#include "../Shared/HdcpTestInitializer.h"
#include "../Shared/HdcpRawTestInitializer.h"

/**
 * @brief   イベントがシグナルされるかどうか。
 */
TEST(Connected, EventHandler)
{
    nnt::hdcp::HdcpTestInitializer initializer;

    bool isSignaled;
    nn::hdcp::HdcpAuthenticationState state;

    // イベント登録。
    nn::os::SystemEventType eventType;
    hdcpc::GetHdcpStateTransitionEvent(&eventType);

    // HDCPを無効にする。状態遷移が起きなくなったらテストスタート。
    hdcpc::SetCurrentHdcpMode(nn::hdcp::HdcpMode_Disabled);
    while (nn::os::TimedWaitSystemEvent(&eventType, nn::TimeSpan::FromMilliSeconds(300)))
    {
        nn::os::ClearSystemEvent(&eventType);
    }

    hdcpc::GetHdcpAuthenticationState(&state);
    EXPECT_EQ(nn::hdcp::HdcpAuthenticationState_Unauthenticated, state);

    // HDCPを有効にする。
    hdcpc::SetCurrentHdcpMode(nn::hdcp::HdcpMode_Enabled);

    // 認証中を示す状態はすぐに返ってくる。
    isSignaled = nn::os::TimedWaitSystemEvent(&eventType, nn::TimeSpan::FromMilliSeconds(300));
    ASSERT_EQ(true, isSignaled);
    nn::os::ClearSystemEvent(&eventType);
    hdcpc::GetHdcpAuthenticationState(&state);
    EXPECT_EQ(nn::hdcp::HdcpAuthenticationState_Processing, state);

    // 認証完了までの時間はごく小数のテレビで差が出る場合がある。
    isSignaled = nn::os::TimedWaitSystemEvent(&eventType, nn::TimeSpan::FromMilliSeconds(10 * 1000));
    ASSERT_EQ(true, isSignaled);
    nn::os::ClearSystemEvent(&eventType);
    hdcpc::GetHdcpAuthenticationState(&state);
    EXPECT_EQ(nn::hdcp::HdcpAuthenticationState_Authenticated, state);

    // HDCPを無効にする。
    hdcpc::SetCurrentHdcpMode(nn::hdcp::HdcpMode_Disabled);

    // 未認証を示す状態はすぐに返ってくる。
    isSignaled = nn::os::TimedWaitSystemEvent(&eventType, nn::TimeSpan::FromMilliSeconds(300));
    ASSERT_EQ(true, isSignaled);
    nn::os::ClearSystemEvent(&eventType);
    hdcpc::GetHdcpAuthenticationState(&state);
    EXPECT_EQ(nn::hdcp::HdcpAuthenticationState_Unauthenticated, state);
}

/**
 * @brief   解像度変更後も意図したイベントシグナルがされるか。
 */
TEST(Connected, ResolutionChange)
{
    nnt::hdcp::HdcpTestInitializer initializer;
    nnt::hdcp::HdcpRawTestInitializer rawInitializer;

    bool isSignaled;
    nn::hdcp::HdcpAuthenticationState state;

    // イベント登録。
    nn::os::SystemEventType eventType;
    hdcpc::GetHdcpStateTransitionEvent(&eventType);

    // HDCPを無効にする。状態遷移が起きなくなったらテストスタート。
    hdcpc::SetCurrentHdcpMode(nn::hdcp::HdcpMode_Disabled);
    while (nn::os::TimedWaitSystemEvent(&eventType, nn::TimeSpan::FromMilliSeconds(300)))
    {
        nn::os::ClearSystemEvent(&eventType);
    }

    hdcpc::GetHdcpAuthenticationState(&state);
    EXPECT_EQ(nn::hdcp::HdcpAuthenticationState_Unauthenticated, state);

    // 残念ながらアプリ(単体テスト)から正攻法で NotPlugged を確認できない。

    //// HDCPを有効にする。
    //hdcpc::SetCurrentHdcpMode(nn::hdcp::HdcpMode_Enabled);

    //// 認証中を示す状態はすぐに返ってくる。
    //isSignaled = nn::os::TimedWaitSystemEvent(&eventType, nn::TimeSpan::FromMilliSeconds(300));
    //ASSERT_EQ(true, isSignaled);
    //nn::os::ClearSystemEvent(&eventType);
    //hdcpc::GetHdcpAuthenticationState(&state);
    //EXPECT_EQ(nn::hdcp::HdcpAuthenticationState_Processing, state);

    //// 認証完了までの時間はごく小数のテレビで差が出る場合がある。
    //isSignaled = nn::os::TimedWaitSystemEvent(&eventType, nn::TimeSpan::FromMilliSeconds(10 * 1000));
    //ASSERT_EQ(true, isSignaled);
    //nn::os::ClearSystemEvent(&eventType);
    //hdcpc::GetHdcpAuthenticationState(&state);
    //EXPECT_EQ(nn::hdcp::HdcpAuthenticationState_Authenticated, state);

    //isSignaled = nn::os::TimedWaitSystemEvent(&eventType, nn::TimeSpan::FromMilliSeconds(3 * 1000));
    //ASSERT_EQ(false, isSignaled);

    //// 解像度を変更する。
    //rawInitializer.ChangeDisplayResolution();
    //hdcpc::GetHdcpAuthenticationState(&state);

    //isSignaled = nn::os::TimedWaitSystemEvent(&eventType, nn::TimeSpan::FromMilliSeconds(1000));
    //ASSERT_EQ(true, isSignaled);
    //nn::os::ClearSystemEvent(&eventType);
    //hdcpc::GetHdcpAuthenticationState(&state);
    //EXPECT_EQ(nn::hdcp::HdcpAuthenticationState_NotPlugged, state);

    //// 認証中を示す状態はすぐに返ってくる。
    //isSignaled = nn::os::TimedWaitSystemEvent(&eventType, nn::TimeSpan::FromMilliSeconds(1000));
    //ASSERT_EQ(true, isSignaled);
    //nn::os::ClearSystemEvent(&eventType);
    //hdcpc::GetHdcpAuthenticationState(&state);
    //EXPECT_EQ(nn::hdcp::HdcpAuthenticationState_Processing, state);

    //// 認証完了までの時間はごく小数のテレビで差が出る場合がある。
    //isSignaled = nn::os::TimedWaitSystemEvent(&eventType, nn::TimeSpan::FromMilliSeconds(10 * 1000));
    //ASSERT_EQ(true, isSignaled);
    //nn::os::ClearSystemEvent(&eventType);
    //hdcpc::GetHdcpAuthenticationState(&state);
    //EXPECT_EQ(nn::hdcp::HdcpAuthenticationState_Authenticated, state);

    // バックグラウンドで解像度変更処理を回しながらGetStateを叩いて NotPlugged が返ることを確認する。
    rawInitializer.StartThreadToImposeStressOnDpcd();
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));

    // 10回まわせば十分再現する。
    int trails = 10;
    for (int i1 = 0; i1 < trails && nn::hdcp::HdcpAuthenticationState_NotPlugged != state; ++i1)
    {
        // HDCPを有効にする。
        hdcpc::SetCurrentHdcpMode(nn::hdcp::HdcpMode_Enabled);

        int waitingTime[] = {
            100, 200, 400, 800
        };
        for (int time : waitingTime)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(time));
            hdcpc::GetHdcpAuthenticationState(&state);
            if (nn::hdcp::HdcpAuthenticationState_NotPlugged == state)
            {
                break;
            }
        }

        // HDCPを無効にする。
        hdcpc::SetCurrentHdcpMode(nn::hdcp::HdcpMode_Disabled);
    }
    EXPECT_EQ(nn::hdcp::HdcpAuthenticationState_NotPlugged, state);

    // バックグラウンドでの解像度変更を止める。
    rawInitializer.StopThreadToImposeStressOnDpcd();
} // NOLINT(impl/function_size)
