﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/vi.h>

#include "testGrc_Macro.h"
#include "testGrc_MemoryManagement.h"
#include "testGrc_NvnUtility.h"
#include "testGrc_SceneUtility.h"

NNT_GRC_TEST_SERVERSTATE(Sandbox)
{
    static const size_t ControlMemorySize     = 4 * 4096;
    static const size_t CommandPoolMemorySize = 10 * 4096;
    static const size_t FramePoolMemorySize = 8 * 1024 * 1024;

    static const int FrameCount = 600;

    InitializeGraphics();

    nn::vi::Initialize();
    NN_UTIL_SCOPE_EXIT{ nn::vi::Finalize(); };

    nn::vi::Display* pDisplay;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));
    NN_UTIL_SCOPE_EXIT{ nn::vi::CloseDisplay(pDisplay); };

    nn::vi::Layer* pDisplayLayer;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::vi::CreateLayer(&pDisplayLayer, pDisplay));

    nn::vi::NativeWindowHandle hNativeLayer;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::vi::GetNativeWindow(&hNativeLayer, pDisplayLayer));


    auto pControlMemory     = aligned_alloc(4096, ControlMemorySize);
    auto pCommandPoolMemory = aligned_alloc(4096, CommandPoolMemorySize);
    auto pFramePoolMemory   = aligned_alloc(4096, FramePoolMemorySize);
    NN_ABORT_UNLESS_NOT_NULL(pControlMemory);
    NN_ABORT_UNLESS_NOT_NULL(pCommandPoolMemory);
    NN_ABORT_UNLESS_NOT_NULL(pFramePoolMemory);
    NN_UTIL_SCOPE_EXIT{
        free(pControlMemory);
        free(pCommandPoolMemory);
        free(pFramePoolMemory);
    };

    // init nvn
    NNT_GRC_SCOPED_DEVICE           (device);
    NNT_GRC_SCOPED_COMMANDMEMORYPOOL(commandPool, device, pCommandPoolMemory, CommandPoolMemorySize);
    NNT_GRC_SCOPED_QUEUE            (queue, device);
    NNT_GRC_SCOPED_COMMANDBUFFER    (commandBuffer, device);
    NNT_GRC_SCOPED_TEXTUREMEMORYPOOL(framePool, device, pFramePoolMemory, FramePoolMemorySize);
    NNT_GRC_SCOPED_FRAMETEXTURE     (frameTexs, 2, 1280, 720, device, framePool, 0, FramePoolMemorySize);
    NNT_GRC_SCOPED_WINDOW           (window, frameTexs, 2, hNativeLayer, device);
    NNT_GRC_SCOPED_FRAMETEXTUREVIEW (texView);

    for(int frame = 0; frame < FrameCount; frame++)
    {
        NNT_GRC_SCOPED_SYNC(acqSync, device);
        int texIdx = -1;
        NN_ABORT_UNLESS_EQUAL(nvnWindowAcquireTexture(&window, &acqSync, &texIdx), NVN_WINDOW_ACQUIRE_TEXTURE_RESULT_SUCCESS);
        NN_LOG("Frame %d (%d)\n", frame, texIdx);
        auto pTargetTexture = windowTextureList[texIdx];

        nvnCommandBufferAddCommandMemory(&commandBuffer, &commandPool, 0, sizeof(CommandPoolMemorySize));
        nvnCommandBufferAddControlMemory(&commandBuffer, pControlMemory, ControlMemorySize);
        nvnCommandBufferBeginRecording(&commandBuffer);

        // wait display
        nvnCommandBufferWaitSync(&commandBuffer, &acqSync);

        nnt::grc::scene::DrawRotationBoxScene(&commandBuffer, pTargetTexture, &texView, frame / 60.f);

        auto hCommand = nvnCommandBufferEndRecording(&commandBuffer);
        nvnQueueSubmitCommands(&queue, 1, &hCommand);
        nvnQueuePresentTexture(&queue, &window, texIdx);
        nvnQueueFinish(&queue);
    }

}

