﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// テスト用のコードを有効にします。
// この定義を行わない場合、サンプルの Ui2dSimple と同じコードになります。
// 更新する際にはサンプルと diff を取ることで簡単にアップデートできるはずです。
#define NNT_UI2D_TESTCODE

#include <cstdlib>

#include <nn/gfx.h>

#include <nn/vi.h>

#include <nn/nn_Assert.h>

#include <nn/ui2d.h>
#include <nn/font.h>
#include <nn/font/font_ScalableFontUtil.h>
#include <nn/fs.h>
#include <nn/nn_Log.h>
#include <nns/gfx/gfx_GraphicsFramework.h>
#include <nn/util/util_Vector.h>
#include <nn/util/util_Matrix.h>
#include <nn/mem.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <nn/nn_Windows.h>
#endif

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#include <nn/init.h>
#endif

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
#include <nv/nv_MemoryManagement.h>
#endif

#include <nnt.h>
#include <nnt/graphics/testGraphics_Path.h>
#include <nnt/graphics/testGraphics_GetHostExecutableFilepath.h>
#include <nnt/graphics/testGraphics_CreateDirectories.h>
#include <nnt/graphics/testGraphics_InitializeAllocatorFunctionForStandardAllocator.h>
#include <nnt/graphics/testGraphics_Png.h>
#include "testUi2d_Dataset.h"
#include "testUi2d_ColorTargetViewCapture.h"
#include "testUi2d_CustomTagProcessor.h"

//------------------------------------------------------------------------------
//  変数の宣言
//------------------------------------------------------------------------------
static int g_AllocationCount = 0;

// アプリケーションヒープとして確保されるメモリ領域の大きさ(byte)
static const size_t ApplicationHeapSize = 256 * 1024 * 1024;

// アプリケーションヒープ
static nn::util::BytePtr g_pMemoryHeap(NULL);
static nn::mem::StandardAllocator g_ApplicationHeapHandle;

// ui2d レイアウト
static nn::ui2d::Layout* g_pLayout = NULL;
// ui2d アーカイブリソースアクセッサ
static nn::ui2d::ArcResourceAccessor* g_pArcResourceAccessor = NULL;
// レイアウトアーカイブバイナリデータ
static void* g_pLayoutArchiveBinary;
// ui2d が描画に使用するグラフィックスリソース
static nn::ui2d::GraphicsResource g_GraphicsResource;

// ui2d 描画の際に使用するコンスタントバッファ
static nn::font::GpuBuffer  g_Ui2dConstantBuffer;
// ui2d のコンスタントバッファのダブルバッファリング用インデックス
static uint32_t     g_Ui2dConstantBufferIndex = 0;
// ui2d 描画情報
static nn::ui2d::DrawInfo g_DrawInfo;
// アニメーションコントロール用の ui2d アニメータ
static nn::ui2d::GroupAnimator* g_pAnimator;
// 初回フレームキャプチャ済みフラグ

static const int CaptureImageWidth = 1280;
static const int CaptureImageHeight = 720;

// グラフィックスフレームワーク
nns::gfx::GraphicsFramework                 g_GfxFramework;
ptrdiff_t                                   g_offsetToUi2dDrawMemoryPool;
ptrdiff_t                                   g_offsetToRenderTargetTextureMemoryPoolHead = NULL;
ptrdiff_t                                   g_offsetToRenderTargetTextureMemoryPoolStatic = NULL;
ptrdiff_t                                   g_offsetToRenderTargetTextureMemoryPoolDynamic = NULL;

static const char* ContentBasePathRoot = "Contents:/Resources/";
static const char* LayoutArchiveSubPath = "testUi2d_RenderAndCapture.arc";
static const char* DataSetSubPath = "dataset.txt";
static const char* BinVerOptionName = "--binver=";


static std::string ContentBasePath;
static std::string LayoutArchivePath;
static std::string FontDirectoryPath;
static std::string DataSetPath;
static std::string OutputPath;

static nnt::ui2d::Dataset g_Dataset;

static const int MaxNumberOfFonts = 256;
int g_NumberOfFonts = 0;
nn::font::ResFont* g_Fonts[MaxNumberOfFonts];

static const int MaxNumberOfFcpxs = 256;
int g_NumberOfFcpxs = 0;
int g_FontFaceHead[MaxNumberOfFcpxs];
nn::font::Font* g_Fcpxs[MaxNumberOfFcpxs];
nn::font::TextureCache::InitializeArg* g_pTextureCacheInitializeArg = NULL;
nn::font::TextureCache* g_pTextureCache = NULL;

static const int g_FileCount = 5;
static const char* g_FileNames[g_FileCount] =
{
    "sample.bffnt",
    "nintendo_NTLG-DB_002.bfttf",
    "color.bffnt",
    "IconFont-Regular.ttf",
    "NotoEmoji-Regular.ttf",
};
static size_t g_FontBinarySize[g_FileCount] = { 0 };
static void* g_pFontBinary[g_FileCount] = { NULL };
static void* g_pFontRawResource[g_FileCount] = { NULL };

static nnt::ui2d::CustomTagProcessor* g_pCustomTagProcessor = NULL;

//------------------------------------------------------------------------------
//  アプリケーションヒープからメモリを確保する
//------------------------------------------------------------------------------
static void* AllocateFromApplicationHeap(size_t size, size_t alignment)
{
    void* ptr = g_ApplicationHeapHandle.Allocate(size, alignment);
    if (ptr != NULL)
    {
        g_AllocationCount++;
    }
    return ptr;
}

static void* AllocateFromApplicationHeap(size_t size)
{
    return AllocateFromApplicationHeap(size, 4);
}

//------------------------------------------------------------------------------
//  アプリケーションヒープから確保したメモリを開放する
//------------------------------------------------------------------------------
static void DeallocateApplicationHeap(void* ptr)
{
    if (ptr == NULL)
    {
        return;
    }
    g_AllocationCount--;
    g_ApplicationHeapHandle.Free(ptr);
}

template <typename TObject>
TObject* AllocAndConstruct()
{
    return new(AllocateFromApplicationHeap(sizeof(TObject))) TObject();
}

template <typename TObject>
void DestructAndFree(TObject* ptr)
{
    ptr->TObject::~TObject();
    DeallocateApplicationHeap(ptr);
}

//------------------------------------------------------------------------------
// ui2d 用メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
static void* Ui2dAllocateFunction(size_t size, size_t alignment, void* pUserData)
{
    NN_UNUSED(pUserData);
    return AllocateFromApplicationHeap(size, alignment);
}

static void Ui2dDeallocateFunction(void* ptr, void* pUserData)
{
    NN_UNUSED(pUserData);
    DeallocateApplicationHeap(ptr);
}

//------------------------------------------------------------------------------
// nn::fs用メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
static void* FsAllocateFunction( size_t size )
{
    return malloc(size);
}

static void FsDeallocateFunction( void* ptr, size_t size )
{
    NN_UNUSED( size );
    free(ptr);
}

//------------------------------------------------------------------------------

bool RegisterSlotForTexture_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::TextureView& textureView, void* pUserData)
{
    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);

        const int slotIndex = graphicsFramework->AllocateDescriptorSlot( nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView, 1 );
        if (slotIndex == nn::gfx::util::DescriptorPoolAllocator::InvalidIndex)
        {
            return false;
        }

        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView);

        pPool->BeginUpdate();
        {
            pPool->SetTextureView(slotIndex, &textureView);
        }
        pPool->EndUpdate();
        pPool->GetDescriptorSlot(pDstSlot, slotIndex);
        return true;
    }

    return false;
}

//------------------------------------------------------------------------------

bool RegisterSlotForSampler_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::Sampler& sampler, void* pUserData)
{
    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);

        const int slotIndex = graphicsFramework->AllocateDescriptorSlot( nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler, 1 );
        if (slotIndex == nn::gfx::util::DescriptorPoolAllocator::InvalidIndex)
        {
            return false;
        }

        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler);

        pPool->BeginUpdate();
        {
            pPool->SetSampler(slotIndex, &sampler);
        }
        pPool->EndUpdate();
        pPool->GetDescriptorSlot(pDstSlot, slotIndex);
        return true;
    }

    return false;
}

//------------------------------------------------------------------------------

void UnregisterSlotForSampler_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::Sampler& sampler, void* pUserData)
{
    NN_UNUSED(sampler);
    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);
        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler);
        const int slotIndex = pPool->GetDescriptorSlotIndex(*pDstSlot);
        graphicsFramework->FreeDescriptorSlot(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler, slotIndex);
    }

}

//------------------------------------------------------------------------------

void UnregisterSlotForTexture_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::TextureView& textureView, void* pUserData)
{
    NN_UNUSED(textureView);
    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);
        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView);
        int slotIndex = pPool->GetDescriptorSlotIndex(*pDstSlot);
        graphicsFramework->FreeDescriptorSlot(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView, slotIndex);
    }
}

//------------------------------------------------------------------------------
// メモリを確保してファイルを読み込みます
//------------------------------------------------------------------------------
void* ReadFileWithAllocate(const char* pFileName, size_t alignment)
{
    nn::Result result;
    nn::fs::FileHandle fileHandle;
    int64_t fileSize;

    NN_LOG("Loading file: %s ...", pFileName);

    result = nn::fs::OpenFile(&fileHandle, pFileName, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    result = nn::fs::GetFileSize(&fileSize, fileHandle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    void* binary = AllocateFromApplicationHeap(static_cast<size_t>(fileSize), alignment);
    size_t readSize;
    result = nn::fs::ReadFile(&readSize, fileHandle, 0, binary, static_cast<size_t>(fileSize));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::fs::CloseFile(fileHandle);

    NN_LOG("ok\n", pFileName);

    return binary;
}

//-----------------------------------------------------------------------------
// ファイルシステムの初期化
//-----------------------------------------------------------------------------
void InitializeFileSystem()
{
    nn::fs::SetAllocator( FsAllocateFunction, FsDeallocateFunction );

    nn::Result result;

    nnt::graphics::Path contentsPath(nnt::graphics::GetHostExecutableFilepath());
    contentsPath.MakeParent();
    contentsPath.MakeParent();
    contentsPath.CombineAssign("Contents");

#if defined(NN_BUILD_APISET_NX)
    contentsPath.CombineAssign("NX");
#elif defined(NN_BUILD_CONFIG_SPEC_GENERIC)
    contentsPath.CombineAssign("Generic");
#else
    NN_ABORT("Unexpected spec.");
#endif
    NN_LOG("Mount Contents: %s\n", contentsPath.GetString());
    result = nn::fs::MountHost("Contents", contentsPath.GetString());
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nnt::graphics::Path outputsPath(nnt::graphics::GetHostExecutableFilepath());
    outputsPath.MakeParent();
    NN_LOG("Mount Outputs: %s\n", outputsPath.GetString());
    result = nn::fs::MountHost("Outputs", outputsPath.GetString());
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    result = nnt::graphics::CreateDirectories(OutputPath.c_str());
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    NN_UNUSED(result);
}

//------------------------------------------------------------------------------
// フォントの登録
//------------------------------------------------------------------------------
static void RegisterDatasetFonts(const char* fontname, void* param)
{
    NN_UNUSED(param);
    NN_ASSERT(g_NumberOfFonts + 1 < MaxNumberOfFonts);

    nn::gfx::Device*    pDevice = g_GfxFramework.GetDevice();

    // フォントの初期化
    nn::font::ResFont* pFont = AllocAndConstruct<nn::font::ResFont>();

    // フォントの読み込み
    nnt::graphics::Path fontPath(FontDirectoryPath.c_str());
    fontPath.CombineAssign(fontname);

    void* pFontBinary = ReadFileWithAllocate(fontPath.GetString(), nn::font::ResourceAlignment);
    bool    result = pFont->SetResource(pDevice, pFontBinary);
    NN_ASSERT(result);
    pFont->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture_, &g_GfxFramework);

    g_pArcResourceAccessor->RegisterFont(fontname, pFont);

    g_Fonts[g_NumberOfFonts] = pFont;
    g_NumberOfFonts++;
}

//------------------------------------------------------------------------------
// フォント名からフォントデータを探すコールバック関数
//------------------------------------------------------------------------------
static void* AcquireFontFunctionForComplexFont(size_t* pOutFontDataSize, const char* pFontName, nn::ui2d::ResType resType, void* pUserData)
{
    for (int i = 0; i < g_FileCount; i++)
    {
        // フォントを直接ロードする場合
        if (strcmp(pFontName, g_FileNames[i]) == 0)
        {
            if (g_pFontBinary[i] != NULL)
            {
                *pOutFontDataSize = g_FontBinarySize[i];
                return g_pFontBinary[i];
            }
            else
            {
                nn::Result result;
                nn::fs::FileHandle fileHandle;
                int64_t fileSize;

                char path[256];
                strcpy(path, ContentBasePath.c_str());
                strcat(path, g_FileNames[i]);
                result = nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode_Read);
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                result = nn::fs::GetFileSize(&fileSize, fileHandle);
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                void* pFontFileImage = Ui2dAllocateFunction(static_cast<size_t>(fileSize), nn::font::ResourceAlignment, NULL);
                g_pFontRawResource[i] = pFontFileImage;

                size_t readSize;
                result = nn::fs::ReadFile(&readSize, fileHandle, 0, pFontFileImage, static_cast<size_t>(fileSize));
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                g_FontBinarySize[i] = static_cast<size_t>(fileSize);
                g_pFontBinary[i] = pFontFileImage;

                nn::fs::CloseFile(fileHandle);

                NN_ASSERT_NOT_NULL(g_pFontBinary[i]);
                *pOutFontDataSize = g_FontBinarySize[i];
                return g_pFontBinary[i];
            }
        }
    }

    // リソース内からデータを探す場合
    {
        nn::ui2d::ResourceAccessor* pResAccsessor = reinterpret_cast<nn::ui2d::ResourceAccessor*>(pUserData);
        void* pFont = pResAccsessor->FindResourceByName(pOutFontDataSize, resType, pFontName);
        NN_ASSERT(pFont != NULL, "The font resource [%s] was not found.\n", pFontName);
        return pFont;
    }
}

//------------------------------------------------------------------------------
// fcpx の読み込み
//------------------------------------------------------------------------------
static void LoadDatasetFcpxs(const char* fcpxname, void* param)
{
    NN_UNUSED(param);
    NN_ASSERT(g_NumberOfFcpxs + 1 < MaxNumberOfFcpxs);

    char name[256];
    strcpy(name, fcpxname);
    strcat(name, ".bfcpx");
    const void* pFcpxData = g_pArcResourceAccessor->FindResourceByName(nn::ui2d::ResourceTypeComplexFont, name);

    if (g_pTextureCacheInitializeArg == NULL)
    {
        // fcpx が 1 つでもあれば InitializeArg を初期化する
        g_pTextureCacheInitializeArg = static_cast<nn::font::TextureCache::InitializeArg*>(nn::ui2d::Layout::AllocateMemory(sizeof(nn::font::TextureCache::InitializeArg)));
        g_pTextureCacheInitializeArg->SetDefault();
        g_pTextureCacheInitializeArg->workMemorySize = 1024 * 1024;
        g_pTextureCacheInitializeArg->noPlotWorkMemorySize = 1024 * 1024;
        g_pTextureCacheInitializeArg->allocateFunction = nn::ui2d::Layout::GetAllocateFunction();
        g_pTextureCacheInitializeArg->textureCacheWidth = 1024;
        g_pTextureCacheInitializeArg->textureCacheHeight = 1024;
    }

    // ユーティリティ関数を使って bfcpx から TextureCacheArg を構築する
    g_FontFaceHead[g_NumberOfFcpxs] = nn::ui2d::ComplexFontHelper::SetupTextureCacheArg(
        g_pTextureCacheInitializeArg,
        AcquireFontFunctionForComplexFont,
        g_pArcResourceAccessor,
        pFcpxData);
    g_NumberOfFcpxs++;
}

//------------------------------------------------------------------------------
// TextureCache の初期化
//------------------------------------------------------------------------------
static void InitializeTextureCache()
{
    if (g_pTextureCacheInitializeArg == NULL)
    {
        // fcpx が 1 つもない場合は TextureCache の初期化をスキップする
        return;
    }

    nn::gfx::Device*    pDevice = g_GfxFramework.GetDevice();

    g_pTextureCache = AllocAndConstruct<nn::font::TextureCache>();
    g_pTextureCache->Initialize(pDevice, *g_pTextureCacheInitializeArg);
    g_pTextureCache->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture_, &g_GfxFramework);

    nn::ui2d::Layout::FreeMemory(g_pTextureCacheInitializeArg);
    g_pTextureCacheInitializeArg = NULL;
    g_NumberOfFcpxs = 0;
}

//------------------------------------------------------------------------------
// fcpx の登録
//------------------------------------------------------------------------------
static void RegisterDatasetFcpxs(const char* fcpxname, void* param)
{
    NN_UNUSED(param);
    NN_ASSERT(g_NumberOfFcpxs + 1 < MaxNumberOfFcpxs);

    nn::gfx::Device*    pDevice = g_GfxFramework.GetDevice();

    char name[256];
    strcpy(name, fcpxname);
    strcat(name, ".bfcpx");
    const void* pFcpxData = g_pArcResourceAccessor->FindResourceByName(nn::ui2d::ResourceTypeComplexFont, name);

    g_Fcpxs[g_NumberOfFcpxs] = nn::ui2d::ComplexFontHelper::InitializeComplexFontTree(
        pDevice,
        RegisterSlotForTexture_,
        &g_GfxFramework,
        g_pTextureCache,
        g_FontFaceHead[g_NumberOfFcpxs],
        AcquireFontFunctionForComplexFont,
        g_pArcResourceAccessor,
        pFcpxData);

    strcpy(name, fcpxname);
    strcat(name, ".fcpx");
    g_pArcResourceAccessor->RegisterFont(name, g_Fcpxs[g_NumberOfFcpxs]);

    g_NumberOfFcpxs++;
}

//------------------------------------------------------------------------------
// "Emoticon" という名前のペインがあれば絵文字テキストボックスを登録する
//------------------------------------------------------------------------------
static void RegisterEmoticonTextBox()
{
    nn::ui2d::TextBox* pEmoticon = static_cast<nn::ui2d::TextBox*>(g_pLayout->GetRootPane()->FindPaneByName("Emoticon"));
    if (pEmoticon == NULL)
    {
        // "Emoticon" という名前のペインは存在しなかった
        return;
    }

    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();
    nn::ui2d::TextBox* pTextBox = nn::ui2d::Layout::AllocateAndConstruct<nn::ui2d::TextBox>(true);
    pTextBox->AllocateStringBuffer(pDevice, 32);
    pTextBox->SetFont(pEmoticon->GetFont());
    pTextBox->SetSize(pEmoticon->GetSize());
    {
        char utf8Text[32];
        // サロゲートペア領域の絵文字「😀😁😂」
        uint32_t utf32Text[4] = { 0x1f600, 0x1f601, 0x1f602, 0 };
        nn::util::CharacterEncodingResult result = nn::util::ConvertStringUtf32ToUtf8(utf8Text, sizeof(utf8Text) / sizeof(utf8Text[0]), utf32Text);
        ASSERT_TRUE(result == nn::util::CharacterEncodingResult_Success);
        pTextBox->SetStringUtf8(utf8Text);
    }
    pEmoticon->AppendChild(pTextBox);
}

//------------------------------------------------------------------------------
// "CustomTagProcessor" という名前のペインがあれば TagProcessor を差し替える
//------------------------------------------------------------------------------
static void SetCustomTagProcessor()
{
    nn::ui2d::TextBox* pCustomTagProcessor = static_cast<nn::ui2d::TextBox*>(g_pLayout->GetRootPane()->FindPaneByName("CustomTagProcessor"));
    if (pCustomTagProcessor == NULL)
    {
        // "CustomTagProcessor" という名前のペインは存在しなかった
        return;
    }

    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();
    nn::font::Font* pDefaultFont = g_pArcResourceAccessor->AcquireFont(pDevice, "ComplexFont4.fcpx");
    nn::font::Font* pAlternativeFont = g_pArcResourceAccessor->AcquireFont(pDevice, "sample.bffnt");
    g_pCustomTagProcessor = static_cast<nnt::ui2d::CustomTagProcessor*>(AllocateFromApplicationHeap(sizeof(nnt::ui2d::CustomTagProcessor)));
    new (g_pCustomTagProcessor) nnt::ui2d::CustomTagProcessor(pDefaultFont, pAlternativeFont);

    pCustomTagProcessor->SetTagProcessor(g_pCustomTagProcessor);
}

//------------------------------------------------------------------------------
// TextureCache のアップデート
//------------------------------------------------------------------------------
static void UpdateTextureCache()
{
    if (g_pTextureCache == NULL)
    {
        return;
    }
    nn::font::RegisterGlyphFromTextBoxRecursive(g_pLayout->GetRootPane());
    g_pTextureCache->UpdateTextureCache();
    g_pTextureCache->CompleteTextureCache();
}

//------------------------------------------------------------------------------
nn::ui2d::RenderTargetTextureLifetime CreateRenderTargetTexture(nn::gfx::Texture** pTexture, nn::gfx::TextureView** pTextureView, nn::gfx::DescriptorSlot** pSlot, const nn::ui2d::Layout* pLayout, const nn::gfx::TextureInfo& info, void* pUserData, nn::ui2d::RenderTargetTextureLifetime lifetimeHint)
{
    NN_UNUSED(pLayout);

    NN_SDK_ASSERT_NOT_NULL(pTexture);
    NN_SDK_ASSERT_NOT_NULL(pTextureView);
    NN_SDK_ASSERT_NOT_NULL(pSlot);

    nns::gfx::GraphicsFramework* pFramework = static_cast<nns::gfx::GraphicsFramework*>(pUserData);
    nn::gfx::Device*    pDevice = pFramework->GetDevice();

    const size_t size = nn::gfx::Texture::CalculateMipDataSize(pDevice, info);
    const size_t mipDataAlignment = nn::gfx::Texture::CalculateMipDataAlignment(pDevice, info);

    *pTexture = AllocAndConstruct<nn::gfx::Texture>();

    // テクスチャの初期化
    switch (lifetimeHint)
    {
    case nn::ui2d::RenderTargetTextureLifetime_Layout:
        // nn::gfx::ImageFormat_R8_Unorm のような 1 要素で中途半端なサイズのテクスチャを作成することがあるため、テクスチャ作成時のアライメントを調整する必要があります。
        g_offsetToRenderTargetTextureMemoryPoolStatic = nn::util::align_up(g_offsetToRenderTargetTextureMemoryPoolStatic, mipDataAlignment);
        (*pTexture)->Initialize(pDevice, info, pFramework->GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget), g_offsetToRenderTargetTextureMemoryPoolStatic, size);
        g_offsetToRenderTargetTextureMemoryPoolStatic += size;
        NN_SDK_ASSERT(g_offsetToRenderTargetTextureMemoryPoolStatic - g_offsetToRenderTargetTextureMemoryPoolHead < 8 * 1024 * 1024);
        break;
    case nn::ui2d::RenderTargetTextureLifetime_OneFrame:
        // nn::gfx::ImageFormat_R8_Unorm のような 1 要素で中途半端なサイズのテクスチャを作成することがあるため、テクスチャ作成時のアライメントを調整する必要があります。
        g_offsetToRenderTargetTextureMemoryPoolDynamic = nn::util::align_up(g_offsetToRenderTargetTextureMemoryPoolDynamic, mipDataAlignment);
        (*pTexture)->Initialize(pDevice, info, pFramework->GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget), g_offsetToRenderTargetTextureMemoryPoolDynamic, size);
        g_offsetToRenderTargetTextureMemoryPoolDynamic += size;
        NN_SDK_ASSERT(g_offsetToRenderTargetTextureMemoryPoolDynamic - g_offsetToRenderTargetTextureMemoryPoolHead < 16 * 1024 * 1024);
        break;
    default:
        break;
    }

    // TextureView
    {
        nn::gfx::TextureView::InfoType textureViewInfo;
        textureViewInfo.SetDefault();
        textureViewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
        textureViewInfo.EditSubresourceRange().EditArrayRange().SetArrayLength(1);
        textureViewInfo.EditSubresourceRange().EditArrayRange().SetBaseArrayIndex(0);
        textureViewInfo.SetImageFormat(info.GetImageFormat());
        textureViewInfo.SetTexturePtr(*pTexture);

        *pTextureView = AllocAndConstruct<nn::gfx::TextureView>();

        (*pTextureView)->Initialize(pDevice, textureViewInfo);
    }

    *pSlot = AllocAndConstruct<nn::gfx::DescriptorSlot>();

    RegisterSlotForTexture_(*pSlot, **pTextureView, pFramework);

    return lifetimeHint;
}

//------------------------------------------------------------------------------
void DestroyRenderTargetTexture(nn::gfx::Texture* pTexture, nn::gfx::TextureView* pTextureView, nn::gfx::DescriptorSlot* pSlot, const nn::ui2d::Layout* pLayout, void* pUserData, nn::ui2d::RenderTargetTextureLifetime lifetimeHint)
{
    NN_UNUSED(pLayout);
    NN_UNUSED(lifetimeHint);

    nns::gfx::GraphicsFramework* pFramework = static_cast<nns::gfx::GraphicsFramework*>(pUserData);
    nn::gfx::Device*    pDevice = pFramework->GetDevice();

    UnregisterSlotForTexture_(pSlot, *pTextureView, pUserData);
    pTextureView->Finalize(pDevice);
    pTexture->Finalize(pDevice);

    DestructAndFree(pSlot);
    DestructAndFree(pTextureView);
    DestructAndFree(pTexture);
}

//------------------------------------------------------------------------------
// ui2d 描画のためのバッファ関連初期化処理
//------------------------------------------------------------------------------
void InitializeUi2dBuffers(nn::ui2d::DrawInfo& drawInfo, const nn::ui2d::BuildResultInformation& buildResultInformation)
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    // バッファの再セットアップ
    // ui2d 描画に使用する各種 gfx バッファのためのメモリプールを作成
    nn::gfx::MemoryPoolInfo memPoolInfo;
    memPoolInfo.SetDefault();
    memPoolInfo.SetMemoryPoolProperty( nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached );

    nn::gfx::GpuAccess  gpuAccessFlags = static_cast<nn::gfx::GpuAccess>(nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_IndexBuffer);
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetGpuAccessFlags(gpuAccessFlags);

    size_t  bufferAlignment = nn::gfx::Buffer::GetBufferAlignment(pDevice, info);

    // BuildWithName でレイアウトデータをセットアップした際に BuildResultInformation に収集した各種バッファのサイズを満たすメモリプールを作成する。
    const size_t oneConstantBufferSize = nn::util::align_up(buildResultInformation.requiredUi2dConstantBufferSize + buildResultInformation.requiredFontConstantBufferSize, bufferAlignment);
    const size_t constantBufferTotalSize = oneConstantBufferSize * nn::ui2d::ConstantBufferCount;
    const size_t memoryPoolGranularity = nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(pDevice, memPoolInfo);
    const size_t memoryPoolSize = nn::util::align_up(constantBufferTotalSize, memoryPoolGranularity);
    const size_t MemoryPoolAlignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(pDevice, memPoolInfo);

    if (memoryPoolSize > 0)
    {
        const nns::gfx::GraphicsFramework::MemoryPoolType constantBufferType = nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer;
        const size_t totalMemoryPoolSize = memoryPoolSize * nn::ui2d::ConstantBufferCount;

        nn::gfx::MemoryPool* pMemoryPool = g_GfxFramework.GetMemoryPool(constantBufferType);
        g_offsetToUi2dDrawMemoryPool = g_GfxFramework.AllocatePoolMemory(constantBufferType, totalMemoryPoolSize, MemoryPoolAlignment);

        // ui2d 用のコンスタントバッファを作成する。
        if (buildResultInformation.requiredUi2dConstantBufferSize > 0)
        {
            nn::font::GpuBuffer::InitializeArg  arg;

            arg.SetBufferSize(oneConstantBufferSize);
            arg.SetBufferCount(nn::ui2d::ConstantBufferCount);
            arg.SetGpuAccessFlag(gpuAccessFlags);
            arg.SetMemoryPool(pMemoryPool);
            arg.SetMemoryPoolOffset(g_offsetToUi2dDrawMemoryPool);
            arg.SetAllocator(Ui2dAllocateFunction, NULL);
            arg.SetAllocateSyncFlag(false);

            g_Ui2dConstantBuffer.Initialize(pDevice, arg);
            drawInfo.SetUi2dConstantBuffer(&g_Ui2dConstantBuffer);
            drawInfo.SetFontConstantBuffer(&g_Ui2dConstantBuffer);
        }
        else
        {
            drawInfo.SetUi2dConstantBuffer(NULL);
            drawInfo.SetFontConstantBuffer(NULL);
        }
    }
}

// 動的なレイアウトデータ編集のためのコールバック関数
typedef void(*ModifyLayoutDataCallback)(nn::ui2d::Layout& layout, nn::ui2d::BuildResultInformation& buildResultInformation);

//------------------------------------------------------------------------------
// ui2d 関連の初期化処理
//------------------------------------------------------------------------------
void InitializeUi2dRuntime(const char* pLayoutName, bool isAnimationData, bool perspectiveProj, ModifyLayoutDataCallback pModifyFunc)
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    // レイアウトライブラリの初期化
    nn::ui2d::Initialize(Ui2dAllocateFunction, Ui2dDeallocateFunction, NULL);
    nn::ui2d::Layout::SetRenderTargetTextureResourceManagementCallback(CreateRenderTargetTexture, DestroyRenderTargetTexture, &g_GfxFramework);

    // リソースアクセサの初期化
    g_pArcResourceAccessor = AllocAndConstruct<nn::ui2d::ArcResourceAccessor>();

    // レイアウトアーカイブの読み込み
    g_pLayoutArchiveBinary = ReadFileWithAllocate(LayoutArchivePath.c_str(), nn::ui2d::ArchiveResourceAlignment);
    {
        bool    result = g_pArcResourceAccessor->Attach(g_pLayoutArchiveBinary, ".");
        NN_ASSERT(result);
    }

    // フォントの読み込み
    g_Dataset.ForEachFont(RegisterDatasetFonts, NULL);

    // fcpx の読み込み
    g_Dataset.ForEachFcpx(LoadDatasetFcpxs, NULL);
    InitializeTextureCache();
    g_Dataset.ForEachFcpx(RegisterDatasetFcpxs, NULL);

    // レイアウトの初期化
    g_pLayout = AllocAndConstruct<nn::ui2d::Layout>();

    nn::ui2d::Layout::BuildOption   opt;
    opt.SetDefault();
    nn::ui2d::BuildResultInformation   buildResultInformation;
    buildResultInformation.SetDefault();
    if (!g_pLayout->BuildWithName(&buildResultInformation, pDevice, g_pArcResourceAccessor, NULL, NULL, opt, pLayoutName))
    {
        NN_LOG("Failed to build the layout data[%s].", pLayoutName);
    }

    if (pModifyFunc != NULL)
    {
        (*pModifyFunc)(*g_pLayout, buildResultInformation);
    }

    // グラフィックスリソースの設定
    g_GraphicsResource.Setup(pDevice, 512);
    g_GraphicsResource.RegisterCommonSamplerSlot(RegisterSlotForSampler_, &g_GfxFramework);

    // Ui2d の描画に使用される各種バッファを初期化して DrawInfo へ設定する
    InitializeUi2dBuffers(g_DrawInfo, buildResultInformation);

    // 描画に使用する情報の設定
    {
        nn::util::MatrixT4x4fType   projection;
        nn::util::MatrixT4x3fType   view;
        nn::util::Vector3fType  pos;
        nn::util::Vector3fType  up;
        nn::util::Vector3fType  target;

        if (perspectiveProj)
        {
            // レイアウトエディタでのプレビュー時に使用される設定で行列を作成します。
            const float fovy = nn::util::DegreeToRadian(45.0f);
            const float height = static_cast<float>(g_GfxFramework.GetDisplayHeight()) * 0.5f / nn::util::TanTable(nn::util::RadianToAngleIndex(fovy / 2.f));
            const float aspect = static_cast<float>(g_GfxFramework.GetDisplayWidth()) / static_cast<float>(g_GfxFramework.GetDisplayHeight());
            nn::util::MatrixPerspectiveFieldOfViewRightHanded(&projection, fovy, aspect, 1.0f, 10000.0f);

            nn::util::VectorSet(&pos, 0.0f, 0.0f, height);
            nn::util::VectorSet(&up, 0.0f, 1.0f, 0.0f);
            nn::util::VectorSet(&target, 0.0f, 0.0f, 0.0f);
        }
        else
        {
            nn::font::Rectangle rect = g_pLayout->GetLayoutRect();
            nn::util::MatrixOrthographicOffCenterRightHanded(
                &projection,
                rect.left,
                rect.right,
                rect.bottom,
                rect.top,
                0.0f,
                300.0f);

            nn::util::VectorSet(&pos, 0.0f, 0.0f, 1.0f);
            nn::util::VectorSet(&up, 0.0f, 1.0f, 0.0f);
            nn::util::VectorSet(&target, 0.0f, 0.0f, 0.0f);
        }
        nn::util::MatrixLookAtRightHanded(
            &view,
            pos,
            target,
            up);

        g_DrawInfo.SetGraphicsResource(&g_GraphicsResource);
        g_DrawInfo.SetProjectionMtx(projection);
        g_DrawInfo.SetViewMtx(view);
    }

    // アニメーションの作成と再生
    if(isAnimationData)
    {
        g_pAnimator = g_pLayout->CreateGroupAnimator(g_GfxFramework.GetDevice(), g_Dataset.GetAnimatorTag());
        g_pAnimator->PlayAuto(1.0f);
    }

    // テクスチャパターンアニメーションに利用されるテクスチャが正しく登録されるように、アニメーションの初期化後に実行します。
    g_pArcResourceAccessor->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture_, &g_GfxFramework);

    // "Emoticon" という名前のペインがあれば絵文字テキストボックスを登録する
    RegisterEmoticonTextBox();

    // "CustomTagProcessor" という名前のペインがあれば TagProcessor を差し替える
    SetCustomTagProcessor();

    // TextureCache のアップデート
    UpdateTextureCache();

    // キャプチャペインのためにフレームバッファテクスチャを設定する
    g_DrawInfo.SetFramebufferTexture(g_GfxFramework.GetColorBuffer(), g_GfxFramework.GetDisplayWidth(), g_GfxFramework.GetDisplayHeight());

    // シザーペインの設定復帰用にデフォルトのビューポートとシザー設定を行う。
    nn::gfx::ViewportStateInfo  viewportInfo;
    viewportInfo.SetDefault();
    viewportInfo.SetWidth(static_cast<float>(g_GfxFramework.GetDisplayWidth()));
    viewportInfo.SetHeight(static_cast<float>(g_GfxFramework.GetDisplayHeight()));

    nn::gfx::ScissorStateInfo  scissorInfo;
    scissorInfo.SetDefault();
    scissorInfo.SetWidth(g_GfxFramework.GetDisplayWidth());
    scissorInfo.SetHeight(g_GfxFramework.GetDisplayHeight());
    g_DrawInfo.SetDefaultViewportScissorInfo(viewportInfo, scissorInfo);

    // 描画中にランタイム内部でレンダーターゲットを切り替えたときの復帰用にカラーターゲットビューを設定する。
    g_DrawInfo.SetDefaultColorTargetView(g_GfxFramework.GetColorTargetView());
}

//------------------------------------------------------------------------------
// ui2d 関連の終了処理
//------------------------------------------------------------------------------
void FinalizeUi2dRuntime()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    g_pLayout->Finalize(pDevice);
    DestructAndFree<nn::ui2d::Layout>(g_pLayout);

    g_Ui2dConstantBuffer.Finalize(pDevice, Ui2dDeallocateFunction, NULL);

    g_GraphicsResource.UnregisterCommonSamplerSlot(UnregisterSlotForSampler_, &g_GfxFramework);
    g_GraphicsResource.Finalize(pDevice);

    g_pArcResourceAccessor->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture_, &g_GfxFramework);
    g_pArcResourceAccessor->Detach();
    g_pArcResourceAccessor->Finalize(pDevice);
    DestructAndFree<nn::ui2d::ArcResourceAccessor>(g_pArcResourceAccessor);

    for(int i = 0; i < g_NumberOfFonts; i++)
    {
        auto& pFont = g_Fonts[i];
        pFont->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture_, &g_GfxFramework);
        void* pFontResource = pFont->RemoveResource(pDevice);
        pFont->Finalize(pDevice);
        DeallocateApplicationHeap(pFontResource);

        DestructAndFree<nn::font::ResFont>(pFont);
    }
    g_NumberOfFonts = 0;

    for (int i = 0; i < g_NumberOfFcpxs; i++)
    {
        nn::ui2d::ComplexFontHelper::FinalizeComplexFontTree(
            pDevice,
            g_Fcpxs[i],
            UnregisterSlotForTexture_,
            &g_GfxFramework);
        g_Fcpxs[i] = NULL;
    }
    g_NumberOfFcpxs = 0;

    if (g_pTextureCache != NULL)
    {
        g_pTextureCache->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture_, &g_GfxFramework);
        g_pTextureCache->Finalize(pDevice, nn::ui2d::Layout::GetFreeFunction(), NULL);
        DestructAndFree<nn::font::TextureCache>(g_pTextureCache);
        g_pTextureCache = NULL;
    }

    for (int i = 0; i < g_FileCount; i++)
    {
        Ui2dDeallocateFunction(g_pFontRawResource[i], NULL);

        g_FontBinarySize[i] = 0;
        g_pFontBinary[i] = NULL;
        g_pFontRawResource[i] = NULL;
    }

    if (g_pCustomTagProcessor != NULL)
    {
        DestructAndFree<nnt::ui2d::CustomTagProcessor>(g_pCustomTagProcessor);
        g_pCustomTagProcessor = NULL;
    }

    // ArcResourceAccessor の Finalize でアクセスされるため最後に開放する。
    DeallocateApplicationHeap(g_pLayoutArchiveBinary);
}

//------------------------------------------------------------------------------
// 画面キャプチャの初期化処理
//------------------------------------------------------------------------------
nnt::graphics::AllocatorFunction g_TestGraphicsAllocatorFunction;
nnt::ui2d::ColorTargetViewCapture g_ColorTargetViewCapture;
void InitializeFrameCapture()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    nnt::graphics::InitializeAllocatorFunctionForStandardAllocator::Initialize(
        &g_TestGraphicsAllocatorFunction,
        &g_ApplicationHeapHandle);

    g_ColorTargetViewCapture.Initialize(
        pDevice,
        g_GfxFramework,
        &g_TestGraphicsAllocatorFunction,
        CaptureImageWidth,
        CaptureImageHeight);
}
void FinalizeFrameCapture()
{
    g_ColorTargetViewCapture.Finalize(g_GfxFramework);
}

//------------------------------------------------------------------------------
// メインループ を中断するかどうかを判定
//------------------------------------------------------------------------------
bool CheckToBreakMainLoop()
{
#if defined(NN_BUILD_CONFIG_OS_WIN)
    // WindowsMessege の処理
    {
        MSG  msg;
        if ( PeekMessage( &msg, NULL, 0, 0, PM_REMOVE) )
        {
            TranslateMessage(&msg);
            if ( msg.message == WM_QUIT)
            {
                return true;
            }
            DispatchMessage(&msg);
        }
    }
#elif !defined(NN_BUILD_CONFIG_OS_HORIZON)
    if(g_Pad.IsHold(g_Pad.MASK_L|g_Pad.MASK_R))
    {
        return true;
    }
#endif

    return false;
};

//------------------------------------------------------------------------------
// メインループ (ループを中断する際は true を返します)
//------------------------------------------------------------------------------
bool MainLoop(int frame, bool isCapture, const char* pDir, const char* pCaptureDataName)
{
    // ループを中断する必要があるかどうか。
    if(CheckToBreakMainLoop())
    {
        return true;
    }

    // レイアウトの更新処理
    g_DrawInfo.Map(g_Ui2dConstantBufferIndex);
    {
        g_pLayout->AnimateAndUpdateAnimFrame();
        g_pLayout->Calculate(g_DrawInfo);
    }
    g_DrawInfo.Unmap();
    g_DrawInfo.SetGpuAccessBufferIndex(g_Ui2dConstantBufferIndex);

    ++g_Ui2dConstantBufferIndex;
    g_Ui2dConstantBufferIndex %= nn::ui2d::ConstantBufferCount;

    const int commandBufferIndex = 0;

    // テクスチャ取得と vsync 待ち
    g_GfxFramework.AcquireTexture(commandBufferIndex);
    g_GfxFramework.WaitDisplaySync(commandBufferIndex, nn::TimeSpan::FromSeconds(2));

    // 描画処理
    g_GfxFramework.BeginFrame(commandBufferIndex);
    {
        nn::gfx::CommandBuffer* pCommandBuffer = g_GfxFramework.GetRootCommandBuffer(commandBufferIndex);
        nn::gfx::ColorTargetView* pTarget = g_GfxFramework.GetColorTargetView();
        nn::gfx::DepthStencilView* pDepthStencilView = g_GfxFramework.GetDepthStencilView();

        // バッファのクリア
        pCommandBuffer->ClearColor(pTarget, 0.0f, 0.0f, 0.5f, 1.0f, NULL);
        pCommandBuffer->ClearDepthStencil(
            pDepthStencilView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL);

        pCommandBuffer->FlushMemory(nn::gfx::GpuAccess_ColorBuffer | nn::gfx::GpuAccess_Texture);
        g_offsetToRenderTargetTextureMemoryPoolDynamic = g_offsetToRenderTargetTextureMemoryPoolHead + (8 * 1024 * 1024);

        // キャプチャテクスチャの更新
        g_pLayout->DrawCaptureTexture(g_GfxFramework.GetDevice(), g_DrawInfo, *pCommandBuffer);

        {
            // レンダーステートの設定
            pCommandBuffer->SetRenderTargets(1, &pTarget, pDepthStencilView);
            pCommandBuffer->SetViewportScissorState(g_GfxFramework.GetViewportScissorState());
            pCommandBuffer->SetBlendState(g_GfxFramework.GetBlendState(
                nns::gfx::GraphicsFramework::BlendStateType_Alpha));
            pCommandBuffer->SetDepthStencilState(g_GfxFramework.GetDepthStencilState(
                nns::gfx::GraphicsFramework::DepthStencilStateType_Disabled));
            pCommandBuffer->SetRasterizerState(g_GfxFramework.GetRasterizerState(
                nns::gfx::GraphicsFramework::RasterizerStateType_FillSolid_CullNone));
        }

        g_pLayout->Draw(g_DrawInfo, *pCommandBuffer);

        if(isCapture)
        {
            g_ColorTargetViewCapture.PushCaptureCommand(pCommandBuffer, g_GfxFramework.GetColorTargetView());
        }
    }
    g_GfxFramework.EndFrame(commandBufferIndex);
    // コマンドの実行
    // 結果の表示
    g_GfxFramework.ExecuteCommand(commandBufferIndex);
    g_GfxFramework.QueuePresentTexture(1);
    g_GfxFramework.WaitGpuSync(commandBufferIndex, nn::TimeSpan::FromSeconds(2));

    // キャプチャの保存
    if(g_ColorTargetViewCapture.IsCapturing())
    {
        g_ColorTargetViewCapture.FetchCaptureResult();
        const uint8_t* imgData = g_ColorTargetViewCapture.GetCapturedImageData();
        size_t imgDataSize = g_ColorTargetViewCapture.GetCapturedImageDataSize();
        nnt::graphics::PngIhdr ihdr;
        ihdr.SetDefault();
        ihdr.bitDepth = 8;
        ihdr.channels = 4;
        ihdr.colorType = nnt::graphics::PngColorType_RgbAlpha;
        ihdr.width = CaptureImageWidth;
        ihdr.height = CaptureImageHeight;
        nnt::graphics::PngIO::Write(
            nnt::graphics::Path("%s%s/%s_%d.png", OutputPath.c_str(), pDir, pCaptureDataName, frame).GetString(),
            imgData,
            imgDataSize,
            &ihdr,
            &g_TestGraphicsAllocatorFunction);
    }

    return false;
}

//------------------------------------------------------------------------------
// アプリケーションの初期化処理
//------------------------------------------------------------------------------
void InitializeApp_(int argc, char** argv)
{
    // リソースフォルダパスを初期化します。
    {
        // 引き数で指定されていれば、バイナリバージョン毎のサブディレクトリを取得します。
        const char* pBinVerString = "CURRENT";
        for (int i = 0; i < argc; i++)
        {
            if (strstr(argv[i], BinVerOptionName) != nullptr)
            {
                pBinVerString = argv[i] + strlen(BinVerOptionName);
                break;
            }
        }

        std::string binSubDir(pBinVerString);

        binSubDir += "/";

        ContentBasePath = ContentBasePathRoot;
        ContentBasePath += binSubDir;

        LayoutArchivePath = ContentBasePath + LayoutArchiveSubPath;
        FontDirectoryPath = ContentBasePath;
        DataSetPath = ContentBasePath + DataSetSubPath;
        OutputPath = "Outputs:/Outputs/" + binSubDir;
    }

    // アプリケーションで使用するヒープ領域を初期化
    g_pMemoryHeap.Reset(new uint8_t[ApplicationHeapSize]);
    g_ApplicationHeapHandle.Initialize(g_pMemoryHeap.Get(), ApplicationHeapSize);

    // ファイルシステムの初期化(gfx の初期化より前に実行します)
    InitializeFileSystem();

    // グラフィックスフレームワークの初期化
    nns::gfx::GraphicsFramework::InitializeGraphicsSystem(1024 * 1024 * 32);
    nns::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    fwInfo.SetDefault();
    fwInfo.SetDisplayWidth(CaptureImageWidth);
    fwInfo.SetDisplayHeight(CaptureImageHeight);
    fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, 32 * 1024 * 1024);
    fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, 48 * 1024 * 1024);
    fwInfo.SetColorBufferFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);
    g_GfxFramework.Initialize(fwInfo);

    // RenderTarget 用の領域をフレームワークから確保する
    {
        nn::gfx::MemoryPool::InfoType memorypoolInfo;

        memorypoolInfo.SetDefault();
        memorypoolInfo.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuInvisible | nn::gfx::MemoryPoolProperty_GpuCached | nn::gfx::MemoryPoolProperty_Compressible);

        size_t alignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(g_GfxFramework.GetDevice(), memorypoolInfo);

        const nns::gfx::GraphicsFramework::MemoryPoolType memoryPoolType = nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget;
        g_offsetToRenderTargetTextureMemoryPoolHead = g_GfxFramework.AllocatePoolMemory(memoryPoolType, 16 * 1024 * 1024, alignment);
        g_offsetToRenderTargetTextureMemoryPoolStatic = g_offsetToRenderTargetTextureMemoryPoolHead;
        g_offsetToRenderTargetTextureMemoryPoolDynamic = g_offsetToRenderTargetTextureMemoryPoolHead + (8 * 1024 * 1024);
    }
}

//------------------------------------------------------------------------------
// アプリケーションの終了処理
//------------------------------------------------------------------------------
void FinalizeApp_()
{
    nn::fs::Unmount("Outputs");
    // ファイルシステム終了処理。
    nn::fs::Unmount("Contents");

    // RenderTarget 用の領域をフレームワークに返却する
    if (g_offsetToRenderTargetTextureMemoryPoolHead != NULL)
    {
        const nns::gfx::GraphicsFramework::MemoryPoolType memoryPoolType = nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget;
        g_GfxFramework.FreePoolMemory(memoryPoolType, g_offsetToRenderTargetTextureMemoryPoolHead);
        g_offsetToRenderTargetTextureMemoryPoolHead = NULL;
        g_offsetToRenderTargetTextureMemoryPoolStatic = NULL;
        g_offsetToRenderTargetTextureMemoryPoolDynamic = NULL;
    }

    // グラフィックスフレームワーク 終了処理
    g_GfxFramework.FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, g_offsetToUi2dDrawMemoryPool);
    g_GfxFramework.Finalize();

    // ヒープメモリーの終了処理
    g_ApplicationHeapHandle.Finalize();
    delete[] reinterpret_cast<uint8_t*>(g_pMemoryHeap.Get());
}

void ChangeTextBoxCallback(nn::ui2d::Layout& layout, nn::ui2d::BuildResultInformation& buildResultInformation)
{
    nn::ui2d::Pane* pPane = layout.GetRootPane()->FindPaneByName("textBox_tp");
    if (pPane != NULL)
    {
        nn::ui2d::TextBox* pTextBox = static_cast<nn::ui2d::TextBox*>(pPane);
        pTextBox->AllocateStringBuffer(g_GfxFramework.GetDevice(), 128);
        pTextBox->SetString(reinterpret_cast<const uint16_t*>(NN_CHAR16LITERAL("Change text from\n runtime code!")));

        // 表示する文字数分フォント用のコンスタントバッファの領域を拡張しておく。
        buildResultInformation.requiredFontConstantBufferSize += sizeof(nn::font::detail::VertexShaderCharAttribute) * 128;
    }
}

//------------------------------------------------------------------------------
//  動的にレイアウトの文字列を差し替えるテスト
//------------------------------------------------------------------------------
TEST(RuntimeTests, ChangeLayoutTextRuntime)
{
    // フォントを読み込むためにデータセットのデータを使用しています。
    g_Dataset.Initialize(DataSetPath.c_str());
    g_Dataset.ReadConfig();

    // フォルダをクリーンします。
    const char* pTestName = "ChangeLayoutTextRuntime";
    nn::fs::CleanDirectoryRecursively(nnt::graphics::Path("%s%s/", OutputPath.c_str(), pTestName).GetString());

    //-----------------------
    // テスト用のループ
    //-----------------------
    while (g_Dataset.NextData())
    {
        if (strcmp(g_Dataset.GetDataName(), "Simple") == 0)
        {
            NN_LOG("Start the test to change text on runtime.\n");

            // ui2d ランタイムの初期化処理
            InitializeUi2dRuntime(g_Dataset.GetLayoutName(), false, false, ChangeTextBoxCallback);

            // メインループ
            MainLoop(0, true, pTestName, "RuntimeChangeText");

            FinalizeUi2dRuntime();
            break;
        }
    }

    NN_LOG("Complete test\n");

    g_Dataset.Finalize();

    NN_ASSERT(g_AllocationCount == 0);
    SUCCEED();
}

//------------------------------------------------------------------------------
//  データ再生テスト
//------------------------------------------------------------------------------
TEST(RenderAndCapture, AllDataCopyInstance)
{
    nn::gfx::Device*    pDevice = g_GfxFramework.GetDevice();

    //-----------------------
    // テスト用のループ
    //-----------------------
    g_Dataset.Initialize(DataSetPath.c_str());
    g_Dataset.ReadConfig();

    NN_LOG("Start dataset.\n");

    // フォルダをクリーンします。
    const char* pTestName = "AllDataCopyInstance";
    nn::fs::CleanDirectoryRecursively(nnt::graphics::Path("%s%s/", OutputPath.c_str(), pTestName).GetString());

    while (g_Dataset.NextData())
    {
        if (g_Dataset.IsAnimationData())
        {
            g_Dataset.NextKeyFrame();
        }

        bool perspectiveProj = strcmp(g_Dataset.GetDataName(), "PerspectiveProjectionMapping") == 0;

        // ui2d ランタイムの初期化処理。
        InitializeUi2dRuntime(g_Dataset.GetLayoutName(), g_Dataset.IsAnimationData(), perspectiveProj, NULL);

        nn::ui2d::Layout* pCopiedLayout = new(AllocateFromApplicationHeap(sizeof(nn::ui2d::Layout))) nn::ui2d::Layout(pDevice, *g_pLayout, NULL, "CopiedLayout");

        // コピーしたレイアウトに対してアニメーションの作成と再生。
        if(g_Dataset.IsAnimationData())
        {
            nn::ui2d::GroupAnimator* pAnimator = pCopiedLayout->CreateGroupAnimator(pDevice, g_Dataset.GetAnimatorTag());
            pAnimator->PlayAuto(1.0f);
        }

        // MainLoop の中では g_pLayout が更新されるので、g_pLayout をコピーしたレイアウトのポインタに差し替える。
        nn::ui2d::Layout*   pOriginalLayout = g_pLayout;
        g_pLayout = pCopiedLayout;

        // メインループ
        NN_LOG("Start demo.\n");
        for (int frame = 0; ; frame++)
        {
            if (g_Dataset.IsAnimationData() && frame > g_Dataset.GetKeyFrame() && !g_Dataset.NextKeyFrame())
            {
                break;
            }
            bool isCapture = !g_Dataset.IsAnimationData() || (frame == g_Dataset.GetKeyFrame());
            bool needToBreak = MainLoop(frame, isCapture, pTestName, g_Dataset.GetDataName());

            if (needToBreak || !g_Dataset.IsAnimationData())
            {
                break;
            }
        }

        pCopiedLayout->Finalize(pDevice);
        DestructAndFree<nn::ui2d::Layout>(pCopiedLayout);

        // オリジナルのレイアウトの終了処理を行うため g_pLayout のポインタを元に戻す。
        g_pLayout = pOriginalLayout;

        FinalizeUi2dRuntime();
    }

    NN_LOG("Complete dataset\n");
    g_Dataset.Finalize();

    NN_ASSERT(g_AllocationCount == 0);
    SUCCEED();
}


//------------------------------------------------------------------------------
//  データ再生テスト
//------------------------------------------------------------------------------
TEST(RenderAndCapture, AllData)
{
    //-----------------------
    // テスト用のループ
    //-----------------------
    g_Dataset.Initialize(DataSetPath.c_str());
    g_Dataset.ReadConfig();

    NN_LOG("Start dataset.\n");


    // フォルダをクリーンします。
    const char* pTestName = "AllData";
    nn::fs::CleanDirectoryRecursively(nnt::graphics::Path("%s%s/", OutputPath.c_str(), pTestName).GetString());

    while (g_Dataset.NextData())
    {
        if (g_Dataset.IsAnimationData())
        {
            g_Dataset.NextKeyFrame();
        }

        bool perspectiveProj = strcmp(g_Dataset.GetDataName(), "PerspectiveProjectionMapping") == 0;

        // ui2d ランタイムの初期化処理
        InitializeUi2dRuntime(g_Dataset.GetLayoutName(), g_Dataset.IsAnimationData(), perspectiveProj, NULL);

        // メインループ
        NN_LOG("Start demo.\n");
        for (int frame = 0; ; frame++)
        {
            if (g_Dataset.IsAnimationData() && frame > g_Dataset.GetKeyFrame() && !g_Dataset.NextKeyFrame())
            {
                break;
            }
            bool isCapture = !g_Dataset.IsAnimationData() || (frame == g_Dataset.GetKeyFrame());
            bool needToBreak = MainLoop(frame, isCapture, pTestName, g_Dataset.GetDataName());

            if (needToBreak || !g_Dataset.IsAnimationData())
            {
                break;
            }
        }

        FinalizeUi2dRuntime();
    }

    NN_LOG("Complete dataset\n");
    g_Dataset.Finalize();

    NN_ASSERT(g_AllocationCount == 0);
    SUCCEED();
}
// -------------------------------------------------------------------------- -
//  Main 関数
//---------------------------------------------------------------------------
extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    // 初期化
    InitializeApp_(argc, argv);

    InitializeFrameCapture();

    // すべてのテストを実行する。
    int result = RUN_ALL_TESTS();

    FinalizeFrameCapture();

    // 終了処理
    FinalizeApp_();

    ::nnt::Exit(result);
}


