﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testUi2d_TextureCapture.h"
#include <nns/gfx/gfx_GraphicsFramework.h>

#include <cstdlib>
#include <cstring>
#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

namespace nnt{ namespace ui2d{

    TextureCapture::TextureCapture()
        : m_IsInitialized(false)
        , m_IsCapturing(false)
        , m_pAllocateFunction(nullptr)
        , m_ScreenWidth(0)
        , m_ScreenHeight(0)
        , m_pDevice(nullptr)
        , m_CapturedImageBuffer(nullptr)
        , m_CapturedImageBufferSize(0)
        , m_BufferMemoryPoolOffset(0)
    {
    }

    TextureCapture::~TextureCapture()
    {
    }

    //-------------------------------------------

    void* TextureCapture::AllocateMemory(size_t size, int alignment)
    {
        NN_SDK_ASSERT_NOT_NULL(m_pAllocateFunction);
        return m_pAllocateFunction->Allocate(size, alignment);
    }
    void TextureCapture::DeallocateMemory(void* ptr)
    {
        NN_SDK_ASSERT_NOT_NULL(m_pAllocateFunction);
        m_pAllocateFunction->Free(ptr);
    }

    //-------------------------------------------

    void TextureCapture::Initialize(
        nn::gfx::Device* pDevice,
        nns::gfx::GraphicsFramework& framework,
        nnt::graphics::AllocatorFunction* pAllocatorFunction,
        int screenWidth,
        int screenHeight)
    {
        const nns::gfx::GraphicsFramework::MemoryPoolType memoryPoolType = nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer;
        nn::gfx::MemoryPool*    pMemoryPool = framework.GetMemoryPool(memoryPoolType);
        m_ScreenWidth  = screenWidth;
        m_ScreenHeight = screenHeight;
        m_pDevice      = pDevice;
        m_pAllocateFunction = pAllocatorFunction;

        size_t imageBufferSize = screenWidth * screenHeight * 4;
        {
            nn::gfx::Buffer::InfoType info;
            info.SetDefault();
            info.SetSize(imageBufferSize);
            info.SetGpuAccessFlags(nn::gfx::GpuAccess_ColorBuffer);

            size_t alignment = nn::gfx::Buffer::GetBufferAlignment(pDevice, info);

            m_BufferMemoryPoolOffset = framework.AllocatePoolMemory(memoryPoolType, imageBufferSize, alignment);

            size_t size = info.GetSize();
            m_CaptureBuffer.Initialize(pDevice, info, pMemoryPool, m_BufferMemoryPoolOffset, size);
        }

        //size_t imageBufferSize = screenWidth * screenHeight * 4;
        m_CapturedImageBuffer = static_cast<uint8_t*>(AllocateMemory(imageBufferSize, NN_ALIGNOF(std::max_align_t)));
        m_CapturedImageBufferSize = imageBufferSize;

        m_IsCapturing = false;

        m_IsInitialized = true;
    }

    void TextureCapture::Finalize(nns::gfx::GraphicsFramework& framework)
    {
        if(!m_IsInitialized)
        {
            return;
        }
        framework.FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, m_BufferMemoryPoolOffset);
        m_CaptureBuffer.Finalize(m_pDevice);

        DeallocateMemory(m_CapturedImageBuffer);
        m_CapturedImageBuffer = NULL;
        m_CapturedImageBufferSize = 0;

        m_IsInitialized = false;
    }

    //-------------------------------------------

    bool TextureCapture::IsCapturing() const
    {
        return m_IsCapturing;
    }

    //-------------------------------------------

    void TextureCapture::PushCaptureCommand(nn::gfx::CommandBuffer* pCommandBuffer, nn::gfx::Texture* pSrcTexture)
    {
        if(m_IsCapturing)
        {
            NN_LOG("too many PushCaputureCommand().\n");
            return;
        }
        nn::gfx::TextureCopyRegion region;
        region.SetDefault();
        region.SetWidth(m_ScreenWidth);
        region.SetHeight(m_ScreenHeight);
        pCommandBuffer->CopyImageToBuffer(&m_CaptureBuffer, 0, pSrcTexture, region);
        m_IsCapturing = true;
    }

    //-------------------------------------------

    void TextureCapture::FetchCaptureResult()
    {
        if(!m_IsCapturing)
        {
            NN_LOG("too many FetchCaptureResult()\n");
            return;
        }
        const uint8_t* pBuffer = static_cast<uint8_t*>(m_CaptureBuffer.Map());
        m_CaptureBuffer.InvalidateMappedRange(0, 4 * m_ScreenWidth * m_ScreenHeight);

        size_t stride = 4 * m_ScreenWidth;

        int ySrc = 0;
        int yDst = m_ScreenHeight - 1;
        for(; ySrc < m_ScreenHeight; ySrc++, yDst--)
        {
            const uint8_t* pSrc = pBuffer + stride * ySrc;
            uint8_t* pDst = m_CapturedImageBuffer + stride * yDst;
            for(int x = 0; x < m_ScreenWidth; x++)
            {
                *pDst++ = *pSrc++;
                *pDst++ = *pSrc++;
                *pDst++ = *pSrc++;
                *pDst++ = 0xFF;
                pSrc++;
            }
        }

        m_CaptureBuffer.Unmap();
        m_IsCapturing = false;
    }

    //-------------------------------------------

    const uint8_t* TextureCapture::GetCapturedImageData() const
    {
        return this->m_CapturedImageBuffer;
    }
    size_t TextureCapture::GetCapturedImageDataSize() const
    {
        return this->m_CapturedImageBufferSize;
    }
}}

