﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testUi2d_ColorTargetViewCapture.h"
#include <nns/gfx/gfx_GraphicsFramework.h>

#include <nn/nn_Log.h>

namespace nnt{ namespace ui2d{

    ColorTargetViewCapture::ColorTargetViewCapture()
        : m_IsInitialized(false)
        , m_TextureCapture()
        , m_pDevice(nullptr)
        , m_Texture()
        , m_BufferMemoryPoolOffset(0)
    {
    }

    ColorTargetViewCapture::~ColorTargetViewCapture()
    {
    }

    //-------------------------------------------

    void ColorTargetViewCapture::Initialize(
        nn::gfx::Device* pDevice,
        nns::gfx::GraphicsFramework&    framework,
        nnt::graphics::AllocatorFunction* pAllocatorFunction,
        int screenWidth,
        int screenHeight)
    {
        m_TextureCapture.Initialize(pDevice, framework, pAllocatorFunction, screenWidth, screenHeight);
        m_pDevice = pDevice;

        {
            nn::gfx::TextureInfo info;
            info.SetDefault();
            info.SetWidth( screenWidth );
            info.SetHeight( screenHeight );
            info.SetImageFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm );

            const nns::gfx::GraphicsFramework::MemoryPoolType memoryPoolType = nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget;
            nn::gfx::MemoryPool*    pMemoryPool = framework.GetMemoryPool(memoryPoolType);
            size_t size = m_Texture.CalculateMipDataSize( pDevice, info );

            m_BufferMemoryPoolOffset = framework.AllocatePoolMemory(memoryPoolType, size, m_Texture.CalculateMipDataAlignment( pDevice, info ));
            m_Texture.Initialize( pDevice, info, pMemoryPool, m_BufferMemoryPoolOffset, size );
        }

        m_IsInitialized = true;
    }

    void ColorTargetViewCapture::Finalize(nns::gfx::GraphicsFramework& framework)
    {
        if(!m_IsInitialized)
        {
            return;
        }

        framework.FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, m_BufferMemoryPoolOffset);
        m_Texture.Finalize(m_pDevice);

        m_TextureCapture.Finalize(framework);
        m_IsInitialized = false;
    }

    //-------------------------------------------

    bool ColorTargetViewCapture::IsCapturing() const
    {
        return m_TextureCapture.IsCapturing();
    }

    //-------------------------------------------

    void ColorTargetViewCapture::PushCaptureCommand(nn::gfx::CommandBuffer* pCommandBuffer, nn::gfx::ColorTargetView* pSrcView)
    {
        if(m_TextureCapture.IsCapturing())
        {
            NN_LOG("too many PushCaputureCommand().\n");
            return;
        }
        pCommandBuffer->Resolve(&m_Texture, 0, 0, pSrcView, nullptr);
        m_TextureCapture.PushCaptureCommand(pCommandBuffer, &m_Texture);
    }

    //-------------------------------------------

    void ColorTargetViewCapture::FetchCaptureResult()
    {
        m_TextureCapture.FetchCaptureResult();
    }

    //-------------------------------------------

    const uint8_t* ColorTargetViewCapture::GetCapturedImageData() const
    {
        return m_TextureCapture.GetCapturedImageData();
    }
    size_t ColorTargetViewCapture::GetCapturedImageDataSize() const
    {
        return m_TextureCapture.GetCapturedImageDataSize();
    }
}}

