﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/util/util_BytePtr.h>
#include <nn/mem.h>

// ----------------------------------------------------------------------------
// アロケータ
// ----------------------------------------------------------------------------
class SimpleAllocator
{
public:
    SimpleAllocator()
        : m_OccupiedSize(0)
        , m_MaxSize(0)
        , m_pTop(NULL)
        , m_pBottom(NULL)
        , m_pMemoryHeap(NULL)
    {
    }

    void Initialize(size_t heapSize)
    {
        m_pMemoryHeap.Reset(new uint8_t[heapSize]);
        m_ApplicationHeapHandle.Initialize(m_pMemoryHeap.Get(), heapSize);
    }

    void Finalize()
    {
        m_ApplicationHeapHandle.Finalize();
        delete[] static_cast<uint8_t*>(m_pMemoryHeap.Get());
    }

    void* AllocateMemoryWithoutLogging(size_t size, size_t alignment)
    {
        return m_ApplicationHeapHandle.Allocate(size, static_cast<int>(alignment));
    }

    void FreeMemoryWithoutLogging(void* pointer)
    {
        m_ApplicationHeapHandle.Free(pointer);
    }

    void* AllocateMemory(size_t size, size_t alignment);

    void FreeMemory(void* pointer);

    // 確保中の総メモリサイズを取得します
    int64_t GetOccupiedSize() const;

    // メモリの最大使用量を取得します
    int64_t GetMaxSize() const;

    // メモリの最大使用量をリセットします
    void ResetMaxSize();

    // 確保中のメモリを列挙して表示します
    void PrintOccupiedMemory() const;

private:
    // メモリのアドレスとサイズをリストで管理
    struct MemoryList
    {
        MemoryList* pNext;
        void* pAddress;
        size_t size;
    };

    int64_t m_OccupiedSize;
    int64_t m_MaxSize;
    MemoryList* m_pTop;
    MemoryList* m_pBottom;

    nn::util::BytePtr m_pMemoryHeap;
    nn::mem::StandardAllocator m_ApplicationHeapHandle;
};
