﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testUi2d_Dataset.h"

#include <cstring>
#include <malloc.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Macro.h>
#include <nn/fs.h>

namespace nnt{ namespace ui2d{

    Dataset::Dataset()
        : m_FileBuffer(NULL),
        m_PosFile(NULL),
        m_EndFile(NULL),
        m_FontNameBuffer(NULL),
        m_FontNameBufferSize(0),
        m_FcpxNameBuffer(NULL),
        m_FcpxNameBufferSize(0),
        m_Config(),
        m_Data()
    {
    }

    void Dataset::Initialize(const char* filename)
    {
        nn::Result result;
        nn::fs::FileHandle file = {};
        std::int64_t fileSize = 0;

        result = nn::fs::OpenFile(&file, filename, nn::fs::OpenMode_Read);
        NN_ASSERT(result.IsSuccess());
        result = nn::fs::GetFileSize(&fileSize, file);
        NN_ASSERT(result.IsSuccess());
        m_FileBuffer = static_cast<char*>(malloc(static_cast<size_t>(fileSize) + 1));
        NN_ASSERT(m_FileBuffer != NULL);
        result = nn::fs::ReadFile(file, 0, m_FileBuffer, static_cast<size_t>(fileSize));
        NN_ASSERT(result.IsSuccess());
        nn::fs::CloseFile(file);

        m_PosFile = m_FileBuffer;
        m_EndFile = m_FileBuffer + fileSize;
        *m_EndFile = 0;

        m_FontNameBuffer = static_cast<char*>(malloc(static_cast<size_t>(fileSize) + 1));
        m_FontNameBufferSize = static_cast<size_t>(fileSize) + 1;
        m_FcpxNameBuffer = static_cast<char*>(malloc(static_cast<size_t>(fileSize) + 1));
        m_FcpxNameBufferSize = static_cast<size_t>(fileSize) + 1;
    }
    void Dataset::Finalize()
    {
        free(m_FileBuffer);
        m_FileBuffer = NULL;

        free(m_FontNameBuffer);
        m_FontNameBuffer = NULL;
        m_FontNameBufferSize = 0;
        m_FcpxNameBuffer = NULL;
        m_FcpxNameBufferSize = 0;
    }

    Dataset::~Dataset()
    {
        this->Finalize();
    }


    //---------------------------------------------------------------------------

    bool Dataset::IsAnimationData() const
    {
        return m_Data.beginAnimatorTag != m_Data.endAnimatorTag;
    }
    const char* Dataset::GetDataName() const
    {
        return m_Data.beginDataName;
    }
    const char* Dataset::GetLayoutName() const
    {
        return m_Data.beginLayoutName;
    }
    const char* Dataset::GetAnimatorTag() const
    {
        return m_Data.beginAnimatorTag;
    }
    int Dataset::GetKeyFrame() const
    {
        return m_Data.nextKeyFrame;
    }
    int Dataset::GetDrawCount() const
    {
        return m_Data.drawCount;
    }
    RunMode Dataset::GetRunMode() const
    {
        return m_Config.runMode;
    }
    bool Dataset::GetIsAssertionEnabled() const
    {
        return m_Config.isAssertionEnabled;
    }

    //---------------------------------------------------------------------------

    static bool IsWhiteSpace(char c)
    {
        return
            c == ' ' || c == '\t';
    }
    static bool IsCommentSymbol(char c)
    {
        return c == '#';
    }
    // 現在の位置から始まる連続した空白文字を読み飛ばす。
    // post
    //  **cur は最初に現れた空白文字でない文字、または (*cur == end)
    static void SkipWhiteSpace(char** cur, char* end)
    {
        char* p = *cur;
        for(; p < end; p++)
        {
            if(!IsWhiteSpace(*p))
            {
                break;
            }
        }
        *cur = p;
    }
    // 現在の位置から行末までを読み込む。
    // post
    //  *outLineBeg は呼び出した時点での *cur の値が入る
    //  *outLineEnd は改行文字の直前の位置もしくは end の値が入る
    //  *cur は改行文字の次の位置を指す
    static void GetLine(char** outLineBeg, char**outLineEnd, char** cur, char* end)
    {
        char* p = *cur;
        char* lineBeg = p;
        char* lineEnd = end;
        for(; p < end; p++)
        {
            if(*p == '\n')
            {
                lineEnd = p;
                p++;
                break;
            }
            if(*p == '\r')
            {
                lineEnd = p;
                p++;
                if(p < end && *p == '\n')
                {
                    p++;
                }
                break;
            }
        }

        *outLineBeg = lineBeg;
        *outLineEnd = lineEnd;
        *cur = p;
    }
    // 次の空でない行を読み取る。
    // 行頭の空白文字は読み飛ばされる。
    static void GetNextNonEmptyLine(char** outLineBeg, char** outLineEnd, char** cur, char* end)
    {
        for(;;)
        {
            GetLine(outLineBeg, outLineEnd, cur, end);
            if(*outLineBeg == end)
            {
                break;
            }
            SkipWhiteSpace(outLineBeg, *outLineEnd);
            // 行に空白文字以外の文字がなければ無視
            if(*outLineBeg == *outLineEnd)
            {
                continue;
            }
            // コメント行なら無視
            if(IsCommentSymbol(**outLineBeg))
            {
                continue;
            }
            else
            {
                break;
            }
        }
    }


    //---------------------------------------------------------------------------

    static bool ParseConfig(Dataset::Config* pOutValue, char** cur, char* end)
    {
        NN_ASSERT_NOT_NULL(cur);
        NN_ASSERT_NOT_NULL(*cur);
        NN_ASSERT_NOT_NULL(end);
        if(*cur == end)
        {
            return false;
        }

        Dataset::Config conf = {};
        for(;;)
        {
            char* lineBeg;
            char* lineEnd;
            GetNextNonEmptyLine(&lineBeg, &lineEnd, cur, end);
            // 末尾まで行ったら終了
            if(lineBeg == end)
            {
                break;
            }
            // 次のデータが開始したら終了
            if(*lineBeg == '[')
            {
                *cur = lineBeg;
                break;
            }

            char* p = lineBeg;

            // 名前を取得
            char* nameBeg = p;
            char* nameEnd = p;
            for(; p < lineEnd; ++p)
            {
                char c = *p;
                if((c < 'a' || c > 'z') && (c < 'A' || c > 'Z'))
                {
                    nameEnd = p;
                    break;
                }
            }
            SkipWhiteSpace(&p, lineEnd);
            NN_ASSERT(*p == '=');
            ++p; // skip '='
            SkipWhiteSpace(&p, lineEnd);
            char* valBeg = p;
            char* valEnd = lineEnd;

            *nameEnd = '\0';
            *valEnd = '\0';
            if(strcmp(nameBeg, "CreateReference") == 0)
            {
                int value = atoi(valBeg);
                if(value != 0)
                {
                    conf.runMode = RunMode_CreateReferenceImage;
                }
                else
                {
                    conf.runMode = RunMode_CompareRenderedImage;
                }
            }
            else if(strcmp(nameBeg, "EnableAssertion") == 0)
            {
                int value = atoi(valBeg);
                if(value != 0)
                {
                    conf.isAssertionEnabled = true;
                }
                else
                {
                    conf.isAssertionEnabled = false;
                }
            }
            else
            {
                NN_ASSERT(false, "Unknown config entry '%s = %s'", nameBeg, valBeg);
            }
        }

        if(pOutValue)
        {
            *pOutValue = conf;
        }
        return true;
    }

    //---------------------------------------------------------------------------

    static bool ParseDataName(Dataset::Data* pOutValue, char** cur, char* end)
    {
        char* lineBeg;
        char* lineEnd;
        GetNextNonEmptyLine(&lineBeg, &lineEnd, cur, end);

        NN_ASSERT(*lineBeg == '[');
        ++lineBeg; // skip '['
        pOutValue->beginDataName = lineBeg;
        for(; lineBeg < lineEnd; ++lineBeg)
        {
            if(*lineBeg == ']')
            {
                pOutValue->endDataName = lineBeg;
                break;
            }
        }
        NN_ASSERT(lineBeg < lineEnd);
        ++lineBeg; // skip ']'
        SkipWhiteSpace(&lineBeg, lineEnd);
        NN_ASSERT(lineBeg == lineEnd);

        return true;
    }

    static bool ParseData(Dataset::Data* pOutValue, char** cur, char* end, char* fontBuffer, size_t fontBufferSize, char* fcpxBuffer, size_t fcpxBufferSize)
    {
        NN_ASSERT_NOT_NULL(cur);
        NN_ASSERT_NOT_NULL(*cur);
        NN_ASSERT_NOT_NULL(end);
        if(*cur == end)
        {
            return false;
        }

        Dataset::Data data = {};
        char* fontPos = fontBuffer;
        char* fontEnd = fontBuffer + fontBufferSize;
        char* fcpxPos = fcpxBuffer;
        char* fcpxEnd = fcpxBuffer + fcpxBufferSize;

        data.beginData = *cur;
        ParseDataName(&data, cur, end);
        *data.endDataName = '\0';
        data.drawCount = 1;

        for(;;)
        {
            char* lineBeg;
            char* lineEnd;
            GetNextNonEmptyLine(&lineBeg, &lineEnd, cur, end);
            // 末尾まで行ったら終了
            if(lineBeg == end)
            {
                break;
            }
            // 次のデータが開始したら終了
            if(*lineBeg == '[')
            {
                *cur = lineBeg;
                break;
            }

            char* p = lineBeg;

            // 名前を取得
            char* nameBeg = p;
            char* nameEnd = p;
            for(; p < lineEnd; ++p)
            {
                char c = *p;
                if((c < 'a' || c > 'z') && (c < 'A' || c > 'Z'))
                {
                    nameEnd = p;
                    break;
                }
            }
            SkipWhiteSpace(&p, lineEnd);
            NN_ASSERT(*p == '=');
            ++p; // skip '='
            SkipWhiteSpace(&p, lineEnd);
            char* valBeg = p;
            char* valEnd = lineEnd;

            *nameEnd = '\0';
            *valEnd = '\0';
            if(strcmp(nameBeg, "LayoutName") == 0)
            {
                data.beginLayoutName = valBeg;
                data.endLayoutName = valEnd;
            }
            else if(strcmp(nameBeg, "AnimatorTag") == 0)
            {
                data.beginAnimatorTag = valBeg;
                data.endAnimatorTag = valEnd;
            }
            else if(strcmp(nameBeg, "KeyFrames") == 0)
            {
                data.beginKeyFrames = valBeg;
                data.endKeyFrames = valEnd;
            }
            else if (strcmp(nameBeg, "DrawCount") == 0)
            {
                int count = atoi(valBeg);
                if (count > 0)
                {
                    data.drawCount = count;
                }
            }
            else if(strcmp(nameBeg, "Font") == 0)
            {
                if(fontPos != NULL && valBeg < valEnd)
                {
                    for(const char* src = valBeg; fontPos < fontEnd && src <= valEnd; fontPos++,src++)
                    {
                        *fontPos = *src;
                    }
                    NN_ASSERT(fontPos < fontEnd);
                }
            }
            else if(strcmp(nameBeg, "Fcpx") == 0)
            {
                if(fcpxPos != NULL && valBeg < valEnd)
                {
                    for(const char* src = valBeg; fcpxPos < fcpxEnd && src <= valEnd; fcpxPos++,src++)
                    {
                        *fcpxPos = *src;
                    }
                    NN_ASSERT(fcpxPos < fcpxEnd);
                }
            }
            else
            {
                NN_ASSERT(false, "Unknown dataset entry '%s = %s'", nameBeg, valBeg);
            }
        }

        if(fontPos != NULL)
        {
            *fontPos = 0;
        }

        if(fcpxPos != NULL)
        {
            *fcpxPos = 0;
        }

        data.endData = *cur;
        data.fonts = fontBuffer;
        data.fcpxs = fcpxBuffer;
        data.nextKeyFrame = -1;

        if(pOutValue)
        {
            *pOutValue = data;
        }
        return true;
    }// NOLINT(impl/function_size)

    //---------------------------------------------------------------------------

    void Dataset::ReadConfig()
    {
        Config conf = {};
        ParseConfig(&conf, &m_PosFile, m_EndFile);
        m_Config = conf;
    }

    static void OutputFontListToLog(const char* fontname, void*)
    {
        NN_LOG("Font=%s\n",fontname);
    }
    bool Dataset::NextData()
    {
        Data data = {};
        if(!ParseData(&data, &m_PosFile, m_EndFile, m_FontNameBuffer, m_FontNameBufferSize, m_FcpxNameBuffer, m_FcpxNameBufferSize))
        {
            return false;
        }
        m_Data = data;

        {
            NN_LOG("Test case ----------------------------\n");
            NN_LOG("[%s]\n", data.beginDataName);
        }
        if(data.beginLayoutName != data.endLayoutName)
        {
            NN_LOG("LayoutName=%s\n", data.beginLayoutName);
        }
        if(data.beginAnimatorTag != data.endAnimatorTag)
        {
            NN_LOG("AnimatorTag=%s\n", data.beginAnimatorTag);
        }
        if(data.beginKeyFrames != data.endKeyFrames)
        {
            NN_LOG("KeyFrames=%s\n", data.beginKeyFrames);
        }
        ForEachFont(OutputFontListToLog, NULL);

        return true;
    }
    bool Dataset::NextKeyFrame()
    {
        if(m_Data.beginKeyFrames == m_Data.endKeyFrames)
        {
            m_Data.nextKeyFrame = -1;
            return false;
        }

        int key = 0;
        char* p = m_Data.beginKeyFrames;
        for(; p < m_Data.endKeyFrames; ++p)
        {
            char c = *p;
            if(c >= '0' && c <= '9')
            {
                key *= 10;
                key += c - '0';
            }
            else
            {
                break;
            }
        }
        NN_ASSERT(p != m_Data.beginKeyFrames, "KeyFrame must be list of integers.");

        m_Data.nextKeyFrame = key;

        SkipWhiteSpace(&p, m_Data.endKeyFrames);
        if(p == m_Data.endKeyFrames)
        {
            m_Data.beginKeyFrames = m_Data.endKeyFrames;
            return true;
        }
        NN_ASSERT(*p == ',', "KeyFrame list separeter is ','");
        ++p; // skip ','
        SkipWhiteSpace(&p, m_Data.endKeyFrames);
        m_Data.beginKeyFrames = p;
        return true;
    }

    void Dataset::ForEachFont(void (*func)(const char* fontName, void* param), void* param)
    {
        const char* p = m_Data.fonts;
        while(*p != '\0')
        {
            func(p, param);

            // go to next font name
            for(; *p != '\0'; ++p)
            {
            }
            ++p;
        }
    }

    void Dataset::ForEachFcpx(void (*func)(const char* fcpxName, void* param), void* param)
    {
        const char* p = m_Data.fcpxs;
        while(*p != '\0')
        {
            func(p, param);

            // go to next font name
            for(; *p != '\0'; ++p)
            {
            }
            ++p;
        }
    }
}}
