﻿#===========================================================================
# 負荷計測データの自動ダウンロード＆コメント付加ツール
#
# ※このツールの起動には powershell Version 3.0 以上が必要です。
#===========================================================================
param
(
    [switch]$Help = $false
)

# ヘルプ
if ( $Help )
{
    Write-Host @"

    コマンド名
        UpdateProfilingData.ps1

    説明
        TeamCity 上に存在する負荷計測テストの結果をダウンロードして任意のコメントを埋め込みます。
        NX-NXFP2-a64/Release のデータのダウンロードを試みます。
        スクリプトを実行すると GUI が表示され、ダウンロード元の URL と出力先のディレクトリ、
        書き込むコメントを入力することができます。
        「Artifacts root page url」には
        http://devsedci01.ncl.nintendo.co.jp/TeamCity/repository/download/SigloSdk_Nx_Public_Team_TestModuleSdevGfxRelease/891639:id/GraphicsTestOutputs.zip%21/testUi2d_PerformanceProfile
        （http://devsedci01.ncl.nintendo.co.jp/TeamCity/repository/download/SigloSdk_Nx_Public_Team_TestModuleSdevGfxRelease/891639:id/GraphicsTestOutputs.zip%21/testUi2d_PerformanceProfile<削除>/NX-NXFP2-a32/Debug/Simple_CompareResult.html</削除>）
        のような、比較結果 html からターゲットとビルドを削った位置を指定してください。

    書式
        UpdateProfilingData [-Help]

    オプション
        -Help                   ヘルプを表示します。

"@
    exit 0
}

Add-Type -AssemblyName PresentationFramework

# Windows7 等は標準が Version 2.0 の為、3 に満たない場合は起動しないようにする。
if($PSVersionTable.PSVersion.Major -lt 3){
   $message = "Powershellバージョン 3 以上が必要です。" + "現在のバージョンは " + $PSVersionTable.PSVersion.Major + " です。"
   Write-Output $message
   exit
}

#======================================================================================
# 計測データのダウンロード＆ファイル出力処理
#======================================================================================
function DownloadProfilingData($outputRootPath, $testName, $artifactsTopUrl, $target, $build, $username, $password, $comment)
{
    try
    {
		if ($artifactsTopUrl.EndsWith("/"))
		{
			$artifactsTopUrl = $artifactsTopUrl.substring(0, $artifactsTopUrl.Length - 1)
		}

        # ダウンロードするファイル名にタイムスタンプが入っているため固定の名前で読み込めない。
        # 比較結果の html から参照しているデータの日付を取得してファイル名を決定する。
        $wc = New-Object Net.WebClient
        # TeamCity へ接続するための資格情報を設定する。
        $wc.Credentials = New-Object System.Net.NetworkCredential($username, $password)
        $compareResultUrl = "$artifactsTopUrl/$target/$build/${testName}_CompareResult.html";
        # 比較結果の HTML をダウンロード。
        $content = $wc.DownloadString($compareResultUrl)

        # アカウント情報が正しくないと例外にはならずに空の content が返ってくる。
        if ($content.Length -eq 0)
        {
            throw "The teamcity responsed content size is 0.`nPlease check your teamcity account."
        }
            
        $dateRegExPattern = "^'{`"header`":{`"name`":`".*`",`"date`":`"(?<year>[0-9]{4})-(?<month>[0-9]{2})-(?<day>[0-9]{2}) (?<hour>[0-9]{2}):(?<minutes>[0-9]{2}):(?<second>[0-9]{2})"
        $hitCount = 0

        $jsonFileName = ""

        # HTML 内の json データから日付を取得する。
        # 日付を更新したデータが oldJsonText の2番目に入っているため正規表現で引っ張る。
        $content -split "`n" | %{
            $matches = [RegEx]::Matches($_, $dateRegExPattern)
            if ($matches.Length -gt 0)
            {
                $hitCount++
                if ($hitCount -ge 2)
                {
                    $jsonFileName = $testName + "_" + $matches[0].Groups["year"].Value + $matches[0].Groups["month"].Value + $matches[0].Groups["day"].Value + `
                                    "_" + $matches[0].Groups["hour"].Value + $matches[0].Groups["minutes"].Value + $matches[0].Groups["second"].Value + ".json"
                    # ループ脱出
                    return
                }
            }
        }

        if ($jsonFileName.Length -eq 0)
        {
            throw "Invalid CompareResult HTML file contents.`nTo determine json file date data is not found."
        }

        # 計測結果の URL を確定する。
        $capturedDataUrl = "$artifactsTopUrl/$target/$build/rhs/$jsonFileName"
        $outputPath = Join-Path $outputRootPath $target 
        $outputPath = Join-Path $outputPath $build
        $outputPath = Join-Path $outputPath $jsonFileName

        $content = $wc.DownloadString($capturedDataUrl)

        # コメントを追加する。
        if ($content -match '"comment":')
        {
            # コメントタグが存在するデータ
            $content = $content -replace "`"comment`":`"(.*)`",", "`"comment`":`"$comment`","
        }
        else
        {
            # まだコメントタグが存在しないデータ
            $content = $content -replace ",`"cpu_load_average`":", ",`"comment`":`"$comment`",`"cpu_load_average`":"
        }

        # ファイル出力
		# 末尾に改行を出力したくないため WriteAllText で書き出す
		$Utf8N = New-Object System.Text.UTF8Encoding($false)
		[IO.File]::WriteAllText($outputPath, $content, $Utf8N)

		# 更新日時を LastUpdatedDate.txt へ出力する。
		# このファイルを計測データと一緒に更新することで複数人で同時に PR を作成した際にコンフリクトが発生し
		# 意図しないデータが develop にマージされる問題を防ぎます。
		$date = Get-Date

		$lastUpdateFilePath = $MyInvocation.ScriptName
		$lastUpdateFilePath = Split-Path -Parent $lastUpdateFilePath
		$lastUpdateFilePath = Join-Path $lastUpdateFilePath "LastUpdatedDate.txt"

		[IO.File]::WriteAllText($lastUpdateFilePath, $date.ToString("yyyy/MM/dd hh:mm:ss"), $Utf8N)
    }
    catch [System.Net.WebException]
    {
        $statusCode = [int]$_.Exception.Response.StatusCode

        # 状況によって特定の構成のみテストを行わない設定にすることがある。
        # その時はダウンロード元のファイルが見つからないため WebClient で例外になってしまう。
        # ダウンロードをスキップするためにファイルが見つからない 404 の時は例外を握りつぶす。
        if (($statusCode -ne 404))
        {
            Write-Host $error[0]
            throw "System.Net.WebException occurs at downloading json files."
        }
        else
        {
            Write-Host "$compareResultUrl is skipped."
            return ""
        }
    }
    catch [Exception]
    {
        Write-Host $error[0]
        throw "Exception occurs at downloading json files."
    }
    finally
    {
    }

    return $outputPath
}

#======================================================================================
# TeamCity へ接続できるかテストする
#======================================================================================
function TeamCityAcountValidation($username, $password)
{
    $wc = New-Object Net.WebClient
    # TeamCity へ接続するための資格情報を設定する。
    $wc.Credentials = New-Object System.Net.NetworkCredential($username, $password)
    # テスト用にトップページをダウンロード
    $content = $wc.DownloadString("http://devsedci01.ncl.nintendo.co.jp/TeamCity/")

    # アカウント情報が正しくないと例外にはならずに空の content が返ってくる。
    return $content.Length;
}

#======================================================================================
# 設定ファイルのデフォルト XML
#======================================================================================
[xml]$defaultSettingXml = @"
<?xml version="1.0" encoding="utf-8"?>

<prop>
<settings>
    <ci_username></ci_username>
    <ci_password></ci_password>
</settings>
<outputPathHistories>
    <path></path>
</outputPathHistories>
</prop>
"@

#======================================================================================
# メイン処理
#======================================================================================

# 初期設定読み込み
$settingFilePath = $MyInvocation.MyCommand.Path
$settingFilePath = Split-Path -Parent $settingFilePath
$settingFilePath = Join-Path $settingFilePath "settings.xml"

# 設定ファイルが存在しない場合はデフォルトの値を使用する
if (Test-Path $settingFilePath)
{
    $settingFile = [xml](Get-Content $settingFilePath)
}
else
{
    $settingFile = $defaultSettingXml
}

# UI のための XAML
[xml]$loadXaml = @"
<Window
    xmlns="http://schemas.microsoft.com/winfx/2006/xaml/presentation"
    Title=""
    Width="640" Height="400">
    <StackPanel>
    <Label
        Name="outputPathLabel"
        Content="OutputDirectory"
        Height="25"
        VerticalAlignment="Top"
        HorizontalAlignment="Left"
    />
    <StackPanel
        Orientation="Horizontal"
    >
        <ComboBox
            Name="outputPathComboBox"
            IsEditable="True"
            Width="560"
            Height="25"
            Margin="5"
        />
        <Button
            Name="selectDirectoryButton"
            HorizontalAlignment="Right"
            Width="30"
            Height="25"
            Margin="5"
        >
        ...
        </Button>
    </StackPanel>
    <Label
        Name="artifactsUrlLabel"
        Content="Artifacts root page URL"
        Height="25"
        VerticalAlignment="Top"
        HorizontalAlignment="Left"
    />
    <TextBox
        Name="ArtifactsUrlTextBox"
        Text=""
        Height="25"
        Margin="5"
    />
	<GroupBox Header="Output Data Select" HorizontalAlignment="Left" Margin="5" VerticalAlignment="Top" Height="60" Width="560">
		<StackPanel Orientation="Horizontal">
			<CheckBox Content="Simple" Height="16" HorizontalAlignment="Left" Margin="10,10,10,10" Name="chkBoxSimple" VerticalAlignment="Top" IsChecked="true" /> 
			<CheckBox Content="FillrateOptimization" Height="16" HorizontalAlignment="Left" Margin="10,10,10,10" Name="chkBoxFill" VerticalAlignment="Top" IsChecked="true" /> 
		</StackPanel>
	</GroupBox>
    <Label
        Name="CommentLabel"
        Content="comment"
        Height="25"
        VerticalAlignment="Top"
        HorizontalAlignment="Left"
    />
    <TextBox
        Name="CommentTextBox"
        Text=""
        Height="100"
        Margin="5"
    />
    <Button
        Name="decideButton"
        HorizontalAlignment="Center"
        Margin="5"
    >
    DL and Write Comment
    </Button>
    </StackPanel>
</Window>
"@

$reader = New-Object System.Xml.XmlNodeReader $loadXaml
$window = [Windows.Markup.XamlReader]::Load($reader)

# 各種オブジェクトを取得する
$decideButton = $window.FindName("decideButton")
$directorySelectButton = $window.FindName("selectDirectoryButton")
$urlTextBox = $window.FindName("ArtifactsUrlTextBox")
$outputPathComboBox = $window.FindName("outputPathComboBox")
$commentTextBox = $window.FindName("CommentTextBox")
$usernameTextBox = $window.FindName("usernameTextBox")
$passwordBox = $window.FindName("passwordBox")
# 出力判定のためのチェックボックスを収集する
$outputSelection = $(
	$window.FindName("chkBoxSimple"),
	$window.FindName("chkBoxFill"))

$username = $settingFile.prop.settings.ci_username
if ($settingFile.prop.settings.ci_password.Length -gt 0)
{
    $password = ConvertTo-SecureString $settingFile.prop.settings.ci_password
}

# 出力ディレクトリの履歴を UI へ設定する
$outputPathComboBox.ItemsSource = $settingFile.prop.outputPathHistories.path
# 履歴が残っていたら最新の設定をデフォルト値として設定しておく
if ($settingFile.prop.outputPathHistories.path.Length -gt 0)
{
    $outputPathComboBox.Text = $settingFile.prop.outputPathHistories.path[0]
}

# 出力パスの初期値設定
$outputPath = $MyInvocation.MyCommand.Path
$outputPath = Split-Path -Parent $outputPath

#======================================================================================
# 出力ディレクトリの選択処理。
# WPF だとディレクトリ選択ダイアログが簡単に使用できないためここだけ WinForms で実装。
#======================================================================================
[void][System.Reflection.Assembly]::LoadWithPartialName("System.Windows.Forms")
$directorySelectButton.Add_Click({
    $selector = New-Object System.Windows.Forms.FolderBrowserDialog
    $selector.Description = "Select output directory"
    $result = $selector.ShowDialog();

    if ($result -eq [System.Windows.Forms.DialogResult]::OK)
    {
        $outputPathComboBox.Text = $selector.SelectedPath
    }
})

#======================================================================================
# メイン処理
# Artifacts ルートパスで指定された URL から計測データをダウンロードして
# 負荷計測ディレクトリの構成に従ってファイルを書き込む
#======================================================================================
$decideButton.Add_Click({
    # 出力ディレクトリ設定が空の場合はスクリプトの存在するパスに出力する
    if ($outputPathComboBox.Text -gt 0)
    {
        $outputPath = $outputPathComboBox.Text
    }

    $artifactsUrl = $urlTextBox.Text
    $inputComment = $commentTextBox.Text

    $result = 0

    # 現在設定されているアカウントで TeamCity に接続できるかチェックする
    if ($username.Length -gt 0)
    {
        $result = TeamCityAcountValidation $username $password
    }

    # 接続できない場合は有効なアカウントが入力されるまで下記のループで待つ
    while ($result -eq 0)
    {
        $credential = Get-Credential -Message "TeamCity Account Required"
        if (!$credential)
        {
            $window.Close()
            exit
        }
        $username = $credential.UserName
        $password = $credential.Password

        $result = TeamCityAcountValidation $username $password
    }

    $outputList = ""

    try
    {
        if ($username.Length -eq 0)
        {
            throw "Input TeamCity username is invalid."
        }

        if ($artifactsUrl.Length -eq 0)
        {
            throw "Input ArtifactsURL is invalid."
        }

        # $builds = "Debug", "Develop", "Release"
        # $targets = "NX-NXFP2-a32", "NX-NXFP2-a64"
	
		# テスト対象を絞ったため現在は NXFP2-a64/Release のみダウンロードしている
		$builds = "Release"
        $targets = "NX-NXFP2-a64"

        # http:// から始まっていなければ付与します。
        if(!$artifactsUrl.StartsWith("http://"))
        {
            $artifactsUrl = "http://" + $artifactsUrl;
        }

        foreach($target in $targets)
        {
            foreach($build in $builds)
            {
				foreach($checkBox in $outputSelection)
				{
					# 出力判定用のチェックボックスのチェックが有効なら、チェックボックスの Content を Prefix としてデータを出力する。
					if ($checkBox.IsChecked)
					{
						$result = DownloadProfilingData $outputPath.Trim() $checkBox.Content $artifactsUrl.Trim() $target $build $username $password $inputComment
						if ($result.Length -ne 0)
						{
							$outputList += $result
							$outputList += "`n`n"
						}
					}
				}
            }
        }
    }
    catch [Exception]
    {
        Write-Host $error[0]
    }
    finally
    {
        # 設定ファイルへユーザー名を保存
        $settingFile.prop.settings.ci_username = $username
        $encryptedPassword = ConvertFrom-SecureString -SecureString $password
        [string]$settingFile.prop.settings.ci_password = $encryptedPassword

        # 出力ディレクトリ履歴に存在しないディレクトリを指定されていたらリストへ加える
        $found =  0
        foreach($path in $settingFile.prop.outputPathHistories.path)
        {
            if ($path -eq $outputPath)
            {
                $found = 1
                break
            }
        }

        if ($found -eq 0)
        {
            $newPath = $settingFile.CreateElement("path")
            $newPath.InnerText = $outputPath

            if ($settingFile.prop.outputPathHistories.path.Length -eq 1)
            {
                $settingFile.prop.outputPathHistories.path[0] = $newPath
            }
            else
            {
                $settingFile.prop.outputPathHistories.PrependChild($newPath)
            }
        }

        $window.Close()

        if ($outputList.Length -eq 0)
        {
            [System.Windows.Forms.MessageBox]::Show("データは出力されませんでした。", "処理結果")
        }
        else
        {
            [System.Windows.Forms.MessageBox]::Show("$outputList`nが出力されました。", "処理結果")
        }
    }
})

$window.ShowDialog()

$settingFile.Save($settingFilePath)
