﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testPerf_Main.h"
#include <nnt.h>

// 繰り返し初期化と終了を行う
TEST(GpuMeasureLoadMeterCenter, RepeatInitialize)
{
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);
    nn::gfx::CommandBuffer* rootCommandBuffer = GetGfw()->GetRootCommandBuffer(0);

    for(int i = 1; i < 1000; i++)
    {
        LoadMeterCenterManager::Initialize(true, 3, 0, 2, 2, i);

        NN_PERF_BEGIN_FRAME();
        GetGfw()->BeginFrame(0);
        NN_PERF_BEGIN_MEASURE_GPU(rootCommandBuffer);
        SleepGpu();
        NN_PERF_END_MEASURE_GPU(rootCommandBuffer);
        GetGfw()->EndFrame(0);
        NN_PERF_END_FRAME();

        LoadMeterCenterManager::Finalize();
    }

    SUCCEED();
}

// 設定
TEST(GpuMeasureLoadMeterCenter, Set)
{
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);
    LoadMeterCenterManager::Initialize(true, 1, 1, 2, 2, 1);

    nn::perf::CpuMeter* frameMeter = NN_PERF_GET_FRAME_METER();
    nn::perf::CpuMeter* coreMeter = NN_PERF_GET_CORE_METER(0);
    nn::perf::CpuMeter* userMeter = NN_PERF_GET_USER_METER(0);
    nn::perf::GpuMeter* gpuMeter = NN_PERF_GET_GPU_METER();

    EXPECT_EQ(nn::util::Color4u8(16,16,16,150), frameMeter->GetColor());
    EXPECT_EQ(nn::util::Color4u8::Green(), coreMeter->GetColor());
    EXPECT_EQ(nn::util::Color4u8::Green(), userMeter->GetColor());
    EXPECT_EQ(nn::util::Color4u8::Green(), gpuMeter->GetColor());

    EXPECT_EQ(60.f, nn::perf::LoadMeterCenter::GetFrameRate());
    EXPECT_EQ(0, nn::util::Strncmp(frameMeter->GetName(), "Frame", 128));
    EXPECT_EQ(0, nn::util::Strncmp(coreMeter->GetName(), "CPU0", 128));
    EXPECT_EQ(0, nn::util::Strncmp(userMeter->GetName(), "USER0", 128));
    EXPECT_EQ(0, nn::util::Strncmp(gpuMeter->GetName(), "GPU", 128));

    NN_PERF_SET_FRAME_RATE(30.f);
    NN_PERF_SET_COLOR(nn::util::Color4u8::Red());
    NN_PERF_SET_COLOR_INDEX(0, nn::util::Color4u8::Yellow());
    NN_PERF_SET_COLOR_GPU( nn::util::Color4u8::Black());
    NN_PERF_SET_METER_NAME(0, "Test1");

    EXPECT_EQ(nn::util::Color4u8(16,16,16,150), frameMeter->GetColor());
    EXPECT_EQ(nn::util::Color4u8::Red(), coreMeter->GetColor());
    EXPECT_EQ(nn::util::Color4u8::Yellow(), userMeter->GetColor());
    EXPECT_EQ(nn::util::Color4u8::Black(), gpuMeter->GetColor());

    EXPECT_EQ(30.f, nn::perf::LoadMeterCenter::GetFrameRate());
    EXPECT_EQ(0, nn::util::Strncmp(frameMeter->GetName(), "Frame", 128));
    EXPECT_EQ(0, nn::util::Strncmp(coreMeter->GetName(), "CPU0", 128));
    EXPECT_EQ(0, nn::util::Strncmp(userMeter->GetName(), "Test1", 128));
    EXPECT_EQ(0, nn::util::Strncmp(gpuMeter->GetName(), "GPU", 128));

    LoadMeterCenterManager::Finalize();
}

// アタッチされているかをテスト
TEST(GpuMeasureLoadMeterCenter, TestAttach)
{
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);
    LoadMeterCenterManager::Initialize(true, 3, 3, 2, 2, 1);

    nn::perf::CpuMeter* frameMeter = NN_PERF_GET_FRAME_METER();
    nn::perf::CpuMeter* coreMeter[3];
    coreMeter[0] = NN_PERF_GET_CORE_METER(0);
    coreMeter[1] = NN_PERF_GET_CORE_METER(1);
    coreMeter[2] = NN_PERF_GET_CORE_METER(2);
    nn::perf::CpuMeter* userMeter[3];
    userMeter[0] = NN_PERF_GET_USER_METER(0);
    userMeter[1] = NN_PERF_GET_USER_METER(1);
    userMeter[2] = NN_PERF_GET_USER_METER(2);
    nn::perf::GpuMeter* gpuMeter = NN_PERF_GET_GPU_METER();

    ASSERT_FALSE(frameMeter->IsLinked());

    ASSERT_TRUE(coreMeter[0]->IsLinked());
    ASSERT_TRUE(coreMeter[1]->IsLinked());
    ASSERT_TRUE(coreMeter[2]->IsLinked());

    ASSERT_TRUE(userMeter[0]->IsLinked());
    ASSERT_TRUE(userMeter[1]->IsLinked());
    ASSERT_TRUE(userMeter[2]->IsLinked());

    ASSERT_TRUE(gpuMeter->IsLinked());

    LoadMeterCenterManager::Finalize();
}

// 設定した最大区間を計測する
TEST(GpuMeasureLoadMeterCenter, MeasureMaxSection)
{
    int sectionCount = 1;
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);
    nn::gfx::CommandBuffer* rootCommandBuffer = GetGfw()->GetRootCommandBuffer(0);
    LoadMeterCenterManager::Initialize(true, 1, 0, 2, 2, sectionCount);

    nn::perf::CpuMeter* frameMeter = NN_PERF_GET_FRAME_METER();
    nn::perf::GpuMeter* gpuMeter = NN_PERF_GET_GPU_METER();

    NN_PERF_BEGIN_FRAME();
    GetGfw()->BeginFrame(0);
    for(int i = 0; i < sectionCount; i++)
    {
        NN_PERF_BEGIN_MEASURE_NAME_AND_TAG_GPU(rootCommandBuffer, "Test1", i);
        SleepGpu();
        NN_PERF_END_MEASURE_GPU(rootCommandBuffer);
    }
    GetGfw()->EndFrame(0);
    GetGfw()->ExecuteCommand(0);
    GetGfw()->QueuePresentTexture(1);
    GetGfw()->QueueFinish();
    NN_PERF_END_FRAME();

    EXPECT_EQ(1, frameMeter->GetLastSectionCount());
    EXPECT_LT(0, (frameMeter->GetLastTotalEnd() - frameMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(sectionCount, gpuMeter->GetLastSectionCount());
    EXPECT_LT(0, (gpuMeter->GetLastTotalEnd() - gpuMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(gpuMeter->GetLastTotalBegin(), gpuMeter->GetLastResult(0).begin);
    EXPECT_EQ(gpuMeter->GetLastTotalEnd(), gpuMeter->GetLastResult(sectionCount - 1).end);

    for(int i = 0; i < sectionCount; i++)
    {
        const nn::perf::LoadMeterBase::Section& section = gpuMeter->GetLastResult(i);
        EXPECT_LT(0, (section.end - section.begin).ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, section.depth);
        EXPECT_EQ(i, section.tag);
        EXPECT_EQ("Test1", section.name);
    }
    LoadMeterCenterManager::Finalize();

    sectionCount = 5000;
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);
    LoadMeterCenterManager::Initialize(true, 1, 0, 2, 2, sectionCount);
    frameMeter = NN_PERF_GET_FRAME_METER();
    gpuMeter = NN_PERF_GET_GPU_METER();

    NN_PERF_BEGIN_FRAME();
    GetGfw()->BeginFrame(0);
    for(int i = 0; i < sectionCount; i++)
    {
        NN_PERF_BEGIN_MEASURE_NAME_AND_TAG_GPU(rootCommandBuffer, "Test2", i);
        SleepGpu();
        NN_PERF_END_MEASURE_GPU(rootCommandBuffer);
    }
    GetGfw()->EndFrame(0);
    GetGfw()->ExecuteCommand(0);
    GetGfw()->QueuePresentTexture(1);
    GetGfw()->QueueFinish();
    NN_PERF_END_FRAME();

    EXPECT_EQ(1, frameMeter->GetLastSectionCount());
    EXPECT_LT(0, (frameMeter->GetLastTotalEnd() - frameMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(sectionCount, gpuMeter->GetLastSectionCount());
    EXPECT_LT(0, (gpuMeter->GetLastTotalEnd() - gpuMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(gpuMeter->GetLastTotalBegin(), gpuMeter->GetLastResult(0).begin);
    EXPECT_EQ(gpuMeter->GetLastTotalEnd(), gpuMeter->GetLastResult(sectionCount - 1).end);

    for(int i = 0; i < sectionCount; i++)
    {
        const nn::perf::LoadMeterBase::Section& section = gpuMeter->GetLastResult(i);
        EXPECT_LT(0, (section.end - section.begin).ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, section.depth);
        EXPECT_EQ(i, section.tag);
        EXPECT_EQ("Test2", section.name);
    }
    LoadMeterCenterManager::Finalize();
}

// バッファ数に対応した結果が取得できるか
TEST(GpuMeasureLoadMeterCenter, TestBufferCount)
{
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);
    nn::gfx::CommandBuffer* rootCommandBuffer = GetGfw()->GetRootCommandBuffer(0);

    // バッファ数を 4 に設定
    // CPU は 2(Buffer-1)回目に呼ばれた NN_PERF_END_FRAME で計測した結果が集計される。
    // GPU は 3(Buffer-1)回目に呼ばれた NN_PERF_END_FRAME で計測した結果が集計される。
    // (デフォルトのバッファ数 2 では、計測後 1(buffer-1)回目に呼ばれた NN_PERF_END_FRAME で集計される)
    LoadMeterCenterManager::Initialize(true, 1, 0, 3, 4, 1);
    nn::perf::CpuMeter* frameMeter = NN_PERF_GET_FRAME_METER();
    nn::perf::CpuMeter* coreMeter = NN_PERF_GET_CORE_METER(0);
    nn::perf::GpuMeter* gpuMeter = NN_PERF_GET_GPU_METER();

    // 1 フレーム目
    NN_PERF_BEGIN_FRAME();
    NN_PERF_BEGIN_MEASURE_NAME_AND_TAG("Test1", 1);
    Sleep();
    NN_PERF_END_MEASURE();

    GetGfw()->BeginFrame(0);
    NN_PERF_BEGIN_MEASURE_NAME_AND_TAG_GPU(rootCommandBuffer, "TestGpu1", 11);
    SleepGpu();
    NN_PERF_END_MEASURE_GPU(rootCommandBuffer);
    GetGfw()->EndFrame(0);

    GetGfw()->ExecuteCommand(0);
    GetGfw()->QueuePresentTexture(1);
    GetGfw()->QueueFinish();
    NN_PERF_END_FRAME();

    // 1 フレーム後。空の結果を取得 (GetLastSectionCount() == 0 の場合はそもそもアクセスしない方がベター)。
    {
        EXPECT_EQ(0, frameMeter->GetLastSectionCount());
        EXPECT_EQ(0, frameMeter->GetLastTotalBegin().ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, frameMeter->GetLastTotalEnd().ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, coreMeter->GetLastSectionCount());
        EXPECT_EQ(0, coreMeter->GetLastTotalBegin().ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, coreMeter->GetLastTotalEnd().ToTimeSpan().GetNanoSeconds());
        const nn::perf::LoadMeterBase::Section& section = coreMeter->GetLastResult(0);
        EXPECT_EQ(0, section.begin.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, section.end.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(NULL, section.name);
        EXPECT_EQ(0, section.tag);
        EXPECT_EQ(nn::util::Color4u8::Green(), section.color);

        EXPECT_EQ(0, gpuMeter->GetLastSectionCount());
        EXPECT_EQ(0, gpuMeter->GetLastTotalBegin().ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, gpuMeter->GetLastTotalEnd().ToTimeSpan().GetNanoSeconds());
        const nn::perf::LoadMeterBase::Section& sectionGpu = coreMeter->GetLastResult(0);
        EXPECT_EQ(0, sectionGpu.begin.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, sectionGpu.end.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(NULL, sectionGpu.name);
        EXPECT_EQ(0, sectionGpu.tag);
        EXPECT_EQ(nn::util::Color4u8::Green(), sectionGpu.color);
    }

    // 2 フレーム目
    NN_PERF_BEGIN_FRAME();
    NN_PERF_BEGIN_MEASURE_NAME_AND_TAG("Test2", 2);
    Sleep();
    NN_PERF_END_MEASURE();

    GetGfw()->BeginFrame(0);
    NN_PERF_BEGIN_MEASURE_NAME_AND_TAG_GPU(rootCommandBuffer, "TestGpu2", 22);
    SleepGpu();
    NN_PERF_END_MEASURE_GPU(rootCommandBuffer);
    GetGfw()->EndFrame(0);

    GetGfw()->ExecuteCommand(0);
    GetGfw()->QueuePresentTexture(1);
    GetGfw()->QueueFinish();
    NN_PERF_END_FRAME();

    // 2 フレーム後。CPU は 1フレーム目の結果を取得、 GPU は空。
    {
        EXPECT_EQ(1, frameMeter->GetLastSectionCount());
        EXPECT_LT(0, (frameMeter->GetLastTotalEnd() - frameMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(1, coreMeter->GetLastSectionCount());
        EXPECT_LT(0, (coreMeter->GetLastTotalEnd() - coreMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
        const nn::perf::LoadMeterBase::Section& section = coreMeter->GetLastResult(0);
        EXPECT_LT(0, (section.end - section.begin).ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ("Test1", section.name);
        EXPECT_EQ(1, section.tag);

        EXPECT_EQ(0, gpuMeter->GetLastSectionCount());
        EXPECT_EQ(0, gpuMeter->GetLastTotalBegin().ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, gpuMeter->GetLastTotalEnd().ToTimeSpan().GetNanoSeconds());
        const nn::perf::LoadMeterBase::Section& sectionGpu = gpuMeter->GetLastResult(0);
        EXPECT_EQ(0, sectionGpu.begin.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, sectionGpu.end.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(NULL, sectionGpu.name);
        EXPECT_EQ(0, sectionGpu.tag);
    }

    // 3 フレーム目
    NN_PERF_BEGIN_FRAME();
    NN_PERF_BEGIN_MEASURE_NAME_AND_TAG("Test3", 3);
    Sleep();
    NN_PERF_END_MEASURE();

    GetGfw()->BeginFrame(0);
    NN_PERF_BEGIN_MEASURE_NAME_AND_TAG_GPU(rootCommandBuffer, "TestGpu3", 33);
    SleepGpu();
    NN_PERF_END_MEASURE_GPU(rootCommandBuffer);
    GetGfw()->EndFrame(0);

    GetGfw()->ExecuteCommand(0);
    GetGfw()->QueuePresentTexture(1);
    GetGfw()->QueueFinish();
    NN_PERF_END_FRAME();

    // 3  フレーム後。CPU は 2フレーム目の結果を取得、GPU は 1フレーム目の結果を取得。
    {
        EXPECT_EQ(1, frameMeter->GetLastSectionCount());
        EXPECT_LT(0, (frameMeter->GetLastTotalEnd() - frameMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(1, coreMeter->GetLastSectionCount());
        EXPECT_LT(0, (coreMeter->GetLastTotalEnd() - coreMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
        const nn::perf::LoadMeterBase::Section& section = coreMeter->GetLastResult(0);
        EXPECT_LT(0, (section.end - section.begin).ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ("Test2", section.name);
        EXPECT_EQ(2, section.tag);

        EXPECT_EQ(1, gpuMeter->GetLastSectionCount());
        EXPECT_LT(0, (gpuMeter->GetLastTotalEnd() - gpuMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
        const nn::perf::LoadMeterBase::Section& sectionGpu = gpuMeter->GetLastResult(0);
        EXPECT_LT(0, (sectionGpu.end - sectionGpu.begin).ToTimeSpan().GetNanoSeconds());
        EXPECT_LT(0, sectionGpu.begin.ToTimeSpan().GetNanoSeconds());
        EXPECT_LT(0, sectionGpu.end.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ("TestGpu1", sectionGpu.name);
        EXPECT_EQ(11, sectionGpu.tag);
    }

    // 4 フレーム目
    NN_PERF_BEGIN_FRAME();
    GetGfw()->BeginFrame(0);
    SleepGpu();
    GetGfw()->EndFrame(0);
    GetGfw()->ExecuteCommand(0);
    GetGfw()->QueuePresentTexture(1);
    GetGfw()->QueueFinish();
    NN_PERF_END_FRAME();

    // 4 フレーム後。CPU は 3フレーム目の結果を取得、GPU は 2フレーム目の結果を取得。
    {
        EXPECT_EQ(1, frameMeter->GetLastSectionCount());
        EXPECT_LT(0, (frameMeter->GetLastTotalEnd() - frameMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(1, coreMeter->GetLastSectionCount());
        EXPECT_LT(0, (coreMeter->GetLastTotalEnd() - coreMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
        const nn::perf::LoadMeterBase::Section& section = coreMeter->GetLastResult(0);
        EXPECT_LT(0, (section.end - section.begin).ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ("Test3", section.name);
        EXPECT_EQ(3, section.tag);

        EXPECT_EQ(1, gpuMeter->GetLastSectionCount());
        EXPECT_LT(0, (gpuMeter->GetLastTotalEnd() - gpuMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
        const nn::perf::LoadMeterBase::Section& sectionGpu = gpuMeter->GetLastResult(0);
        EXPECT_LT(0, (sectionGpu.end - sectionGpu.begin).ToTimeSpan().GetNanoSeconds());
        EXPECT_LT(0, sectionGpu.begin.ToTimeSpan().GetNanoSeconds());
        EXPECT_LT(0, sectionGpu.end.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ("TestGpu2", sectionGpu.name);
        EXPECT_EQ(22, sectionGpu.tag);
    }

    // 5 フレーム目
    NN_PERF_BEGIN_FRAME();
    GetGfw()->BeginFrame(0);
    SleepGpu();
    GetGfw()->EndFrame(0);
    GetGfw()->ExecuteCommand(0);
    GetGfw()->QueuePresentTexture(1);
    GetGfw()->QueueFinish();
    NN_PERF_END_FRAME();

    // 5 フレーム後。CPU は 4フレーム目の結果（空）、GPU は 3フレーム目の結果を取得。
    {
        EXPECT_EQ(1, frameMeter->GetLastSectionCount());
        EXPECT_LT(0, (frameMeter->GetLastTotalEnd() - frameMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, coreMeter->GetLastSectionCount());

        EXPECT_EQ(1, gpuMeter->GetLastSectionCount());
        EXPECT_LT(0, (gpuMeter->GetLastTotalEnd() - gpuMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
        const nn::perf::LoadMeterBase::Section& sectionGpu = gpuMeter->GetLastResult(0);
        EXPECT_LT(0, (sectionGpu.end - sectionGpu.begin).ToTimeSpan().GetNanoSeconds());
        EXPECT_LT(0, sectionGpu.begin.ToTimeSpan().GetNanoSeconds());
        EXPECT_LT(0, sectionGpu.end.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ("TestGpu3", sectionGpu.name);
        EXPECT_EQ(33, sectionGpu.tag);
    }
    LoadMeterCenterManager::Finalize();
} //NOLINT(impl/function_size)

// Generic 版では連続でタイムスタンプを呼ぶと止まる不具合があるので以下のテストは行わない
#if defined(NN_BUILD_CONFIG_SPEC_NX)
// 入れ子で計測できるか判断する
TEST(GpuMeasureLoadMeterCenter, MeasureNest)
{
    int sectionCount = 10240;
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);
    nn::gfx::CommandBuffer* rootCommandBuffer = GetGfw()->GetRootCommandBuffer(0);
    LoadMeterCenterManager::Initialize(true, 1, 0, 2, 2, sectionCount);
    nn::perf::GpuMeter* gpuMeter = NN_PERF_GET_GPU_METER();

    NN_PERF_BEGIN_FRAME();
    GetGfw()->BeginFrame(0);
    for(int i = 0; i < sectionCount; i++)
    {
        NN_PERF_BEGIN_MEASURE_NAME_AND_TAG_GPU(rootCommandBuffer, "Test3", 3);
    }
    SleepGpu();
    for(int i = 0; i < sectionCount; i++)
    {
        NN_PERF_END_MEASURE_GPU(rootCommandBuffer);
    }
    GetGfw()->EndFrame(0);
    GetGfw()->ExecuteCommand(0);
    GetGfw()->QueuePresentTexture(1);
    GetGfw()->QueueFinish();
    NN_PERF_END_FRAME();

    EXPECT_EQ(sectionCount, gpuMeter->GetLastSectionCount());
    EXPECT_LT(0, (gpuMeter->GetLastTotalEnd() - gpuMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(gpuMeter->GetLastTotalBegin(), gpuMeter->GetLastResult(0).begin);
    EXPECT_EQ(gpuMeter->GetLastTotalEnd(), gpuMeter->GetLastResult(0).end);

    for(int i = 0; i < sectionCount; i++)
    {
        const nn::perf::LoadMeterBase::Section& section = gpuMeter->GetLastResult(i);
        EXPECT_LT(0, (section.end - section.begin).ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(i, section.depth);
    }
    LoadMeterCenterManager::Finalize();
}
#endif

// Next() が呼ばれない時に結果を取得した場合
TEST(GpuMeasureLoadMeterCenter, MeasureNull)
{
    nn::gfx::CommandBuffer* rootCommandBuffer = GetGfw()->GetRootCommandBuffer(0);
    LoadMeterCenterManager::Initialize(true, 1, 0, 2, 2, 1);
    nn::perf::CpuMeter* frameMeter = NN_PERF_GET_FRAME_METER();
    nn::perf::GpuMeter* gpuMeter = NN_PERF_GET_GPU_METER();

    NN_PERF_BEGIN_FRAME();
    GetGfw()->BeginFrame(0);
    NN_PERF_BEGIN_MEASURE_NAME_AND_TAG_GPU(rootCommandBuffer, "TestGpu3", 33);
    SleepGpu();
    NN_PERF_END_MEASURE_GPU(rootCommandBuffer);
    GetGfw()->EndFrame(0);

    // 空の結果を取得。
    {
        EXPECT_EQ(0, frameMeter->GetLastSectionCount());
        EXPECT_EQ(0, frameMeter->GetLastTotalBegin().ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, frameMeter->GetLastTotalEnd().ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, gpuMeter->GetLastSectionCount());
        EXPECT_EQ(0, gpuMeter->GetLastTotalBegin().ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, gpuMeter->GetLastTotalEnd().ToTimeSpan().GetNanoSeconds());
        const nn::perf::LoadMeterBase::Section& section = gpuMeter->GetLastResult(0);
        EXPECT_EQ(0, section.begin.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, section.end.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(NULL, section.name);
        EXPECT_EQ(0, section.tag);
        EXPECT_EQ(nn::util::Color4u8::Green(), section.color);
    }
    GetGfw()->ExecuteCommand(0);
    GetGfw()->QueuePresentTexture(1);
    GetGfw()->QueueFinish();
    NN_PERF_END_FRAME();
    LoadMeterCenterManager::Finalize();
}

// 計測結果が妥当か判断する
TEST(GpuMeasureLoadMeterCenter, CheckResult)
{
    nn::gfx::CommandBuffer* rootCommandBuffer = GetGfw()->GetRootCommandBuffer(0);
    int sectionCount = 64;
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);
    LoadMeterCenterManager::Initialize(true, 1, 0, 2, 2, sectionCount);
    nn::perf::GpuMeter* gpuMeter = NN_PERF_GET_GPU_METER();

    NN_PERF_BEGIN_FRAME();
    GetGfw()->BeginFrame(0);

    EXPECT_EQ(nn::util::Color4u8::Green(), gpuMeter->GetColor());

    NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::White());

    EXPECT_EQ(nn::util::Color4u8::White(), gpuMeter->GetColor());

    NN_PERF_BEGIN_MEASURE_GPU(rootCommandBuffer);
    SleepGpu();
    NN_PERF_END_MEASURE_GPU(rootCommandBuffer);

    NN_PERF_BEGIN_MEASURE_NAME_GPU(rootCommandBuffer, "Test1");
    SleepGpu();
    NN_PERF_END_MEASURE_GPU(rootCommandBuffer);

    NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::Blue());

    EXPECT_EQ(nn::util::Color4u8::Blue(), gpuMeter->GetColor());

    NN_PERF_BEGIN_MEASURE_NAME_AND_TAG_GPU(rootCommandBuffer, "Test2", 32);
    SleepGpu();
    NN_PERF_END_MEASURE_GPU(rootCommandBuffer);

    NN_PERF_BEGIN_MEASURE_TAG_GPU(rootCommandBuffer, 16);
    SleepGpu();
    NN_PERF_END_MEASURE_GPU(rootCommandBuffer);

    GetGfw()->EndFrame(0);
    GetGfw()->ExecuteCommand(0);
    GetGfw()->QueuePresentTexture(1);
    GetGfw()->QueueFinish();
    NN_PERF_END_FRAME();

    EXPECT_EQ(4, gpuMeter->GetLastSectionCount());
    EXPECT_LT(0, (gpuMeter->GetLastTotalEnd() - gpuMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(gpuMeter->GetLastTotalBegin(), gpuMeter->GetLastResult(0).begin);
    EXPECT_EQ(gpuMeter->GetLastTotalEnd(), gpuMeter->GetLastResult(4 - 1).end);

    const nn::perf::LoadMeterBase::Section& section = gpuMeter->GetLastResult(0);
    EXPECT_LT(0, (section.end - section.begin).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(NULL, section.name);
    EXPECT_EQ(0, section.tag);
    EXPECT_EQ(nn::util::Color4u8::White(), section.color);

    const nn::perf::LoadMeterBase::Section& section1 = gpuMeter->GetLastResult(1);
    EXPECT_LT(0, (section1.end - section1.begin).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ("Test1", section1.name);
    EXPECT_EQ(0, section1.tag);
    EXPECT_EQ(nn::util::Color4u8::White(), section1.color);

    const nn::perf::LoadMeterBase::Section& section2 = gpuMeter->GetLastResult(2);
    EXPECT_LT(0, (section2.end - section2.begin).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ("Test2", section2.name);
    EXPECT_EQ(32, section2.tag);
    EXPECT_EQ(nn::util::Color4u8::Blue(), section2.color);

    const nn::perf::LoadMeterBase::Section& section3 = gpuMeter->GetLastResult(3);
    EXPECT_LT(0, (section3.end - section3.begin).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(NULL, section3.name);
    EXPECT_EQ(16, section3.tag);
    EXPECT_EQ(nn::util::Color4u8::Blue(), section3.color);

    LoadMeterCenterManager::Finalize();
}

// 有効無効
TEST(GpuMeasureLoadMeterCenter, CheckEnabled)
{
    nn::gfx::CommandBuffer* rootCommandBuffer = GetGfw()->GetRootCommandBuffer(0);
    int sectionCount = 64;
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);
    LoadMeterCenterManager::Initialize(true, 1, 0, 2, 2, sectionCount);
    nn::perf::CpuMeter* frameMeter = NN_PERF_GET_FRAME_METER();
    nn::perf::GpuMeter* gpuMeter = NN_PERF_GET_GPU_METER();

    NN_PERF_SET_ENABLED(false);
    NN_PERF_BEGIN_FRAME();
    GetGfw()->BeginFrame(0);
    NN_PERF_BEGIN_MEASURE_NAME_AND_TAG_GPU(rootCommandBuffer, "Test2", 32);
    Sleep();
    NN_PERF_END_MEASURE_GPU(rootCommandBuffer);
    GetGfw()->EndFrame(0);
    GetGfw()->ExecuteCommand(0);
    GetGfw()->QueuePresentTexture(1);
    GetGfw()->QueueFinish();
    NN_PERF_END_FRAME();

    EXPECT_EQ(0, frameMeter->GetLastSectionCount());
    EXPECT_EQ(0, frameMeter->GetLastTotalBegin().ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(0, frameMeter->GetLastTotalEnd().ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(0, gpuMeter->GetLastSectionCount());
    EXPECT_EQ(0, gpuMeter->GetLastTotalBegin().ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(0, gpuMeter->GetLastTotalEnd().ToTimeSpan().GetNanoSeconds());

    NN_PERF_SET_ENABLED(true);
    NN_PERF_BEGIN_FRAME();
    GetGfw()->BeginFrame(0);
    NN_PERF_BEGIN_MEASURE_NAME_AND_TAG_GPU(rootCommandBuffer, "Test2", 32);
    Sleep();
    NN_PERF_END_MEASURE_GPU(rootCommandBuffer);
    GetGfw()->EndFrame(0);
    GetGfw()->ExecuteCommand(0);
    GetGfw()->QueuePresentTexture(1);
    GetGfw()->QueueFinish();
    NN_PERF_END_FRAME();

    EXPECT_EQ(1, frameMeter->GetLastSectionCount());
    EXPECT_LT(0, (frameMeter->GetLastTotalEnd() - frameMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(1, gpuMeter->GetLastSectionCount());
    EXPECT_LT(0, (gpuMeter->GetLastTotalEnd() - gpuMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());

    LoadMeterCenterManager::Finalize();
}

namespace{

const size_t                    threadStackSize = 0x4000;
NN_OS_ALIGNAS_THREAD_STACK char threadStack[3][threadStackSize];
nn::os::ThreadType              thread[3];
TlsSlotValue                    tlsSlotValue[3];

nn::os::TlsSlot tlsSlot;
int GetCoreNumberFunction()
{
    int coreNumber = reinterpret_cast<TlsSlotValue*>(nn::os::GetTlsValue(tlsSlot))->coreNumber;
    return *reinterpret_cast<int*>(&coreNumber);
}

void ThreadMeasureGpu(void *tlsValue)
{
    nn::os::SetTlsValue(tlsSlot, reinterpret_cast<uintptr_t>(tlsValue));
    int coreNumber = reinterpret_cast<TlsSlotValue*>(nn::os::GetTlsValue(tlsSlot))->coreNumber;
    const char* name = reinterpret_cast<TlsSlotValue*>(nn::os::GetTlsValue(tlsSlot))->name;
    nn::util::Color4u8 color = reinterpret_cast<TlsSlotValue*>(nn::os::GetTlsValue(tlsSlot))->color;
    nn::gfx::CommandBuffer* commandBuffer = &GetSubCommandBuffer(coreNumber)->object;
    nn::perf::GpuMeter* gpuMeter = NN_PERF_GET_GPU_METER();

    EXPECT_EQ(nn::util::Color4u8::Green(), gpuMeter->GetColor());
    commandBuffer->Begin();
    NN_PERF_SET_COLOR_GPU(color);
    NN_PERF_BEGIN_MEASURE_NAME_GPU(commandBuffer, name);
    SleepGpu(coreNumber);
    NN_PERF_END_MEASURE_GPU(commandBuffer);
    commandBuffer->End();
    EXPECT_EQ(color, gpuMeter->GetColor());
}
}

// 計測結果の取得
TEST(GpuMeasureLoadMeterCenter, CheckProfileResult)
{
    // アフィニティマスクの設定
    nn::Bit64 core0 = 1;
    nn::Bit64 core1 = 2;
    nn::Bit64 core2 = 4;

    {
        nn::Result result = nn::os::AllocateTlsSlot(&tlsSlot, NULL);
        NN_ASSERT(result.IsSuccess(), "Cannot allocate TLS slot.");

        tlsSlotValue[0].coreNumber = 0;
        tlsSlotValue[0].name = "Test0";
        tlsSlotValue[0].color = nn::util::Color4u8::Red();
        result = nn::os::CreateThread(&thread[0], ThreadMeasureGpu, &tlsSlotValue[0], threadStack[0], threadStackSize, nn::os::DefaultThreadPriority);
        nn::os::SetThreadName(&thread[0], "Thread0");
        nn::os::SetThreadCoreMask(&thread[0], 0, core0);

        tlsSlotValue[1].coreNumber = 1;
        tlsSlotValue[1].name = "Test1";
        tlsSlotValue[1].color = nn::util::Color4u8::Blue();
        result = nn::os::CreateThread(&thread[1], ThreadMeasureGpu, &tlsSlotValue[1], threadStack[1], threadStackSize, nn::os::DefaultThreadPriority);
        nn::os::SetThreadName(&thread[1], "Thread1");
        nn::os::SetThreadCoreMask(&thread[1], 1, core1);

        tlsSlotValue[2].coreNumber = 2;
        tlsSlotValue[2].name = "Test2";
        tlsSlotValue[2].color = nn::util::Color4u8::Yellow();
        result = nn::os::CreateThread(&thread[2], ThreadMeasureGpu, &tlsSlotValue[2], threadStack[2], threadStackSize, nn::os::DefaultThreadPriority);
        nn::os::SetThreadName(&thread[2], "Thread2");
        nn::os::SetThreadCoreMask(&thread[2], 2, core2);

        nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, core0);
        nn::os::SetTlsValue(tlsSlot, reinterpret_cast<uintptr_t>(&tlsSlotValue[0]));
    }

    InitializeSubCommandBuffer();
    int sectionCount = 64;
    nn::gfx::CommandBuffer* rootCommandBuffer = GetGfw()->GetRootCommandBuffer(0);
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);
    LoadMeterCenterManager::Initialize(true, 3, 0, 2, 2, sectionCount);
    nn::perf::GpuMeter* gpuMeter = NN_PERF_GET_GPU_METER();
    NN_PERF_SET_GET_CORE_NUMBER_FUNCTION(GetCoreNumberFunction);

    // 計測前の空結果
    EXPECT_EQ(0, NN_PERF_GET_CALL_COUNT_GPU(NULL, 0));
    EXPECT_EQ(0, NN_PERF_GET_CALL_COUNT_GPU(NULL, 8));
    EXPECT_EQ(0, NN_PERF_GET_CALL_COUNT_GPU(NULL, 16));
    EXPECT_EQ(0, NN_PERF_GET_CALL_COUNT_GPU(NULL, 32));
    EXPECT_EQ(0, NN_PERF_GET_CALL_COUNT_GPU("Test1", 0));
    EXPECT_EQ(0, NN_PERF_GET_CALL_COUNT_GPU("Test2", 0));
    EXPECT_EQ(0, NN_PERF_GET_CALL_COUNT_GPU("Test1", 8));

    EXPECT_EQ(0, NN_PERF_GET_AVERAGE_ELAPSED_TIME_GPU(NULL, 0).GetNanoSeconds());
    EXPECT_EQ(0, NN_PERF_GET_AVERAGE_ELAPSED_TIME_GPU(NULL, 0).GetNanoSeconds());
    EXPECT_EQ(0, NN_PERF_GET_AVERAGE_ELAPSED_TIME_GPU("Test1", 0).GetNanoSeconds());

    EXPECT_EQ(0, NN_PERF_GET_ELAPSED_TIME_GPU("Test2", 32).GetNanoSeconds());
    EXPECT_EQ(0, NN_PERF_GET_ELAPSED_TIME_GPU(NULL, 0).GetNanoSeconds());
    EXPECT_EQ(0, NN_PERF_GET_ELAPSED_TIME_GPU(NULL, 16).GetNanoSeconds());

    int64_t min = nn::TimeSpan( std::numeric_limits<nn::TimeSpan>::min() ).GetNanoSeconds();
    int64_t max = nn::TimeSpan::FromSeconds( 999 ).GetNanoSeconds();

    EXPECT_EQ(min, NN_PERF_GET_MAX_ELAPSED_TIME_GPU(NULL, 0).GetNanoSeconds());
    EXPECT_EQ(min, NN_PERF_GET_MAX_ELAPSED_TIME_GPU(NULL, 0).GetNanoSeconds());
    EXPECT_EQ(min, NN_PERF_GET_MAX_ELAPSED_TIME_GPU("Test1", 0).GetNanoSeconds());

    EXPECT_EQ(max, NN_PERF_GET_MIN_ELAPSED_TIME_GPU(NULL, 0).GetNanoSeconds());
    EXPECT_EQ(max, NN_PERF_GET_MIN_ELAPSED_TIME_GPU(NULL, 0).GetNanoSeconds());
    EXPECT_EQ(max, NN_PERF_GET_MIN_ELAPSED_TIME_GPU(NULL, 32).GetNanoSeconds());

    NN_PERF_BEGIN_FRAME();
    GetGfw()->BeginFrame(0);
    {
        GetGfw()->ResetCommandBuffer(GetSubCommandBuffer(0));
        GetGfw()->ResetCommandBuffer(GetSubCommandBuffer(1));
        GetGfw()->ResetCommandBuffer(GetSubCommandBuffer(2));
        nn::os::StartThread(&thread[0]);
        nn::os::StartThread(&thread[1]);
        nn::os::StartThread(&thread[2]);
        nn::os::WaitThread(&thread[0]);
        nn::os::WaitThread(&thread[1]);
        nn::os::WaitThread(&thread[2]);

        rootCommandBuffer->CallCommandBuffer(&GetSubCommandBuffer(0)->object);
        rootCommandBuffer->CallCommandBuffer(&GetSubCommandBuffer(1)->object);
        rootCommandBuffer->CallCommandBuffer(&GetSubCommandBuffer(2)->object);

        NN_PERF_BEGIN_MEASURE_GPU(rootCommandBuffer);
        SleepGpu();
        NN_PERF_END_MEASURE_GPU(rootCommandBuffer);

        NN_PERF_BEGIN_MEASURE_TAG_GPU(rootCommandBuffer, 16);
        SleepGpu();
        NN_PERF_END_MEASURE_GPU(rootCommandBuffer);

        NN_PERF_BEGIN_MEASURE_NAME_GPU(rootCommandBuffer, "Test1");
        SleepGpu();
        NN_PERF_END_MEASURE_GPU(rootCommandBuffer);

        NN_PERF_BEGIN_MEASURE_NAME_AND_TAG_GPU(rootCommandBuffer, "Test2", 32);
        SleepGpu();
        NN_PERF_END_MEASURE_GPU(rootCommandBuffer);

        NN_PERF_BEGIN_MEASURE_NAME_AND_TAG_GPU(rootCommandBuffer, "Test1", 32);
        SleepGpu();
        NN_PERF_END_MEASURE_GPU(rootCommandBuffer);

        NN_PERF_BEGIN_MEASURE_NAME_GPU(rootCommandBuffer, "Test2");
        SleepGpu();
        NN_PERF_END_MEASURE_GPU(rootCommandBuffer);

        NN_PERF_BEGIN_MEASURE_TAG_GPU(rootCommandBuffer, 16);
        SleepGpu();
        NN_PERF_END_MEASURE_GPU(rootCommandBuffer);
    }
    GetGfw()->EndFrame(0);
    GetGfw()->ExecuteCommand(0);
    GetGfw()->QueuePresentTexture(1);
    GetGfw()->QueueFinish();
    NN_PERF_END_FRAME();

    EXPECT_EQ(10, gpuMeter->GetLastSectionCount());

    EXPECT_EQ(10, NN_PERF_GET_CALL_COUNT_GPU(NULL, 0));
    EXPECT_EQ(3, NN_PERF_GET_CALL_COUNT_GPU("Test1", 0));
    EXPECT_EQ(3, NN_PERF_GET_CALL_COUNT_GPU("Test2", 0));
    EXPECT_EQ(2, NN_PERF_GET_CALL_COUNT_GPU(NULL, 32));
    EXPECT_EQ(2, NN_PERF_GET_CALL_COUNT_GPU(NULL, 16));
    EXPECT_EQ(1, NN_PERF_GET_CALL_COUNT_GPU("Test1", 32));
    EXPECT_EQ(0, NN_PERF_GET_CALL_COUNT_GPU("Test2", 8));

    EXPECT_LT(0, NN_PERF_GET_ELAPSED_TIME_GPU(NULL, 0).GetNanoSeconds());
    EXPECT_LT(0, NN_PERF_GET_ELAPSED_TIME_GPU(NULL, 16).GetNanoSeconds());
    EXPECT_LT(0, NN_PERF_GET_ELAPSED_TIME_GPU("Test1", 32).GetNanoSeconds());
    EXPECT_EQ(0, NN_PERF_GET_ELAPSED_TIME_GPU("Test2", 8).GetNanoSeconds());

    EXPECT_LT(min, NN_PERF_GET_AVERAGE_ELAPSED_TIME_GPU(NULL, 0).GetNanoSeconds());
    EXPECT_LT(min, NN_PERF_GET_AVERAGE_ELAPSED_TIME_GPU("Test1", 32).GetNanoSeconds());
    EXPECT_LT(min, NN_PERF_GET_AVERAGE_ELAPSED_TIME_GPU("Test2", 0).GetNanoSeconds());

    EXPECT_GT(max, NN_PERF_GET_AVERAGE_ELAPSED_TIME_GPU(NULL, 0).GetNanoSeconds());
    EXPECT_GT(max, NN_PERF_GET_AVERAGE_ELAPSED_TIME_GPU("Test1", 32).GetNanoSeconds());
    EXPECT_GT(max, NN_PERF_GET_AVERAGE_ELAPSED_TIME_GPU("Test2", 0).GetNanoSeconds());

    EXPECT_LE(NN_PERF_GET_AVERAGE_ELAPSED_TIME_GPU(NULL, 0).GetNanoSeconds(), NN_PERF_GET_MAX_ELAPSED_TIME_GPU(NULL, 0).GetNanoSeconds());
    EXPECT_LE(NN_PERF_GET_AVERAGE_ELAPSED_TIME_GPU("Test1", 32).GetNanoSeconds(), NN_PERF_GET_MAX_ELAPSED_TIME_GPU("Test1", 32).GetNanoSeconds());
    EXPECT_LE(NN_PERF_GET_AVERAGE_ELAPSED_TIME_GPU("Test2", 0).GetNanoSeconds(), NN_PERF_GET_MAX_ELAPSED_TIME_GPU("Test2", 0).GetNanoSeconds());

    EXPECT_GE(NN_PERF_GET_AVERAGE_ELAPSED_TIME_GPU(NULL, 0).GetNanoSeconds(), NN_PERF_GET_MIN_ELAPSED_TIME_GPU(NULL, 0).GetNanoSeconds());
    EXPECT_GE(NN_PERF_GET_AVERAGE_ELAPSED_TIME_GPU("Test1", 32).GetNanoSeconds(), NN_PERF_GET_MIN_ELAPSED_TIME_GPU("Test1", 32).GetNanoSeconds());
    EXPECT_GE(NN_PERF_GET_AVERAGE_ELAPSED_TIME_GPU("Test2", 0).GetNanoSeconds(), NN_PERF_GET_MIN_ELAPSED_TIME_GPU("Test2", 0).GetNanoSeconds());

    EXPECT_EQ(NN_PERF_GET_CALL_COUNT_GPU("Test2", 0), nn::perf::LoadMeterCenter::GetResultAll(nn::perf::UnitType_Gpu, "Test2", 0).callCount);
    EXPECT_EQ(NN_PERF_GET_ELAPSED_TIME_GPU("Test1", 32).GetNanoSeconds(), nn::perf::LoadMeterCenter::GetResultAll(nn::perf::UnitType_Gpu, "Test1", 32).elapsedTime.GetNanoSeconds());
    EXPECT_EQ(NN_PERF_GET_MAX_ELAPSED_TIME_GPU("Test2", 32).GetNanoSeconds(), nn::perf::LoadMeterCenter::GetResultAll(nn::perf::UnitType_Gpu, "Test2", 32).maxElapsedTime.GetNanoSeconds());
    EXPECT_EQ(NN_PERF_GET_MIN_ELAPSED_TIME_GPU("Test2", 0).GetNanoSeconds(), nn::perf::LoadMeterCenter::GetResultAll(nn::perf::UnitType_Gpu, "Test2", 0).minElapsedTime.GetNanoSeconds());

    NN_PERF_CLEAR();

    // CLEAR 後の空結果
    EXPECT_EQ(0, NN_PERF_GET_CALL_COUNT_GPU(NULL, 0));
    EXPECT_EQ(0, NN_PERF_GET_CALL_COUNT_GPU(NULL, 8));
    EXPECT_EQ(0, NN_PERF_GET_CALL_COUNT_GPU(NULL, 16));
    EXPECT_EQ(0, NN_PERF_GET_CALL_COUNT_GPU(NULL, 32));
    EXPECT_EQ(0, NN_PERF_GET_CALL_COUNT_GPU("Test1", 0));
    EXPECT_EQ(0, NN_PERF_GET_CALL_COUNT_GPU("Test2", 0));
    EXPECT_EQ(0, NN_PERF_GET_CALL_COUNT_GPU("Test1", 8));

    EXPECT_EQ(0, NN_PERF_GET_AVERAGE_ELAPSED_TIME_GPU(NULL, 0).GetNanoSeconds());
    EXPECT_EQ(0, NN_PERF_GET_AVERAGE_ELAPSED_TIME_GPU(NULL, 0).GetNanoSeconds());
    EXPECT_EQ(0, NN_PERF_GET_AVERAGE_ELAPSED_TIME_GPU("Test1", 0).GetNanoSeconds());

    EXPECT_EQ(0, NN_PERF_GET_ELAPSED_TIME_GPU("Test2", 32).GetNanoSeconds());
    EXPECT_EQ(0, NN_PERF_GET_ELAPSED_TIME_GPU(NULL, 0).GetNanoSeconds());
    EXPECT_EQ(0, NN_PERF_GET_ELAPSED_TIME_GPU(NULL, 16).GetNanoSeconds());

    EXPECT_EQ(min, NN_PERF_GET_MAX_ELAPSED_TIME_GPU(NULL, 0).GetNanoSeconds());
    EXPECT_EQ(min, NN_PERF_GET_MAX_ELAPSED_TIME_GPU(NULL, 0).GetNanoSeconds());
    EXPECT_EQ(min, NN_PERF_GET_MAX_ELAPSED_TIME_GPU("Test1", 0).GetNanoSeconds());

    EXPECT_EQ(max, NN_PERF_GET_MIN_ELAPSED_TIME_GPU(NULL, 0).GetNanoSeconds());
    EXPECT_EQ(max, NN_PERF_GET_MIN_ELAPSED_TIME_GPU(NULL, 0).GetNanoSeconds());
    EXPECT_EQ(max, NN_PERF_GET_MIN_ELAPSED_TIME_GPU(NULL, 32).GetNanoSeconds());

    nn::os::DestroyThread(&thread[0]);
    nn::os::DestroyThread(&thread[1]);
    nn::os::DestroyThread(&thread[2]);
    nn::os::FreeTlsSlot(tlsSlot);

    FinalizeSubCommandBuffer();
    LoadMeterCenterManager::Finalize();
} //NOLINT(impl/function_size)
