﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{MemoryPool.cpp,PageSampleNvnTutorialLibrary}
 *
 * @brief
 *  This file defines a wrapper around the NVNmemoryPool
 *  object that keeps track of the memory allocated for the
 *  pool, the size of the pool, and the offset into the pool
 *  at which the next memory write could take place. This class
 *  also handles initializing the NVNmemoryPool and makes sure
 *  the memory alignment is handled properly.
 */

#include "SparseMemoryPool.h"
#include <nvn/nvn_FuncPtrInline.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nvngdSupport/TutorialUtil.h>

SparseMemoryPool::SparseMemoryPool() :
    m_CurrentWriteOffset(0),
    m_pMemory(NULL),
    m_Size(0)
{
}

SparseMemoryPool::~SparseMemoryPool()
{
    if (m_pMemory != NULL)
    {
        Shutdown();
    }
}

void SparseMemoryPool::Init(void* pMemory, size_t size, int flags, NVNdevice* pDevice)
{
    m_pMemory = pMemory;

    if (size < g_SparseMinimumPoolSize)
    {
            /* Set memory pool to minimum allowed size */
        size = g_SparseMinimumPoolSize;
    }

        /* Align the pool size to the proper granularity */
    m_Size = Align(size, NVN_MEMORY_POOL_STORAGE_GRANULARITY);
    int x = 0;
    nvnDeviceGetInteger(pDevice, NVN_DEVICE_INFO_MEMORY_POOL_PAGE_SIZE, &x);
    m_Size = Align(m_Size, x);

    if (m_pMemory == NULL)
    {
            /* Allocate memory of the aligned size at the correct address alignment. */
        m_pMemory = AlignedAllocate(m_Size, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
    }

    NVNmemoryPoolBuilder poolBuilder;
    nvnMemoryPoolBuilderSetDefaults(&poolBuilder);
    nvnMemoryPoolBuilderSetDevice(&poolBuilder, pDevice);
    nvnMemoryPoolBuilderSetFlags(&poolBuilder,
        NVNmemoryPoolFlags::NVN_MEMORY_POOL_FLAGS_GPU_NO_ACCESS_BIT |
        NVNmemoryPoolFlags::NVN_MEMORY_POOL_FLAGS_CPU_CACHED_BIT |
        NVNmemoryPoolFlags::NVN_MEMORY_POOL_FLAGS_PHYSICAL_BIT);

    nvnMemoryPoolBuilderSetStorage(&poolBuilder, m_pMemory, m_Size);

    if (nvnMemoryPoolInitialize(&m_PhysicalMemoryPool, &poolBuilder) == NVN_FALSE)
    {
        NN_ASSERT(0, "Failed to initialize physical memory pool");
    }

    nvnMemoryPoolBuilderSetDefaults(&poolBuilder);
    nvnMemoryPoolBuilderSetDevice(&poolBuilder, pDevice);
    nvnMemoryPoolBuilderSetFlags(&poolBuilder,
        NVNmemoryPoolFlags::NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT |
        NVNmemoryPoolFlags::NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT |
        NVNmemoryPoolFlags::NVN_MEMORY_POOL_FLAGS_VIRTUAL_BIT);

    nvnMemoryPoolBuilderSetStorage(&poolBuilder, NULL, m_Size);

    if (nvnMemoryPoolInitialize(&m_VirtualMemoryPool, &poolBuilder) == NVN_FALSE)
    {
        NN_ASSERT(0, "Failed to initialize virtual memory pool");
    }

    m_Flags = flags;
}

void SparseMemoryPool::Shutdown()
{
    if (m_pMemory != NULL)
    {
        m_Size = 0;
        m_CurrentWriteOffset.store(0);

        //NVNmappingRequest m;
        //m.physicalOffset = 0;
        //m.physicalPool = NULL;
        //m.size = 1024;
        //m.storageClass = 0;
        //m.virtualOffset = 0;

        //if(!nvnMemoryPoolMapVirtual(&m_VirtualMemoryPool, 1, &m))
        //{
        //    int x = 0;
        //}
        //NVNboolean sasdf = nvnMemoryPoolMapVirtual(&m_VirtualMemoryPool, 0, NULL);

        nvnMemoryPoolFinalize(&m_VirtualMemoryPool);
        nvnMemoryPoolFinalize(&m_PhysicalMemoryPool);

        AlignedDeallocate(m_pMemory);
        m_pMemory = NULL;
    }
}

ptrdiff_t SparseMemoryPool::GetNewMemoryChunkOffset(size_t size, size_t alignment)
{
    if (static_cast<size_t>(m_CurrentWriteOffset.load()) >= m_Size)
    {
        NN_ASSERT(0, "Memory pool out of memory.");
    }

    m_CurrentWriteOffset.store(Align(m_CurrentWriteOffset.load(), alignment));

    ptrdiff_t dataOffset = m_CurrentWriteOffset.load();

    m_CurrentWriteOffset.fetch_add(size);

    return dataOffset;
}

NVNmemoryPool* SparseMemoryPool::GetVirtualMemoryPool()
{
    return &m_VirtualMemoryPool;
}

NVNmemoryPool* SparseMemoryPool::GetPhysicalMemoryPool()
{
    return &m_PhysicalMemoryPool;
}

void SparseMemoryPool::MapVirtualPool(NVNstorageClass storageClass)
{
    NVNmappingRequest physicalMapRequest;
    physicalMapRequest.physicalPool = &m_PhysicalMemoryPool;
    physicalMapRequest.physicalOffset = 0;
    physicalMapRequest.virtualOffset = 0;
    physicalMapRequest.size = m_Size;
    physicalMapRequest.storageClass = storageClass;
    if (!nvnMemoryPoolMapVirtual(&m_VirtualMemoryPool, 1, &physicalMapRequest))
    {
        NN_ASSERT(false, "Failed to map virtual memory pool")
    }
}
