﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{MemoryPool.cpp,PageSampleNvnTutorialLibrary}
 *
 * @brief
 *  This file defines a wrapper around the NVNmemoryPool
 *  object that keeps track of the memory allocated for the
 *  pool, the size of the pool, and the offset into the pool
 *  at which the next memory write could take place. This class
 *  also handles initializing the NVNmemoryPool and makes sure
 *  the memory alignment is handled properly.
 */

#include <nvn/nvn_FuncPtrInline.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nvngdSupport/MemoryPool.h>
#include <nvngdSupport/TutorialUtil.h>

MemoryPool::MemoryPool() : m_CurrentWriteOffset(0),
                           m_pMemory(NULL),
                           m_Size(0)
{
}

MemoryPool::~MemoryPool()
{
    if (m_pMemory != NULL)
    {
        Shutdown();
    }
}

void MemoryPool::Init(void* pMemory, size_t size, int flags, NVNdevice* pDevice)
{
    m_pMemory = pMemory;

    if (size < g_MinimumPoolSize)
    {
            /* Set memory pool to minimum allowed size */
        size = g_MinimumPoolSize;
    }

        /* Align the pool size to the proper granularity */
    m_Size = Align(size, NVN_MEMORY_POOL_STORAGE_GRANULARITY);

    if (m_pMemory == NULL)
    {
            /* Allocate memory of the aligned size at the correct address alignment. */
        m_pMemory = AlignedAllocate(m_Size, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
    }

    NVNmemoryPoolBuilder poolBuilder;
    nvnMemoryPoolBuilderSetDefaults(&poolBuilder);
    nvnMemoryPoolBuilderSetDevice(&poolBuilder, pDevice);
    nvnMemoryPoolBuilderSetFlags(&poolBuilder, flags);
    nvnMemoryPoolBuilderSetStorage(&poolBuilder, m_pMemory, m_Size);

    if (nvnMemoryPoolInitialize(&m_MemoryPool, &poolBuilder) == NVN_FALSE)
    {
        NN_ASSERT(0, "Failed to initialize buffer memory pool");
    }

    m_Flags = flags;
}

void MemoryPool::Shutdown()
{
    if (m_pMemory != NULL)
    {
        m_Size = 0;
        m_CurrentWriteOffset.store(0);

        nvnMemoryPoolFinalize(&m_MemoryPool);

        AlignedDeallocate(m_pMemory);
        m_pMemory = NULL;
    }
}

ptrdiff_t MemoryPool::GetNewMemoryChunkOffset(size_t size, size_t alignment)
{
    if (static_cast<size_t>(m_CurrentWriteOffset.load()) >= m_Size)
    {
        NN_ASSERT(0, "Memory pool out of memory.");
    }

    m_CurrentWriteOffset.store(Align(m_CurrentWriteOffset.load(), alignment));

    ptrdiff_t dataOffset = m_CurrentWriteOffset.load();

    m_CurrentWriteOffset.fetch_add(size);

    return dataOffset;
}

NVNmemoryPool* MemoryPool::GetMemoryPool()
{
    return &m_MemoryPool;
}
