﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{ManagedCommandBuffer.cpp,PageSampleNvnTutorialLibrary}
 *
 * @brief
 *  This file defines a class that manages memory
 *  for command buffers by using multiple chunks of
 *  of GPU memory. The active chunk is written to and
 *  generates a command handle at rendering end. At the
 *  end of the frame, the next chunk is provided to the
 *  command buffer which then becomes active and writable
 *  for the next frame. Proper syncing to ensure that
 *  memory is not overwritten as it is being used is
 *  necessary; the provided FrameBufferedSyncManager class
 *  handles the syncing of all classes that derive from
 *  the FrameBufferedMemoryManager class.
 *
 *  Requesting a single chunk for the memory pool will provide
 *  an undivided block of memory for the command buffer that is
 *  good for command sets that will never change, though the user
 *  must now handle proper syncing if they do wish to update the
 *  memory.
 */

#include <nn/nn_Assert.h>
#include <nvngdSupport/ManagedCommandBuffer.h>

/*
 * ManagedCommandBuffer Constructor
 * --------------------------------
 * Initialize the managers memory pool with the proper
 * size for command and control memory and set initial
 * values for the manager.
 */
ManagedCommandBuffer::ManagedCommandBuffer(NVNdevice* pDevice, size_t commandChunkSize, size_t controlChunkSize, int numChunks/* = 2 */)
    :
    m_NumChunks(numChunks),
    m_ControlMemoryPool(NULL),
    m_CommandMemoryChunkSize(commandChunkSize),
    m_ControlMemoryChunkSize(controlChunkSize),
    m_CommandMemoryPoolSize(0),
    m_ControlMemoryPoolSize(0),
    m_CurrentChunk(0),
    m_pDevice(pDevice)
{
    NN_ASSERT(m_NumChunks > 0, "Zero or negative number of chunks requested of uniform buffer manager\n");

    int commandBufferCommandAlignment = 0;
    int commandBufferControlAlignment = 0;
    nvnDeviceGetInteger(m_pDevice, NVN_DEVICE_INFO_COMMAND_BUFFER_COMMAND_ALIGNMENT, &commandBufferCommandAlignment);
    nvnDeviceGetInteger(m_pDevice, NVN_DEVICE_INFO_COMMAND_BUFFER_CONTROL_ALIGNMENT, &commandBufferControlAlignment);

        /* Initialize Command memory */
    m_CommandMemoryPoolSize = commandChunkSize * m_NumChunks;

        /* Align the size properly. */
    if (m_CommandMemoryPoolSize < g_MinimumPoolSize)
    {
        m_CommandMemoryPoolSize = g_MinimumPoolSize;
    }
    else
    {
        m_CommandMemoryPoolSize = Align(m_CommandMemoryPoolSize, NVN_MEMORY_POOL_STORAGE_GRANULARITY);
    }

        /* Initialize the command buffer. */
    if (!nvnCommandBufferInitialize(&m_CommandBuffer, m_pDevice))
    {
        NN_ASSERT(0, "nvnCommandBufferInitialize failed");
    }

        /* Create the memory pool. */
    m_CommandMemoryPool.Init(NULL, m_CommandMemoryPoolSize, NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT, m_pDevice);

        /* Initialize Control memory */
    m_ControlMemoryPoolSize = controlChunkSize * m_NumChunks;
    m_ControlMemoryPool = reinterpret_cast<char*>(AlignedAllocate(m_ControlMemoryPoolSize, commandBufferControlAlignment));

        /* Get proper alignment necessary for command and contol memory */
    int commandAlignment = 0;
    nvnDeviceGetInteger(m_pDevice, NVN_DEVICE_INFO_COMMAND_BUFFER_COMMAND_ALIGNMENT, &commandAlignment);

    int controlAlignment = 0;
    nvnDeviceGetInteger(m_pDevice, NVN_DEVICE_INFO_COMMAND_BUFFER_CONTROL_ALIGNMENT, &controlAlignment);

        /*
         * Get the size of the per frame chunks. "Indexing" into different chunks
         * takes the chunk size into account, so if the number of chunks requested
         * does not evenly divide into the resulting memory size you can run into
         * issues of chunks not being sized equally.  Due to the nature of command
         * buffers, if a chunk's memory is overflowed by writing too much to it,
         * the out of memory callback will be hit instead of clobbering another
         * chunk, though in this example an assert will instead be hit to alert
         * the user.
         */
    m_CommandMemoryChunkSize = Align(m_CommandMemoryPoolSize / m_NumChunks, commandAlignment);
    m_ControlMemoryChunkSize = Align(m_ControlMemoryPoolSize / m_NumChunks, controlAlignment);

        /* Provide initial memory for command buffer */
    nvnCommandBufferAddCommandMemory(&m_CommandBuffer, m_CommandMemoryPool.GetMemoryPool(), 0, m_CommandMemoryChunkSize);
    nvnCommandBufferAddControlMemory(&m_CommandBuffer, m_ControlMemoryPool, m_CommandMemoryChunkSize);

        /* Set out of memory callback */
    nvnCommandBufferSetMemoryCallback(&m_CommandBuffer, outOfMemory);
    nvnCommandBufferSetMemoryCallbackData(&m_CommandBuffer, this);
}

/*
 * ManagedCommandBuffer Destructor
 * -------------------------------
 * Clean up NVN data and memory allocations.
 */
ManagedCommandBuffer::~ManagedCommandBuffer()
{
    m_CommandMemoryPool.Shutdown();
    AlignedDeallocate(m_ControlMemoryPool);
    m_ControlMemoryPool = NULL;
}

/*
 * ManagedCommandBuffer::SwapPools
 * -------------------------------
 * Swap the memory memory pools so that memory written to
 * in the previous frame isn't written to until a later frame
 *
 * NOTE: This function should only be called if you can guarantee
 *       that the memory in the next chunk is safe to start writing
 *       to.  This can be done by registering an instance of this
 *       class with an instance of the FrameBufferedSyncManager and
 *       inserting a sync through InsertFence().
 */
void ManagedCommandBuffer::SwapPools()
{
        /* If there is only undivided chunk in the memory pool, then there is nothing to swap */
    if (m_NumChunks == 1)
    {
        return;
    }

    m_CurrentChunk = (m_CurrentChunk + 1) % m_NumChunks;

    ptrdiff_t commandChunkStart = m_CurrentChunk * m_CommandMemoryChunkSize;
    nvnCommandBufferAddCommandMemory(&m_CommandBuffer, m_CommandMemoryPool.GetMemoryPool(), commandChunkStart, m_CommandMemoryChunkSize);

    ptrdiff_t controlChunkStart = m_CurrentChunk * m_ControlMemoryChunkSize;
    nvnCommandBufferAddControlMemory(&m_CommandBuffer, controlChunkStart + m_ControlMemoryPool, m_ControlMemoryChunkSize);
}

/*
 * ManagedCommandBuffer::GetNumChunks
 * ----------------------------------
 * Retrieve number of chunks command memory is using
 */
int ManagedCommandBuffer::GetNumChunks()
{
    return m_NumChunks;
}

/*
 * ManagedCommandBuffer::BeginRecording
 * ------------------------------------
 * Begin recording with the command buffer
 */
void ManagedCommandBuffer::BeginRecording()
{
    nvnCommandBufferBeginRecording(&m_CommandBuffer);
}

/*
 * ManagedCommandBuffer::EndRecording
 * ----------------------------------
 * End recording with the command buffer
 */
NVNcommandHandle ManagedCommandBuffer::EndRecording()
{
    return nvnCommandBufferEndRecording(&m_CommandBuffer);
}

/*
 * ManagedCommandBuffer::GetCommandBuffer
 * --------------------------------------
 * Returns a pointer to the command buffer
 */
NVNcommandBuffer* ManagedCommandBuffer::GetCommandBuffer()
{
    return &m_CommandBuffer;
}

/*
 * ManagedCommandBuffer::outOfMemory
 * ---------------------------------
 * Static function that serves as the callback for out
 * of memory events
 */
inline void NVNAPIENTRY ManagedCommandBuffer::outOfMemory(NVNcommandBuffer *cmdBuf, NVNcommandBufferMemoryEvent event, size_t minSize, void *callbackData)
{
    NN_UNUSED(callbackData);
    NN_ASSERT(cmdBuf);
    NN_ASSERT(minSize);

        /*
         * When an out of memory event is hit, this function will be
         * hit, reporting to the user information about the event including
         * whether command or control memory has run out, the minimum
         * size of memory necessary to provide, and a user-provided
         * pointer.  In this example, this function will just assert to
         * alert the user, however, if memory is successfully provided to
         * the command buffer then it will continue working, though there
         * are some caveats with regards to reporting memory used.
         *
         * When using the functions that query command buffers for memory
         * usage(nvnCommandBufferGetCommandMemoryUsed, etc.), the value
         * returned represents the amount of memory the command buffer is
         * currently managing, not the cumulative amount it has been given
         * over the lifespan of the application.
         *
         * For example :
         * If the command buffer is initialized with a 256 kb sized buffer
         * and 200 kb has been used so far :
         * nvnCommandBufferGetCommandMemoryUsed : 200 kb
         *  Actual Memory Used : 200 kb
         *  Total Memory Given : 256 kb
         *
         * If another 60 kb is used before the next query for memory used,
         * an out of memory event will be triggered.The user can register
         * a function to be used as a callback to provide it with more
         * memory if need be.In the case of providing another 256 kb to the
         * command buffer, calling nvnCommandBufferGetCommandMemoryUsed after
         * the out of memory callback is hit would instead give :
         * nvnCommandBufferGetCommandMemoryUsed : 4 kb
         *  Actual Memory Used : 260 kb
         *  Total Memory Given : 512 kb
         *
         * The function returns 4 kb because, of the new 256 kb chunk that
         * it is currently using, only 4 kb of memory has been used.
         */
    if (event == NVN_COMMAND_BUFFER_MEMORY_EVENT_OUT_OF_COMMAND_MEMORY)
    {
        NN_ASSERT(!"Command buffer has run out of command memory.");
    }
    else if (event == NVN_COMMAND_BUFFER_MEMORY_EVENT_OUT_OF_CONTROL_MEMORY)
    {
        NN_ASSERT(!"Command buffer has run out of control memory");
    }
}
